/*
 * Copyright (C) 2000-2023 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <assert.h>

#include "xine-toolkit/recode.h"

#include "common.h"
#include "help.h"
#include "videowin.h"
#include "actions.h"
#include "event.h"
#include "errors.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/labelbutton.h"
#include "xine-toolkit/tabs.h"
#include "xine-toolkit/browser.h"

#include "lang.h"

#if defined(HAVE_ICONV) && defined(HAVE_LANGINFO_CODESET)
#  include <iconv.h>
#  include <langinfo.h>
#  define USE_CONV
#endif

#define WINDOW_WIDTH             632
#define WINDOW_HEIGHT            473

#define MAX_SECTIONS             50
#define MAX_DISP_ENTRIES         18

typedef struct {
  char                **content;
  int                   line_num;
  int                   marked_line;
} help_section_t;

struct xui_help_s {
  gui_new_window_t      nw;

  xitk_register_key_t   kreg;

  xitk_widget_t        *tabs;
  xitk_widget_t        *ok;
  xitk_widget_t        *browser;
  xitk_widget_t        *copy;

  int                   num_sections;
  int                   tabs_height;
  int                   shown_section;
  char                 *tab_sections[MAX_SECTIONS + 1];
  help_section_t        sections[MAX_SECTIONS];
};

static void _help_unload_string_list (char **list) {
  if (list) {
    free (list[0]);
    free (list);
  }
}

static char **_help_load_text_as_string_list (const char *filename, uint32_t *num_lines, xitk_recode_t *xr) {
  char **list, *p, **rlist, *q;
  uint32_t *bv, *pv, *ev;
  uint32_t have, used, fsize, l;
  FILE *f;

  if (!filename)
    return NULL;
  if (!filename[0])
    return NULL;

  f = fopen (filename, "rb");
  if (!f)
    return NULL;
  if (fseek (f, 0, SEEK_END)) {
    fclose (f);
    return NULL;
  }
  fsize = ftell (f);
  if (fseek (f, 0, SEEK_SET)) {
    fclose (f);
    return NULL;
  }

  have = 256;
  used = 1;
  list = malloc (have * sizeof (*list));
  if (!list) {
    fclose (f);
    return NULL;
  }
  if (fsize > 2 << 20)
    fsize = 2 << 20;
  bv = (uint32_t *)malloc ((fsize + 7) & ~3);
  if (!bv) {
    free (list);
    fclose (f);
    return NULL;
  }
  list[0] = p = (char *)bv;
  fsize = fread (p, 1, fsize, f);
  fclose (f);
  memset (p + fsize, 0x0a, (~(fsize + 3) & 3) + 4);

  ev = bv + ((fsize + 3) >> 2);
  for (pv = bv; pv < ev; pv++) {
    union {uint32_t v; char b[4];} u;
    uint32_t v;
    while (1) {
      v = *pv ^ ~0x0a0a0a0a;
      v = (v & 0x80808080) & ((v & 0x7f7f7f7f) + 0x01010101);
      if (v)
        break;
      pv++;
    }
    if (used + 5 > have) {
      char **new = realloc (list, (have + 256) * sizeof (*list));
      if (!new)
        break;
      list = new;
      have += 256;
    }
    p = (char *)pv;
    u.v = v;
    if (u.b[0]) {
      p[0] = 0;
      list[used++] = p + 1;
      if ((p > (char *)bv) && (p[-1] == '\r'))
        p[-1] = 0;
    }
    if (u.b[1]) {
      p[1] = 0;
      list[used++] = p + 2;
      if (p[0] == '\r')
        p[0] = 0;
    }
    if (u.b[2]) {
      p[2] = 0;
      list[used++] = p + 3;
      if (p[1] == '\r')
        p[1] = 0;
    }
    if (u.b[3]) {
      p[3] = 0;
      list[used++] = p + 4;
      if (p[2] == '\r')
        p[2] = 0;
    }
  }
  while (used && (list[used - 1] >= list[0] + fsize))
    used--;
  *num_lines = used;

  if (!xr || !used) {
    list[used] = NULL;
    return list;
  }

  rlist = malloc ((used + 1) * sizeof (*rlist));
  if (!rlist) {
    list[used] = NULL;
    return list;
  }
  q = malloc (2 * ((fsize + 7) & ~3));
  if (!q) {
    free (rlist);
    list[used] = NULL;
    return list;
  }
  p = q + 2 * fsize;
  for (l = 0; l < used; l++) {
    xitk_recode_string_t rr = {
      .src = list[l],
      .ssize = list[l + 1] - list[l] - 1,
      .buf = rlist[l] = q,
      .bsize = p - q,
      .res = NULL,
      .rsize = 0
    };
    xitk_recode2_do (xr, &rr);
    if (rr.res != q) {
      /* no recoding done/needed. */
      free (rlist[0]);
      free (rlist);
      list[used] = NULL;
      return list;
    }
    q += rr.rsize;
    *q++ = 0;
  }
  rlist[used] = NULL;
  _help_unload_string_list (list);
  return rlist;
}

static void help_change_section (xitk_widget_t *w, void *data, int section) {
  xui_help_t *help = data;

  (void)w;
  if (section < help->num_sections) {
    help->shown_section = section;
    xitk_browser_update_list (help->browser, (const char * const *)help->sections[section].content, NULL,
      help->sections[section].line_num, 0);
    xitk_browser_set_select (help->browser, help->sections[section].marked_line);
    xitk_widgets_state (&help->copy, 1, XITK_WIDGET_STATE_ENABLE,
      (help->sections[section].marked_line < 0) ? 0 : ~0);
  }
}

static void help_add_section (xui_help_t *help, const char *filename, const char *doc_encoding, char *section_name) {
  /* ensure that the file is not empty */
  xitk_recode_t *xr = xitk_recode_init (doc_encoding, NULL, 0);
  uint32_t num_lines = 0;
  char **rlist = _help_load_text_as_string_list (filename, &num_lines, xr);

  xitk_recode_done (xr);
  if (rlist && num_lines) {
    help_section_t *section = help->sections + help->num_sections;
    char *p;

    help->tab_sections[help->num_sections++] = p = strdup (section_name[0] ? section_name : _("Undefined"));
    section->content = rlist;
    section->line_num = num_lines;
    section->marked_line = -1;
    rlist = NULL;
    /* substitute underscores by spaces (nicer) */
    while (*p) {
      if (*p == '_')
        *p = ' ';
      p++;
    }
  }
  _help_unload_string_list (rlist);
}

static void help_sections (xui_help_t *help) {
  DIR *dir = opendir (XINE_DOCDIR);

  if (!dir) {
    gui_msg (help->nw.gui, XUI_MSG_ERROR, "Cannot open help files: %s", strerror (errno));
  } else {
    struct dirent *dir_entry;
    langs_t lang;
    uint32_t ext_len, u;
    uint16_t have[256] = {[0] = 0}, title[256] = {[0] = 0};
    char buf[2048], *q = buf, *e = buf + sizeof (buf);
    char tbuf[XITK_PATH_MAX + 1 + XITK_NAME_MAX + 1], *add;

    get_lang (&lang);
    ext_len = strlen (lang.ext);

    /* look for files named either * README.en.<num>.<title> or README.<lang>.<num>.<title>
     * if both exist with same <num>, keep the <lang> version. */

    memcpy (tbuf, XINE_DOCDIR, sizeof (XINE_DOCDIR) - 1);
    add = tbuf + sizeof (XINE_DOCDIR) - 1;
    *add++ = '/';
    *q++ = 0;
    while ((dir_entry = readdir (dir)) != NULL) {
      const uint8_t *p = (const uint8_t *)dir_entry->d_name;
      uint32_t type, num, d;

      if (strncmp ((const char *)p, "README.", 7))
        continue;
      p += 7;
      if (!strncmp ((const char *)p, "en.", 3)) {
        type = 0;
        p += 3;
      } else if (!strncmp ((const char *)p, lang.ext, ext_len) && (p[ext_len] == '.')) {
        type = 1;
        p += ext_len + 1;
      } else {
        continue;
      }
      for (num = 0; (d = *p ^ '0') < 10u; p++)
        num = num * 10u + d;
      if (*p != '.')
        continue;
      p++;
      if (num >= 256)
        continue;
      if (have[num] && !type)
        continue;
      strcpy (add, dir_entry->d_name);
      if (!is_a_file (tbuf))
        continue;
      have[num] = q - buf;
      title[num] = q - buf + (p - (const uint8_t *)dir_entry->d_name) - 7;
      q += strlcpy (q, dir_entry->d_name + 7, e - q);
      if (q >= e)
        break;
      q++;
    }
    closedir (dir);

    memcpy (add, "README.", 7);
    add += 7;
    for (u = 0; u < 256; u++) {
      if (!have[u])
        continue;
      if (help->num_sections >= MAX_SECTIONS)
        break;
      strlcpy (add, buf + have[u], tbuf + sizeof (tbuf) - add);
      help_add_section (help, tbuf, !memcmp (buf + have[u], "en.", 3) ? "ISO-8859-1" : lang.doc_encoding, buf + title[u]);
    }
    help->tab_sections[help->num_sections] = NULL;
  }
}

static void help_line (xitk_widget_t *w, void *data, int state, int qual) {
  xui_help_t *help = data;

  (void)w;
  (void)qual;
  help->sections[help->shown_section].marked_line = state;
  xitk_widgets_state (&help->copy, 1, XITK_WIDGET_STATE_ENABLE, state < 0 ? 0 : ~0);
}

static void help_copy (xitk_widget_t *w, void *data, int state) {
  xui_help_t *help = data;
  const char *line;
  uint32_t len;

  (void)w;
  (void)state;
  if (help->sections[help->shown_section].marked_line < 0)
    return;
  line = help->sections[help->shown_section].content[help->sections[help->shown_section].marked_line];
  len = xitk_find_byte (line, 0);
  xitk_clipboard_set_text (help->copy, line, len);
}

static void help_exit (xitk_widget_t *w, void *data, int state) {
  xui_help_t *help = data;

  (void)w;
  (void)state;
  if(help) {
    gui_save_window_pos (help->nw.gui, "help", help->kreg);

    xitk_unregister_event_handler (help->nw.gui->xitk, &help->kreg);
    xitk_window_destroy_window(help->nw.xwin);
    help->nw.xwin = NULL;
    /* xitk_dlist_init (&help->nw.wl->list); */

    if(help->num_sections) {
      int i;

      for(i = 0; i < help->num_sections; i++) {
	free (help->tab_sections[i]);
        _help_unload_string_list (help->sections[i].content);
      }
    }

    video_window_set_input_focus(help->nw.gui->vwin);

    help->nw.gui->help = NULL;
    SAFE_FREE (help);
  }
}

static int help_event (void *data, const xitk_be_event_t *e) {
  xui_help_t *help = data;

  if (((e->type == XITK_EV_KEY_DOWN) && (e->utf8[0] == XITK_CTRL_KEY_PREFIX) && (e->utf8[1] == XITK_KEY_ESCAPE))
    || (e->type == XITK_EV_DEL_WIN)) {
    help_exit (NULL, help, 0);
    return 1;
  }
  if ((e->type == XITK_EV_KEY_DOWN) && (e->utf8[0] == ('c' & 0x1f))) {
    help_copy (NULL, help, 0);
    return 1;
  }
  return gui_handle_be_event (help->nw.gui, e);
}

void help_main (xitk_widget_t *mode, void *data) {
  gGui_t *gui = data;
  xui_help_t *help;

  if (!gui)
    return;

  help = gui->help;
  if (mode == XUI_W_OFF) {
    if (!help)
      return;
    help_exit (NULL, help, 0);
    return;
  } else if (mode == XUI_W_ON) {
    if (help) {
      gui_raise_window (gui, help->nw.xwin, 1, 0);
      xitk_window_set_input_focus (help->nw.xwin);
      return;
    }
  } else { /* toggle */
    if (help) {
      help_exit (NULL, help, 0);
      return;
    }
  }

  help = (xui_help_t *)calloc (1, sizeof (*help));
  if (!help)
    return;

  /* Create window */
  help->nw.gui = gui;
  help->nw.title = _("Help");
  help->nw.id = "help";
  if (xitk_init_NULL ()) {
    help->nw.skin = NULL;
    help->nw.wfskin = NULL;
    help->nw.adjust = NULL;
  }
  help->nw.wr.x = 80;
  help->nw.wr.y = 80;
  help->nw.wr.width = WINDOW_WIDTH;
  help->nw.wr.height = WINDOW_HEIGHT;
  if (gui_window_new (&help->nw) < 0) {
    free (help);
    return;
  }

  help->sections[0].content = NULL;
  help->sections[0].line_num = 0;
  help_sections (help);
  help->shown_section = 0;
  if (help->num_sections <= 0) {
    help->tab_sections[0] = _("No Help Section Available");
    help->tab_sections[1] = NULL;
    help->num_sections = 1;
  }

  {
    xitk_tabs_widget_t tab = {
      .nw = { .wl = help->nw.wl, .userdata = help, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .num_entries = help->num_sections,
      .entries     = (const char * const *)help->tab_sections,
      .callback = help_change_section
    };

    help->tabs = xitk_noskin_tabs_create (&tab, 15, 24, WINDOW_WIDTH - 30, tabsfontname);
    if (help->tabs)
      help->tabs_height = xitk_get_widget_height (help->tabs) - 1;
  }

  xitk_image_draw_rectangular_box (help->nw.bg, 15, 24 + help->tabs_height,
    WINDOW_WIDTH - 30, MAX_DISP_ENTRIES * 20 + 10, XITK_DRAW_OUTTER);
  xitk_window_set_background_image (help->nw.xwin, help->nw.bg);

  {
    xitk_browser_widget_t br = {
      .nw = { .wl = help->nw.wl, .userdata = help, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .browser = {
        .max_displayed_entries = MAX_DISP_ENTRIES,
        .num_entries           = help->sections[0].line_num,
        .entries               = (const char * const *)help->sections[0].content
      },
      .callback = help_line
    };
    help->browser = xitk_noskin_browser_create (&br,
      15 + 5, 24 + help->tabs_height + 5, WINDOW_WIDTH - (30 + 10), 20, -16, br_fontname);
  }
  if (ALIGN_DEFAULT != ALIGN_LEFT)
    xitk_browser_set_alignment (help->browser, ALIGN_LEFT);

  {
    uint32_t style = XITK_DRAW_SAT (help->nw.gui->gfx_saturation);
    xitk_labelbutton_widget_t lb = {
      .nw = { .wl = help->nw.wl, .userdata = help },
      .button_type    = CLICK_BUTTON,
      .align          = ALIGN_CENTER,
    };
    char buf[80];

    snprintf (buf, sizeof (buf), "%s-C", xitk_gettext ("Ctrl"));
    lb.nw.add_state = XITK_WIDGET_STATE_VISIBLE;
    lb.nw.tips  = buf;
    lb.label    = xitk_gettext ("Copy");
    lb.callback = help_copy;
    lb.style    = XITK_DRAW_R | XITK_DRAW_G | style;
    help->copy = xitk_noskin_labelbutton_create (&lb,
      (WINDOW_WIDTH - (100 + 2 * 15)) / 2, WINDOW_HEIGHT - (23 + 15), 100, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);

    lb.nw.add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE;
    lb.nw.tips  = "Esc";
    lb.label = _("Close");
    lb.callback = help_exit;
    lb.style    = XITK_DRAW_R | style;
    xitk_noskin_labelbutton_create (&lb,
      WINDOW_WIDTH - (100 + 15), WINDOW_HEIGHT - (23 + 15), 100, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, tabsfontname);
  }

  help->kreg = xitk_be_register_event_handler ("help", help->nw.xwin, help_event, help, NULL, NULL);

  gui_raise_window (help->nw.gui, help->nw.xwin, 1, 0);

  xitk_window_set_input_focus (help->nw.xwin);
  help->nw.gui->help = help;
}
