"""Abstract base class and loader for probe plugins for jobs."""

import re
from arcnagios import substitution
from arcnagios.nagutils import ServiceUnknown
from arcnagios.utils import ident, unspecified

def boolean(s):
    s_lc = s.lower()
    if s_lc in ['0', 'false', 'no', 'off']:
        return False
    if s_lc in ['1', 'true', 'yes', 'on']:
        return True
    raise ValueError('invalid literal for boolean: %r' % s)

_interp_re = re.compile(r'%\(([a-zA-Z0-9_]+)\)')

class JobPlugin:
    """A base-class for tests to run within a job script.  Implementations
    provide commands to run, and how to extract the result.  Optionally it may
    specify staging and cleanup."""

    def __init__(
            self, name, config, config_section, reputation_tracker,
            log, arcclient, env = None):
        self.name = name
        self.config = config
        self.config_section = config_section
        self.reputation_tracker = reputation_tracker
        self.test_name = config_section[6:] # Strip "arcce."
        self.log = log
        self.arcclient = arcclient
        self.environment = env or {}

    def _import_interpolations(self, var):
        if not var in self.environment \
                and self.config.has_option(self.config_section, var):
            raw_value = self.config.get(self.config_section, var, raw = True)
            for mo in re.finditer(_interp_re, raw_value):
                v = mo.group(1)
                if not v in self.environment \
                        and self.config.has_section('variable.' + v):
                    substitution.import_variable(
                            self.config, v, self.reputation_tracker,
                            self.environment)

    def _update_vars(self, kwargs):
        if 'vars' in kwargs:
            kwargs['vars'].update(self.environment)
        else:
            kwargs['vars'] = self.environment

    def hasconf(self, var):
        return self.config.has_option(self.config_section, var)

    def getconf(self, var, default = unspecified, typ = ident, **kwargs):
        if default is not unspecified and not self.hasconf(var):
            return default
        self._import_interpolations(var)
        self._update_vars(kwargs)
        try:
            return typ(self.config.get(self.config_section, var, **kwargs))
        except ValueError as exn:
            raise ServiceUnknown(
                    'Bad value for configuration parameter %s in section %s: %s'
                    % (var, self.config_section, exn)) from exn

    def getconf_int(self, var, default = unspecified, **kwargs):
        self.getconf(var, default = default, type = int, **kwargs)

    def getconf_bool(self, var, default = unspecified, **kwargs):
        self.getconf(var, default = default, type = boolean, **kwargs)

    def getconf_float(self, var, default = unspecified, **kwargs):
        self.getconf(var, default = default, type = float, **kwargs)

    def getconf_strlist(self, var, default = unspecified, sep = None, **kwargs):
        if default is not unspecified and not self.hasconf(var):
            return default
        self._import_interpolations(var)
        self._update_vars(kwargs)
        raw = self.config.get(self.config_section, var, **kwargs)
        return [s.strip() for s in raw.split(sep)]

    @property
    def service_description(self):
        return self.getconf('service_description', None)

    def staged_inputs(self):
        """Override this method to specify files used by the script."""
        return []

    def staged_outputs(self):
        """Override this method to specify files produced by the script, which
        are needed by `extract_result`."""
        return []

    def runtime_environments(self):
        rtes = self.getconf('runtime_environments', '')
        return [x for x in map(str.strip, rtes.split(',')) if x]

    def write_script(self, fh):
        """This method may write commands to run in the job script.  The
        commands are run in the standard shell (/bin/sh), and must not call
        exit or otherwise disrupt the control flow of the script, since other
        commands are run in the same script."""

    def check(self, report, jobdir, stored_urls):
        """This method is run to check the output of a job."""
        raise NotImplementedError('extract_result')

    def cleanup(self, job_state):
        pass
