#!/usr/bin/env python

"""
Handlers for a person for whom scheduling is performed.

Copyright (C) 2014, 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.data import get_address, uri_dict
from imiptools.handlers import Handler
from imiptools.handlers.common import CommonFreebusy, CommonEvent

class PersonHandler(CommonEvent, Handler):

    "Event handling mechanisms specific to people."

    def _process(self, handle, from_organiser=True, **kw):

        """
        Obtain valid organiser and attendee details in order to invoke the given
        'handle' callable, with 'from_organiser' being indicated to obtain the
        details. Any additional keyword arguments will be passed to 'handle'.
        """

        oa = self.require_organiser_and_attendees(from_organiser)
        if not oa:
            return False

        (organiser, organiser_attr), attendees = oa
        return handle(organiser, attendees, **kw)

    def _add(self, organiser, attendees, queue=True):

        """
        Add an event occurrence for the current object or produce a response
        that requests the event details to be sent again.
        """

        # Request details where configured, doing so for unknown objects anyway.

        if self.will_refresh():
            self.make_refresh()
            return

        # Record the event as a recurrence of the parent object.

        self.update_recurrenceid()

        # Update the recipient's record of the organiser's schedule.

        self.update_freebusy_from_organiser(organiser)

        # Set the additional occurrence.

        self.store.set_event(self.user, self.uid, self.recurrenceid, self.obj.to_node())

        # Remove any previous cancellations involving this event.

        self.store.remove_cancellation(self.user, self.uid, self.recurrenceid)

        # Queue any request, if appropriate.

        if queue:
            self.store.queue_request(self.user, self.uid, self.recurrenceid)

        return True

    def _counter(self, organiser, attendees):

        """
        Record details from a counter-proposal, updating the stored object with
        attendance information.
        """

        # Update the attendance for the sender.

        attendee = self.get_sending_attendee()
        if not attendee:
            return False

        self.merge_attendance({attendee : attendees[attendee]})

        # Queue any counter-proposal for perusal.

        self.store.set_counter(self.user, attendee, self.obj.to_node(), self.uid, self.recurrenceid)
        self.store.queue_request(self.user, self.uid, self.recurrenceid, "COUNTER")

        return True

    def _cancel(self):

        "Record an event cancellation."

        # Handle an event being published by the sender to themself.

        organiser_item = self.require_organiser()
        if organiser_item:
            organiser, organiser_attr = organiser_item
            if self.user == organiser:
                self.store.set_event(self.user, self.uid, self.recurrenceid, self.obj.to_node())
                self.store.cancel_event(self.user, self.uid, self.recurrenceid)
                self.store.dequeue_request(self.user, self.uid, self.recurrenceid)
                self.store.remove_counters(self.user, self.uid, self.recurrenceid)
                self.remove_event_from_freebusy()
                self.remove_freebusy_from_attendees(uri_dict(self.obj.get_value_map("ATTENDEE")))
                return True

        return self._process(self._schedule_for_attendee, queue=False, cancel=True)

    def _declinecounter(self, organiser, attendees):

        "Revoke any counter-proposal recorded as a free/busy offer."

        self.remove_event_from_freebusy_offers()
        return True

    def _publish(self):

        "Record details of a published event."

        # Handle an event being published by the sender to themself.

        organiser_item = self.require_organiser()
        if organiser_item:
            organiser, organiser_attr = organiser_item
            if self.user == organiser:
                self.store.set_event(self.user, self.uid, self.recurrenceid, self.obj.to_node())
                self.update_event_in_freebusy()
                return True

        return self._process(self._schedule_for_attendee, queue=False)

    def _schedule_for_attendee(self, organiser, attendees, queue=False, cancel=False):

        """
        Record details from the current object given a message originating
        from an organiser if 'from_organiser' is set to a true value, queuing a
        request if 'queue' is set to a true value, or cancelling an event if
        'cancel' is set to a true value.
        """

        # Process for the current user, an attendee.

        if not self.have_new_object():
            return False

        # Remove additional recurrences if handling a complete event.
        # Also remove any previous cancellations involving this event.

        if not self.recurrenceid:
            self.store.remove_recurrences(self.user, self.uid)
            self.store.remove_cancellations(self.user, self.uid)
        else:
            self.store.remove_cancellation(self.user, self.uid, self.recurrenceid)

        # Queue any request, if appropriate.

        if queue:
            self.store.queue_request(self.user, self.uid, self.recurrenceid)

        # Set the complete event or an additional occurrence.

        self.store.set_event(self.user, self.uid, self.recurrenceid, self.obj.to_node())

        # Cancel complete events or particular occurrences in recurring
        # events.

        if cancel:
            self.store.cancel_event(self.user, self.uid, self.recurrenceid)

            # Remove any associated request.

            self.store.dequeue_request(self.user, self.uid, self.recurrenceid)
            self.store.remove_counters(self.user, self.uid, self.recurrenceid)

            # No return message will occur to update the free/busy
            # information, so this is done here using outgoing message
            # functionality.

            self.remove_event_from_freebusy()

            # Update the recipient's record of the organiser's schedule.

            self.remove_freebusy_from_organiser(organiser)

        else:
            self.update_freebusy_from_organiser(organiser)

        return True

    def _schedule_for_organiser(self, organiser, attendees):

        "As organiser, update attendance from valid attendees."

        # Occurrences that are still part of a parent object are separated,
        # attendance information transferred, and the free/busy details updated.

        if self.is_newly_separated_occurrence():
            if self.make_separate_occurrence(for_organiser=True):

                # Update free/busy details for the event.

                self.update_event_in_freebusy(for_organiser=True)

                # Produce a REQUEST for the created occurrence for other
                # attendees of the parent event.

                obj = self.get_parent_object()
                stored_attendees = set(obj.get_values("ATTENDEE"))
                attendees = stored_attendees.difference(attendees)

                for attendee in attendees:
                    methods, parts = self.get_message_parts(self.obj, "REQUEST", attendee)
                    self.add_results(methods, [get_address(attendee)], parts)

                return True

        # Merge the attendance for the received object.

        elif self.merge_attendance(attendees):
            return self.update_freebusy_from_attendees(attendees)

        return False

    def _refresh(self, organiser, attendees):

        """
        Respond to a refresh message by providing complete event details to
        attendees.
        """

        # Filter by stored attendees.

        obj = self.get_stored_object_version()
        stored_attendees = set(obj.get_values("ATTENDEE"))
        attendees = stored_attendees.intersection(attendees)

        if not attendees:
            return False

        # Produce REQUEST and CANCEL results.

        for attendee in attendees:
            methods, parts = self.get_message_parts(obj, "REQUEST", attendee)
            self.add_results(methods, [get_address(attendee)], parts)

        return True

class Event(PersonHandler):

    "An event handler."

    def add(self):

        "Queue a suggested additional recurrence for any active event."

        _ = self.get_translator()

        if self.allow_add() and self._process(self._add, queue=True):
            self.wrap(_("An addition to an event has been received."))

    def cancel(self):

        "Queue a cancellation of any active event."

        _ = self.get_translator()

        if self._cancel():
            self.wrap(_("An event cancellation has been received."), link=False)

    def counter(self):

        "Record a counter-proposal to a proposed event."

        _ = self.get_translator()

        if self._process(self._counter, from_organiser=False):
            self.wrap(_("A counter proposal to an event invitation has been received."), link=True)

    def declinecounter(self):

        "Record a rejection of a counter-proposal."

        _ = self.get_translator()

        if self._process(self._declinecounter):
            self.wrap(_("Your counter proposal to an event invitation has been declined."), link=True)

    def publish(self):

        "Register details of any relevant event."

        _ = self.get_translator()

        if self._publish():
            self.wrap(_("Details of an event have been received."))

    def refresh(self):

        "Requests to refresh events are handled either here or by the client."

        _ = self.get_translator()

        if self.is_refreshing():
            self._process(self._refresh, from_organiser=False)
        else:
            self.wrap(_("A request for updated event details has been received."))

    def reply(self):

        "Record replies and notify the recipient."

        _ = self.get_translator()

        if self._process(self._schedule_for_organiser, from_organiser=False):
            self.wrap(_("A reply to an event invitation has been received."))

    def request(self):

        "Hold requests and notify the recipient."

        _ = self.get_translator()

        if self._process(self._schedule_for_attendee, queue=True):
            self.wrap(_("An event invitation has been received."))

class Freebusy(CommonFreebusy, Handler):

    "A free/busy handler."

    def publish(self):

        "Register free/busy information."

        _ = self.get_translator()

        self._record_freebusy(from_organiser=True)

        # Produce a message if configured to do so.

        if self.is_notifying():
            self.wrap(_("A free/busy update has been received."), link=False)

    def reply(self):

        "Record replies and notify the recipient."

        _ = self.get_translator()

        self._record_freebusy(from_organiser=False)

        # Produce a message if configured to do so.

        if self.is_notifying():
            self.wrap(_("A reply to a free/busy request has been received."), link=False)

    def request(self):

        """
        Respond to a request by preparing a reply containing free/busy
        information for the recipient.
        """

        # Produce a reply if configured to do so.

        if self.is_sharing():
            return CommonFreebusy.request(self)

# Handler registry.

handlers = [
    ("VFREEBUSY",   Freebusy),
    ("VEVENT",      Event),
    ]

# vim: tabstop=4 expandtab shiftwidth=4
