/*****************************************************************************
 * Copyright (c) 2014-2025 OpenRCT2 developers
 *
 * For a complete list of all authors, please refer to contributors.md
 * Interested in contributing? Visit https://github.com/OpenRCT2/OpenRCT2
 *
 * OpenRCT2 is licensed under the GNU General Public License version 3.
 *****************************************************************************/

#include <gtest/gtest.h>
#include <openrct2/core/MemoryStream.h>
#include <openrct2/core/SawyerCoding.h>
#include <openrct2/rct12/SawyerChunkReader.h>

constexpr size_t BUFFER_SIZE = 0x600000;

using namespace OpenRCT2;

class SawyerCodingTest : public testing::Test
{
protected:
    static const uint8_t randomdata[1024];
    static const uint8_t nonedata[1029];
    static const uint8_t rledata[1038];
    static const uint8_t rlecompresseddata[1949];
    static const uint8_t rotatedata[1029];
    static const uint8_t invalid1[11];
    static const uint8_t invalid2[11];
    static const uint8_t invalid3[14];
    static const uint8_t invalid4[7];
    static const uint8_t invalid5[8];
    static const uint8_t invalid6[7];
    static const uint8_t invalid7[6];
    static const uint8_t empty[1];

    void TestEncodeDecode(uint8_t encoding_type)
    {
        // Encode
        SawyerCoding::ChunkHeader chdr_in;
        chdr_in.encoding = encoding_type;
        chdr_in.length = sizeof(randomdata);
        uint8_t* encodedDataBuffer = new uint8_t[BUFFER_SIZE];
        size_t encodedDataSize = SawyerCoding::WriteChunkBuffer(
            encodedDataBuffer, reinterpret_cast<const uint8_t*>(randomdata), chdr_in);
        ASSERT_GT(encodedDataSize, sizeof(SawyerCoding::ChunkHeader));

        // Decode
        OpenRCT2::MemoryStream ms(encodedDataBuffer, encodedDataSize);
        SawyerChunkReader reader(&ms);
        auto chunk = reader.ReadChunk();
        ASSERT_EQ(static_cast<uint8_t>(chunk->GetEncoding()), chdr_in.encoding);
        ASSERT_EQ(chunk->GetLength(), chdr_in.length);
        auto result = memcmp(chunk->GetData(), randomdata, sizeof(randomdata));
        ASSERT_EQ(result, 0);

        delete[] encodedDataBuffer;
    }

    void TestDecode(const uint8_t* data, size_t size)
    {
        auto expectedLength = size - sizeof(SawyerCoding::ChunkHeader);
        auto chdr_in = reinterpret_cast<const SawyerCoding::ChunkHeader*>(data);
        ASSERT_EQ(chdr_in->length, expectedLength);

        OpenRCT2::MemoryStream ms(data, size);
        SawyerChunkReader reader(&ms);
        auto chunk = reader.ReadChunk();
        ASSERT_EQ(static_cast<uint8_t>(chunk->GetEncoding()), chdr_in->encoding);
        ASSERT_EQ(chunk->GetLength(), sizeof(randomdata));
        auto result = memcmp(chunk->GetData(), randomdata, sizeof(randomdata));
        ASSERT_EQ(result, 0);
    }
};

TEST_F(SawyerCodingTest, write_read_chunk_none)
{
    TestEncodeDecode(CHUNK_ENCODING_NONE);
}

TEST_F(SawyerCodingTest, write_read_chunk_rle)
{
    TestEncodeDecode(CHUNK_ENCODING_RLE);
}

TEST_F(SawyerCodingTest, write_read_chunk_rle_compressed)
{
    TestEncodeDecode(CHUNK_ENCODING_RLECOMPRESSED);
}

TEST_F(SawyerCodingTest, write_read_chunk_rotate)
{
    TestEncodeDecode(CHUNK_ENCODING_ROTATE);
}

// Note we only check if provided data decompresses to the same data, not if it compresses the same.
// The reason for that is we may improve encoding at some point, but the test won't be affected,
// as we already do a decode test and roundtrip (encode + decode), which validates all uses.

TEST_F(SawyerCodingTest, decode_chunk_none)
{
    TestDecode(nonedata, sizeof(nonedata));
}

TEST_F(SawyerCodingTest, DecodeChunkRLE)
{
    TestDecode(rledata, sizeof(rledata));
}

TEST_F(SawyerCodingTest, decode_chunk_rlecompressed)
{
    TestDecode(rlecompresseddata, sizeof(rlecompresseddata));
}

TEST_F(SawyerCodingTest, decode_chunk_rotate)
{
    // Rotate
    TestDecode(rotatedata, sizeof(rotatedata));
}

TEST_F(SawyerCodingTest, invalid1)
{
    OpenRCT2::MemoryStream ms(invalid1, sizeof(invalid1));
    SawyerChunkReader reader(&ms);
    std::shared_ptr<SawyerChunk> ptr;
    EXPECT_THROW(ptr = reader.ReadChunk(), SawyerChunkException);
}

TEST_F(SawyerCodingTest, invalid2)
{
    OpenRCT2::MemoryStream ms(invalid2, sizeof(invalid2));
    SawyerChunkReader reader(&ms);
    std::shared_ptr<SawyerChunk> ptr;
    EXPECT_THROW(ptr = reader.ReadChunk(), SawyerChunkException);
}

TEST_F(SawyerCodingTest, invalid3)
{
    OpenRCT2::MemoryStream ms(invalid3, sizeof(invalid3));
    SawyerChunkReader reader(&ms);
    std::shared_ptr<SawyerChunk> ptr;
    EXPECT_THROW(ptr = reader.ReadChunk(), SawyerChunkException);
}

TEST_F(SawyerCodingTest, invalid4)
{
    OpenRCT2::MemoryStream ms(invalid4, sizeof(invalid4));
    SawyerChunkReader reader(&ms);
    std::shared_ptr<SawyerChunk> ptr;
    EXPECT_THROW(ptr = reader.ReadChunk(), SawyerChunkException);
}

TEST_F(SawyerCodingTest, invalid5)
{
    OpenRCT2::MemoryStream ms(invalid5, sizeof(invalid5));
    SawyerChunkReader reader(&ms);
    std::shared_ptr<SawyerChunk> ptr;
    EXPECT_THROW(ptr = reader.ReadChunk(), SawyerChunkException);
}

TEST_F(SawyerCodingTest, invalid6)
{
    OpenRCT2::MemoryStream ms(invalid6, sizeof(invalid6));
    SawyerChunkReader reader(&ms);
    std::shared_ptr<SawyerChunk> ptr;
    EXPECT_THROW(ptr = reader.ReadChunk(), SawyerChunkException);
}

TEST_F(SawyerCodingTest, invalid7)
{
    OpenRCT2::MemoryStream ms(invalid7, sizeof(invalid7));
    SawyerChunkReader reader(&ms);
    std::shared_ptr<SawyerChunk> ptr;
    EXPECT_THROW(ptr = reader.ReadChunk(), SawyerChunkException);
}

TEST_F(SawyerCodingTest, empty)
{
    OpenRCT2::MemoryStream ms(empty, 0);
    SawyerChunkReader reader(&ms);
    std::shared_ptr<SawyerChunk> ptr;
    EXPECT_THROW(ptr = reader.ReadChunk(), IOException);
}

// 1024 bytes of random data
// use `dd if=/dev/urandom bs=1024 count=1 | xxd -i` to get your own
const uint8_t SawyerCodingTest::randomdata[] = {
    0x3a, 0x97, 0x63, 0x8b, 0xbf, 0xe5, 0x6e, 0x0e, 0xc4, 0xac, 0xdc, 0x84, 0xd7, 0x68, 0xf1, 0x4d, 0xcb, 0xaf, 0x1e, 0x5a,
    0x29, 0x40, 0x87, 0x80, 0x3f, 0xf9, 0xb8, 0xad, 0x01, 0xd3, 0x79, 0x3d, 0xe9, 0x87, 0xa8, 0x95, 0x68, 0xc0, 0xc2, 0x3d,
    0x15, 0x87, 0xdb, 0xa6, 0x90, 0x8c, 0x26, 0x98, 0x2a, 0x3f, 0x2e, 0x0c, 0x82, 0x43, 0x00, 0x10, 0x6d, 0x60, 0xb9, 0xd4,
    0xed, 0xf1, 0x49, 0xbb, 0xf6, 0x7f, 0x21, 0x24, 0xc3, 0xfb, 0x42, 0xe1, 0xfc, 0xb8, 0x82, 0x5e, 0x01, 0x5d, 0x96, 0x2d,
    0x0f, 0x48, 0x12, 0xdf, 0x4b, 0x6c, 0x7e, 0x99, 0x01, 0xfa, 0x46, 0x7d, 0x2c, 0xeb, 0xd6, 0xf3, 0x77, 0xa3, 0x85, 0x8e,
    0x00, 0x34, 0xee, 0x73, 0x31, 0x76, 0x53, 0x17, 0x5a, 0x2b, 0x19, 0x65, 0x2f, 0x32, 0x99, 0x57, 0xc4, 0xf8, 0x05, 0x0d,
    0xc3, 0x59, 0x29, 0x9a, 0x19, 0x3c, 0x33, 0x58, 0x8c, 0xd1, 0x1f, 0x3b, 0xa5, 0x66, 0xa7, 0xfc, 0x7c, 0x99, 0xa9, 0x1a,
    0x5e, 0x65, 0x92, 0x9e, 0x71, 0x27, 0x07, 0xbc, 0x71, 0xed, 0xe5, 0xc6, 0x0e, 0x74, 0xdb, 0x15, 0x91, 0xd7, 0xff, 0xf7,
    0x30, 0x6e, 0x6a, 0x00, 0x40, 0x0e, 0x76, 0x8a, 0xb2, 0x31, 0xa3, 0x81, 0x83, 0xc8, 0xed, 0x08, 0x1f, 0xe5, 0x8d, 0x13,
    0x3e, 0x9a, 0xca, 0x9c, 0xd4, 0x01, 0x28, 0xe6, 0xaf, 0xd6, 0x2f, 0xbc, 0xe1, 0xb7, 0xbb, 0x48, 0x88, 0x86, 0x19, 0xbb,
    0x52, 0xfc, 0x18, 0xc9, 0xe3, 0x90, 0x88, 0x6c, 0x54, 0x53, 0x08, 0x26, 0xff, 0x38, 0xe3, 0x5e, 0x06, 0xa1, 0xac, 0xc2,
    0x83, 0x23, 0x0d, 0x42, 0xe6, 0xa9, 0x81, 0x38, 0x0b, 0x4f, 0x16, 0x2a, 0x77, 0x3b, 0x24, 0xfb, 0x94, 0x53, 0x78, 0x6a,
    0xf3, 0x17, 0x6a, 0x60, 0x44, 0x58, 0x5c, 0x50, 0x80, 0xc8, 0x0f, 0x87, 0x7b, 0x95, 0x0a, 0xdc, 0xf1, 0x1a, 0x6a, 0xa3,
    0xff, 0x79, 0xda, 0x14, 0xd4, 0x6c, 0x7a, 0xfd, 0x7e, 0x1e, 0xfe, 0xd2, 0x28, 0xd0, 0x06, 0x95, 0x40, 0x80, 0xa1, 0x84,
    0xc9, 0xc0, 0x4b, 0xa5, 0x97, 0xb2, 0xb9, 0x20, 0x27, 0x94, 0x87, 0x91, 0xba, 0xb2, 0xc6, 0xdb, 0x78, 0x19, 0x40, 0x35,
    0x9f, 0xbd, 0xfb, 0xc1, 0x45, 0xbe, 0xdc, 0x30, 0xad, 0x38, 0xd4, 0x14, 0x46, 0x98, 0x2a, 0x90, 0xc8, 0x85, 0x5e, 0x7e,
    0x76, 0x0f, 0xc4, 0x20, 0x07, 0x54, 0x1a, 0x60, 0x3a, 0x8c, 0x75, 0xb0, 0x43, 0xab, 0xbb, 0x05, 0xca, 0x93, 0xeb, 0x18,
    0x94, 0xf1, 0x64, 0x8a, 0x01, 0xe0, 0xe6, 0x24, 0x78, 0x50, 0x93, 0xd3, 0xb9, 0xe1, 0xbf, 0x1c, 0xb0, 0x5c, 0xad, 0x94,
    0x17, 0x8a, 0xd1, 0xef, 0x66, 0x08, 0xa7, 0x08, 0x6d, 0x6a, 0xb7, 0x06, 0x31, 0xa6, 0x9d, 0xe5, 0xb2, 0xd8, 0xf9, 0xa6,
    0xf2, 0xe1, 0x63, 0xba, 0x36, 0x7a, 0x53, 0xc0, 0x22, 0x40, 0x3e, 0xa1, 0x95, 0xe5, 0xae, 0x8c, 0x35, 0x7d, 0x34, 0xae,
    0x0c, 0x69, 0x8a, 0x0e, 0xd4, 0x53, 0x19, 0xc9, 0x90, 0x78, 0x36, 0x38, 0xc8, 0xa0, 0xc6, 0x49, 0xef, 0xe3, 0xb2, 0x1d,
    0x7c, 0xd3, 0x38, 0x33, 0x51, 0x69, 0x2b, 0xd2, 0x8e, 0xa8, 0x6d, 0x9c, 0x01, 0x06, 0x9f, 0x93, 0xa5, 0x13, 0x86, 0x67,
    0x72, 0x3e, 0xc7, 0x38, 0x97, 0xe6, 0x1d, 0xce, 0x61, 0x5a, 0x57, 0x10, 0xf7, 0x49, 0x32, 0xea, 0x8f, 0xce, 0x37, 0xea,
    0x38, 0x65, 0x76, 0x86, 0xf4, 0x63, 0x9c, 0x44, 0xe6, 0x4c, 0x7b, 0x3f, 0x9b, 0x61, 0xf7, 0xb4, 0x2a, 0x7a, 0x9c, 0x03,
    0x29, 0xf3, 0x6f, 0xff, 0x76, 0x1a, 0xb8, 0x21, 0xd7, 0xd0, 0xbf, 0x10, 0x1e, 0x82, 0x96, 0x9f, 0xe3, 0xc4, 0xcf, 0x19,
    0x39, 0xd0, 0x4b, 0x4b, 0xa4, 0x7e, 0xd7, 0x4e, 0x3b, 0x2c, 0x0c, 0x35, 0xe1, 0xd4, 0x45, 0x10, 0x28, 0x67, 0x18, 0xdc,
    0x2d, 0x77, 0x68, 0xf2, 0x39, 0x87, 0x52, 0xd7, 0xd1, 0x50, 0x04, 0x45, 0xf6, 0x1d, 0xb8, 0x9d, 0xfa, 0xe8, 0xd8, 0x94,
    0xe0, 0xae, 0x31, 0x66, 0xc7, 0xee, 0xa9, 0xa8, 0x51, 0xe9, 0x14, 0x67, 0x53, 0x4c, 0x40, 0xf6, 0xee, 0x50, 0x67, 0x57,
    0xf4, 0xd8, 0xd1, 0x95, 0x17, 0x2c, 0x5a, 0x72, 0x8f, 0xdc, 0xeb, 0x6c, 0x9b, 0x24, 0xfc, 0x67, 0x11, 0x07, 0x82, 0x2a,
    0xae, 0xfb, 0xd9, 0xfd, 0x89, 0x75, 0x71, 0x75, 0x74, 0x08, 0x0d, 0xf7, 0xba, 0x5d, 0x56, 0xdf, 0x7e, 0x52, 0x5b, 0xce,
    0xef, 0xeb, 0xf6, 0x32, 0x11, 0x93, 0x5f, 0xab, 0xfe, 0x08, 0xec, 0x20, 0x18, 0x0a, 0x4e, 0xc5, 0xb6, 0x42, 0xe4, 0x70,
    0xbb, 0xba, 0x1b, 0xec, 0x01, 0x9c, 0xe5, 0x1c, 0xf4, 0xd2, 0x68, 0x0b, 0x30, 0x2d, 0xec, 0xda, 0x23, 0xff, 0xbd, 0x95,
    0x46, 0x55, 0x59, 0xab, 0x03, 0x0f, 0xe4, 0xeb, 0x8c, 0xca, 0xc1, 0x13, 0x28, 0x49, 0x12, 0x66, 0xd7, 0x85, 0xcb, 0x21,
    0x8d, 0x8a, 0x34, 0x4c, 0x2e, 0x28, 0x25, 0x79, 0xbf, 0x98, 0x04, 0xcd, 0x8a, 0x3e, 0xad, 0x08, 0xd3, 0x21, 0x32, 0xed,
    0x54, 0x7e, 0x17, 0xfe, 0x89, 0xe9, 0x09, 0x18, 0xac, 0x96, 0xda, 0x51, 0x61, 0x85, 0x44, 0xd2, 0xdf, 0xd9, 0x61, 0xa2,
    0x44, 0x07, 0x29, 0xa5, 0xdc, 0x16, 0xac, 0x3d, 0x6e, 0x27, 0xb6, 0x5a, 0x15, 0x87, 0x6c, 0x48, 0x3e, 0x34, 0xdd, 0xec,
    0xf2, 0x7c, 0xc6, 0x87, 0x0f, 0xcf, 0xac, 0xde, 0x42, 0x02, 0x93, 0xff, 0x4f, 0x10, 0x0f, 0x03, 0xf3, 0x1b, 0x1e, 0xaf,
    0x94, 0x8e, 0x77, 0x7c, 0x66, 0x65, 0xfb, 0xd9, 0x0d, 0xf8, 0x36, 0x0c, 0xac, 0xdc, 0x8e, 0xd1, 0x1e, 0x19, 0xa8, 0x87,
    0x97, 0x39, 0x77, 0x98, 0x6c, 0xfb, 0x5d, 0xc1, 0x09, 0x7e, 0x5d, 0xab, 0xde, 0xdc, 0x1f, 0x21, 0xd0, 0x7a, 0xaa, 0xeb,
    0x96, 0x60, 0xf9, 0x95, 0xed, 0x13, 0xf4, 0x82, 0xa9, 0x94, 0x7e, 0xa8, 0x5e, 0xe2, 0xca, 0x08, 0xed, 0xfe, 0x9d, 0x94,
    0x3f, 0x08, 0xd8, 0xd8, 0x9d, 0xb0, 0x84, 0xd6, 0x6a, 0xcb, 0xc1, 0xd4, 0xd0, 0xe8, 0xe7, 0xae, 0x44, 0x28, 0x47, 0x75,
    0xa6, 0x2a, 0x40, 0x0b, 0x82, 0x57, 0xd5, 0x49, 0x40, 0x85, 0x5c, 0x92, 0x95, 0x6b, 0x7d, 0xbe, 0xcc, 0xb6, 0x2d, 0x2b,
    0x71, 0x85, 0x63, 0x39, 0xfa, 0xbc, 0x19, 0x7c, 0xe2, 0x3a, 0x81, 0xf1, 0x86, 0x9e, 0x46, 0x3d, 0x5f, 0x71, 0xa4, 0xb7,
    0xd4, 0x2c, 0x98, 0xfa, 0xb5, 0x95, 0xac, 0x33, 0x1a, 0x53, 0x9d, 0xe3, 0x4e, 0x5d, 0x4f, 0x54, 0x0d, 0xe8, 0xa4, 0x32,
    0x25, 0x4d, 0x5e, 0x4c, 0xe1, 0xae, 0x46, 0x85, 0x1e, 0xc4, 0x20, 0xdf, 0x73, 0x1e, 0x81, 0x40, 0xa2, 0x33, 0x75, 0x1e,
    0x1d, 0x1d, 0x7c, 0x1f, 0x07, 0xe5, 0x9e, 0x76, 0x9f, 0x42, 0xeb, 0xc4, 0xb4, 0xd3, 0xcb, 0x55, 0xe8, 0xbf, 0xf7, 0xeb,
    0x7c, 0x2d, 0xa5, 0xfd, 0xda, 0xb4, 0x5b, 0x1f, 0xa8, 0xd2, 0xeb, 0xdf, 0xd9, 0xf2, 0x35, 0xc6, 0x0c, 0x36, 0x66, 0x28,
    0x95, 0xe2, 0x34, 0x64, 0xea, 0xaf, 0x3a, 0x36, 0x3c, 0x38, 0x31, 0x99, 0x8d, 0x80, 0x78, 0x18, 0x48, 0x2c, 0xe9, 0xe5,
    0xb7, 0xe6, 0x2e, 0x73, 0xf1, 0x7c, 0x84, 0x21, 0x04, 0x00, 0x8e, 0x12, 0x66, 0x6c, 0x9b, 0x09, 0x88, 0xe2, 0xd5, 0xec,
    0xb4, 0xdd, 0x2b, 0xda, 0x29, 0x6f, 0x1b, 0x81, 0x21, 0xe4, 0x5c, 0x36, 0x73, 0x2d, 0x9d, 0xdd, 0xb6, 0x69, 0x21, 0x40,
    0x4e, 0x70, 0xde, 0x9d, 0x0c, 0x17, 0x1d, 0x15, 0x29, 0x5b, 0xd0, 0x66, 0x72, 0xb8, 0x38, 0x80, 0xbe, 0x9e, 0xd7, 0x5e,
    0xb5, 0x72, 0x22, 0xbc,
};

// Following are compressed versions of the data above.

const uint8_t SawyerCodingTest::nonedata[] = {
    0x00, 0x00, 0x04, 0x00, 0x00, 0x3a, 0x97, 0x63, 0x8b, 0xbf, 0xe5, 0x6e, 0x0e, 0xc4, 0xac, 0xdc, 0x84, 0xd7, 0x68, 0xf1,
    0x4d, 0xcb, 0xaf, 0x1e, 0x5a, 0x29, 0x40, 0x87, 0x80, 0x3f, 0xf9, 0xb8, 0xad, 0x01, 0xd3, 0x79, 0x3d, 0xe9, 0x87, 0xa8,
    0x95, 0x68, 0xc0, 0xc2, 0x3d, 0x15, 0x87, 0xdb, 0xa6, 0x90, 0x8c, 0x26, 0x98, 0x2a, 0x3f, 0x2e, 0x0c, 0x82, 0x43, 0x00,
    0x10, 0x6d, 0x60, 0xb9, 0xd4, 0xed, 0xf1, 0x49, 0xbb, 0xf6, 0x7f, 0x21, 0x24, 0xc3, 0xfb, 0x42, 0xe1, 0xfc, 0xb8, 0x82,
    0x5e, 0x01, 0x5d, 0x96, 0x2d, 0x0f, 0x48, 0x12, 0xdf, 0x4b, 0x6c, 0x7e, 0x99, 0x01, 0xfa, 0x46, 0x7d, 0x2c, 0xeb, 0xd6,
    0xf3, 0x77, 0xa3, 0x85, 0x8e, 0x00, 0x34, 0xee, 0x73, 0x31, 0x76, 0x53, 0x17, 0x5a, 0x2b, 0x19, 0x65, 0x2f, 0x32, 0x99,
    0x57, 0xc4, 0xf8, 0x05, 0x0d, 0xc3, 0x59, 0x29, 0x9a, 0x19, 0x3c, 0x33, 0x58, 0x8c, 0xd1, 0x1f, 0x3b, 0xa5, 0x66, 0xa7,
    0xfc, 0x7c, 0x99, 0xa9, 0x1a, 0x5e, 0x65, 0x92, 0x9e, 0x71, 0x27, 0x07, 0xbc, 0x71, 0xed, 0xe5, 0xc6, 0x0e, 0x74, 0xdb,
    0x15, 0x91, 0xd7, 0xff, 0xf7, 0x30, 0x6e, 0x6a, 0x00, 0x40, 0x0e, 0x76, 0x8a, 0xb2, 0x31, 0xa3, 0x81, 0x83, 0xc8, 0xed,
    0x08, 0x1f, 0xe5, 0x8d, 0x13, 0x3e, 0x9a, 0xca, 0x9c, 0xd4, 0x01, 0x28, 0xe6, 0xaf, 0xd6, 0x2f, 0xbc, 0xe1, 0xb7, 0xbb,
    0x48, 0x88, 0x86, 0x19, 0xbb, 0x52, 0xfc, 0x18, 0xc9, 0xe3, 0x90, 0x88, 0x6c, 0x54, 0x53, 0x08, 0x26, 0xff, 0x38, 0xe3,
    0x5e, 0x06, 0xa1, 0xac, 0xc2, 0x83, 0x23, 0x0d, 0x42, 0xe6, 0xa9, 0x81, 0x38, 0x0b, 0x4f, 0x16, 0x2a, 0x77, 0x3b, 0x24,
    0xfb, 0x94, 0x53, 0x78, 0x6a, 0xf3, 0x17, 0x6a, 0x60, 0x44, 0x58, 0x5c, 0x50, 0x80, 0xc8, 0x0f, 0x87, 0x7b, 0x95, 0x0a,
    0xdc, 0xf1, 0x1a, 0x6a, 0xa3, 0xff, 0x79, 0xda, 0x14, 0xd4, 0x6c, 0x7a, 0xfd, 0x7e, 0x1e, 0xfe, 0xd2, 0x28, 0xd0, 0x06,
    0x95, 0x40, 0x80, 0xa1, 0x84, 0xc9, 0xc0, 0x4b, 0xa5, 0x97, 0xb2, 0xb9, 0x20, 0x27, 0x94, 0x87, 0x91, 0xba, 0xb2, 0xc6,
    0xdb, 0x78, 0x19, 0x40, 0x35, 0x9f, 0xbd, 0xfb, 0xc1, 0x45, 0xbe, 0xdc, 0x30, 0xad, 0x38, 0xd4, 0x14, 0x46, 0x98, 0x2a,
    0x90, 0xc8, 0x85, 0x5e, 0x7e, 0x76, 0x0f, 0xc4, 0x20, 0x07, 0x54, 0x1a, 0x60, 0x3a, 0x8c, 0x75, 0xb0, 0x43, 0xab, 0xbb,
    0x05, 0xca, 0x93, 0xeb, 0x18, 0x94, 0xf1, 0x64, 0x8a, 0x01, 0xe0, 0xe6, 0x24, 0x78, 0x50, 0x93, 0xd3, 0xb9, 0xe1, 0xbf,
    0x1c, 0xb0, 0x5c, 0xad, 0x94, 0x17, 0x8a, 0xd1, 0xef, 0x66, 0x08, 0xa7, 0x08, 0x6d, 0x6a, 0xb7, 0x06, 0x31, 0xa6, 0x9d,
    0xe5, 0xb2, 0xd8, 0xf9, 0xa6, 0xf2, 0xe1, 0x63, 0xba, 0x36, 0x7a, 0x53, 0xc0, 0x22, 0x40, 0x3e, 0xa1, 0x95, 0xe5, 0xae,
    0x8c, 0x35, 0x7d, 0x34, 0xae, 0x0c, 0x69, 0x8a, 0x0e, 0xd4, 0x53, 0x19, 0xc9, 0x90, 0x78, 0x36, 0x38, 0xc8, 0xa0, 0xc6,
    0x49, 0xef, 0xe3, 0xb2, 0x1d, 0x7c, 0xd3, 0x38, 0x33, 0x51, 0x69, 0x2b, 0xd2, 0x8e, 0xa8, 0x6d, 0x9c, 0x01, 0x06, 0x9f,
    0x93, 0xa5, 0x13, 0x86, 0x67, 0x72, 0x3e, 0xc7, 0x38, 0x97, 0xe6, 0x1d, 0xce, 0x61, 0x5a, 0x57, 0x10, 0xf7, 0x49, 0x32,
    0xea, 0x8f, 0xce, 0x37, 0xea, 0x38, 0x65, 0x76, 0x86, 0xf4, 0x63, 0x9c, 0x44, 0xe6, 0x4c, 0x7b, 0x3f, 0x9b, 0x61, 0xf7,
    0xb4, 0x2a, 0x7a, 0x9c, 0x03, 0x29, 0xf3, 0x6f, 0xff, 0x76, 0x1a, 0xb8, 0x21, 0xd7, 0xd0, 0xbf, 0x10, 0x1e, 0x82, 0x96,
    0x9f, 0xe3, 0xc4, 0xcf, 0x19, 0x39, 0xd0, 0x4b, 0x4b, 0xa4, 0x7e, 0xd7, 0x4e, 0x3b, 0x2c, 0x0c, 0x35, 0xe1, 0xd4, 0x45,
    0x10, 0x28, 0x67, 0x18, 0xdc, 0x2d, 0x77, 0x68, 0xf2, 0x39, 0x87, 0x52, 0xd7, 0xd1, 0x50, 0x04, 0x45, 0xf6, 0x1d, 0xb8,
    0x9d, 0xfa, 0xe8, 0xd8, 0x94, 0xe0, 0xae, 0x31, 0x66, 0xc7, 0xee, 0xa9, 0xa8, 0x51, 0xe9, 0x14, 0x67, 0x53, 0x4c, 0x40,
    0xf6, 0xee, 0x50, 0x67, 0x57, 0xf4, 0xd8, 0xd1, 0x95, 0x17, 0x2c, 0x5a, 0x72, 0x8f, 0xdc, 0xeb, 0x6c, 0x9b, 0x24, 0xfc,
    0x67, 0x11, 0x07, 0x82, 0x2a, 0xae, 0xfb, 0xd9, 0xfd, 0x89, 0x75, 0x71, 0x75, 0x74, 0x08, 0x0d, 0xf7, 0xba, 0x5d, 0x56,
    0xdf, 0x7e, 0x52, 0x5b, 0xce, 0xef, 0xeb, 0xf6, 0x32, 0x11, 0x93, 0x5f, 0xab, 0xfe, 0x08, 0xec, 0x20, 0x18, 0x0a, 0x4e,
    0xc5, 0xb6, 0x42, 0xe4, 0x70, 0xbb, 0xba, 0x1b, 0xec, 0x01, 0x9c, 0xe5, 0x1c, 0xf4, 0xd2, 0x68, 0x0b, 0x30, 0x2d, 0xec,
    0xda, 0x23, 0xff, 0xbd, 0x95, 0x46, 0x55, 0x59, 0xab, 0x03, 0x0f, 0xe4, 0xeb, 0x8c, 0xca, 0xc1, 0x13, 0x28, 0x49, 0x12,
    0x66, 0xd7, 0x85, 0xcb, 0x21, 0x8d, 0x8a, 0x34, 0x4c, 0x2e, 0x28, 0x25, 0x79, 0xbf, 0x98, 0x04, 0xcd, 0x8a, 0x3e, 0xad,
    0x08, 0xd3, 0x21, 0x32, 0xed, 0x54, 0x7e, 0x17, 0xfe, 0x89, 0xe9, 0x09, 0x18, 0xac, 0x96, 0xda, 0x51, 0x61, 0x85, 0x44,
    0xd2, 0xdf, 0xd9, 0x61, 0xa2, 0x44, 0x07, 0x29, 0xa5, 0xdc, 0x16, 0xac, 0x3d, 0x6e, 0x27, 0xb6, 0x5a, 0x15, 0x87, 0x6c,
    0x48, 0x3e, 0x34, 0xdd, 0xec, 0xf2, 0x7c, 0xc6, 0x87, 0x0f, 0xcf, 0xac, 0xde, 0x42, 0x02, 0x93, 0xff, 0x4f, 0x10, 0x0f,
    0x03, 0xf3, 0x1b, 0x1e, 0xaf, 0x94, 0x8e, 0x77, 0x7c, 0x66, 0x65, 0xfb, 0xd9, 0x0d, 0xf8, 0x36, 0x0c, 0xac, 0xdc, 0x8e,
    0xd1, 0x1e, 0x19, 0xa8, 0x87, 0x97, 0x39, 0x77, 0x98, 0x6c, 0xfb, 0x5d, 0xc1, 0x09, 0x7e, 0x5d, 0xab, 0xde, 0xdc, 0x1f,
    0x21, 0xd0, 0x7a, 0xaa, 0xeb, 0x96, 0x60, 0xf9, 0x95, 0xed, 0x13, 0xf4, 0x82, 0xa9, 0x94, 0x7e, 0xa8, 0x5e, 0xe2, 0xca,
    0x08, 0xed, 0xfe, 0x9d, 0x94, 0x3f, 0x08, 0xd8, 0xd8, 0x9d, 0xb0, 0x84, 0xd6, 0x6a, 0xcb, 0xc1, 0xd4, 0xd0, 0xe8, 0xe7,
    0xae, 0x44, 0x28, 0x47, 0x75, 0xa6, 0x2a, 0x40, 0x0b, 0x82, 0x57, 0xd5, 0x49, 0x40, 0x85, 0x5c, 0x92, 0x95, 0x6b, 0x7d,
    0xbe, 0xcc, 0xb6, 0x2d, 0x2b, 0x71, 0x85, 0x63, 0x39, 0xfa, 0xbc, 0x19, 0x7c, 0xe2, 0x3a, 0x81, 0xf1, 0x86, 0x9e, 0x46,
    0x3d, 0x5f, 0x71, 0xa4, 0xb7, 0xd4, 0x2c, 0x98, 0xfa, 0xb5, 0x95, 0xac, 0x33, 0x1a, 0x53, 0x9d, 0xe3, 0x4e, 0x5d, 0x4f,
    0x54, 0x0d, 0xe8, 0xa4, 0x32, 0x25, 0x4d, 0x5e, 0x4c, 0xe1, 0xae, 0x46, 0x85, 0x1e, 0xc4, 0x20, 0xdf, 0x73, 0x1e, 0x81,
    0x40, 0xa2, 0x33, 0x75, 0x1e, 0x1d, 0x1d, 0x7c, 0x1f, 0x07, 0xe5, 0x9e, 0x76, 0x9f, 0x42, 0xeb, 0xc4, 0xb4, 0xd3, 0xcb,
    0x55, 0xe8, 0xbf, 0xf7, 0xeb, 0x7c, 0x2d, 0xa5, 0xfd, 0xda, 0xb4, 0x5b, 0x1f, 0xa8, 0xd2, 0xeb, 0xdf, 0xd9, 0xf2, 0x35,
    0xc6, 0x0c, 0x36, 0x66, 0x28, 0x95, 0xe2, 0x34, 0x64, 0xea, 0xaf, 0x3a, 0x36, 0x3c, 0x38, 0x31, 0x99, 0x8d, 0x80, 0x78,
    0x18, 0x48, 0x2c, 0xe9, 0xe5, 0xb7, 0xe6, 0x2e, 0x73, 0xf1, 0x7c, 0x84, 0x21, 0x04, 0x00, 0x8e, 0x12, 0x66, 0x6c, 0x9b,
    0x09, 0x88, 0xe2, 0xd5, 0xec, 0xb4, 0xdd, 0x2b, 0xda, 0x29, 0x6f, 0x1b, 0x81, 0x21, 0xe4, 0x5c, 0x36, 0x73, 0x2d, 0x9d,
    0xdd, 0xb6, 0x69, 0x21, 0x40, 0x4e, 0x70, 0xde, 0x9d, 0x0c, 0x17, 0x1d, 0x15, 0x29, 0x5b, 0xd0, 0x66, 0x72, 0xb8, 0x38,
    0x80, 0xbe, 0x9e, 0xd7, 0x5e, 0xb5, 0x72, 0x22, 0xbc,
};

const uint8_t SawyerCodingTest::rledata[] = {
    0x01, 0x09, 0x04, 0x00, 0x00, 0x7d, 0x3a, 0x97, 0x63, 0x8b, 0xbf, 0xe5, 0x6e, 0x0e, 0xc4, 0xac, 0xdc, 0x84, 0xd7, 0x68,
    0xf1, 0x4d, 0xcb, 0xaf, 0x1e, 0x5a, 0x29, 0x40, 0x87, 0x80, 0x3f, 0xf9, 0xb8, 0xad, 0x01, 0xd3, 0x79, 0x3d, 0xe9, 0x87,
    0xa8, 0x95, 0x68, 0xc0, 0xc2, 0x3d, 0x15, 0x87, 0xdb, 0xa6, 0x90, 0x8c, 0x26, 0x98, 0x2a, 0x3f, 0x2e, 0x0c, 0x82, 0x43,
    0x00, 0x10, 0x6d, 0x60, 0xb9, 0xd4, 0xed, 0xf1, 0x49, 0xbb, 0xf6, 0x7f, 0x21, 0x24, 0xc3, 0xfb, 0x42, 0xe1, 0xfc, 0xb8,
    0x82, 0x5e, 0x01, 0x5d, 0x96, 0x2d, 0x0f, 0x48, 0x12, 0xdf, 0x4b, 0x6c, 0x7e, 0x99, 0x01, 0xfa, 0x46, 0x7d, 0x2c, 0xeb,
    0xd6, 0xf3, 0x77, 0xa3, 0x85, 0x8e, 0x00, 0x34, 0xee, 0x73, 0x31, 0x76, 0x53, 0x17, 0x5a, 0x2b, 0x19, 0x65, 0x2f, 0x32,
    0x99, 0x57, 0xc4, 0xf8, 0x05, 0x0d, 0xc3, 0x59, 0x29, 0x9a, 0x19, 0x3c, 0x7d, 0x33, 0x58, 0x8c, 0xd1, 0x1f, 0x3b, 0xa5,
    0x66, 0xa7, 0xfc, 0x7c, 0x99, 0xa9, 0x1a, 0x5e, 0x65, 0x92, 0x9e, 0x71, 0x27, 0x07, 0xbc, 0x71, 0xed, 0xe5, 0xc6, 0x0e,
    0x74, 0xdb, 0x15, 0x91, 0xd7, 0xff, 0xf7, 0x30, 0x6e, 0x6a, 0x00, 0x40, 0x0e, 0x76, 0x8a, 0xb2, 0x31, 0xa3, 0x81, 0x83,
    0xc8, 0xed, 0x08, 0x1f, 0xe5, 0x8d, 0x13, 0x3e, 0x9a, 0xca, 0x9c, 0xd4, 0x01, 0x28, 0xe6, 0xaf, 0xd6, 0x2f, 0xbc, 0xe1,
    0xb7, 0xbb, 0x48, 0x88, 0x86, 0x19, 0xbb, 0x52, 0xfc, 0x18, 0xc9, 0xe3, 0x90, 0x88, 0x6c, 0x54, 0x53, 0x08, 0x26, 0xff,
    0x38, 0xe3, 0x5e, 0x06, 0xa1, 0xac, 0xc2, 0x83, 0x23, 0x0d, 0x42, 0xe6, 0xa9, 0x81, 0x38, 0x0b, 0x4f, 0x16, 0x2a, 0x77,
    0x3b, 0x24, 0xfb, 0x94, 0x53, 0x78, 0x6a, 0xf3, 0x17, 0x6a, 0x60, 0x44, 0x58, 0x5c, 0x50, 0x80, 0xc8, 0x0f, 0x87, 0x7d,
    0x7b, 0x95, 0x0a, 0xdc, 0xf1, 0x1a, 0x6a, 0xa3, 0xff, 0x79, 0xda, 0x14, 0xd4, 0x6c, 0x7a, 0xfd, 0x7e, 0x1e, 0xfe, 0xd2,
    0x28, 0xd0, 0x06, 0x95, 0x40, 0x80, 0xa1, 0x84, 0xc9, 0xc0, 0x4b, 0xa5, 0x97, 0xb2, 0xb9, 0x20, 0x27, 0x94, 0x87, 0x91,
    0xba, 0xb2, 0xc6, 0xdb, 0x78, 0x19, 0x40, 0x35, 0x9f, 0xbd, 0xfb, 0xc1, 0x45, 0xbe, 0xdc, 0x30, 0xad, 0x38, 0xd4, 0x14,
    0x46, 0x98, 0x2a, 0x90, 0xc8, 0x85, 0x5e, 0x7e, 0x76, 0x0f, 0xc4, 0x20, 0x07, 0x54, 0x1a, 0x60, 0x3a, 0x8c, 0x75, 0xb0,
    0x43, 0xab, 0xbb, 0x05, 0xca, 0x93, 0xeb, 0x18, 0x94, 0xf1, 0x64, 0x8a, 0x01, 0xe0, 0xe6, 0x24, 0x78, 0x50, 0x93, 0xd3,
    0xb9, 0xe1, 0xbf, 0x1c, 0xb0, 0x5c, 0xad, 0x94, 0x17, 0x8a, 0xd1, 0xef, 0x66, 0x08, 0xa7, 0x08, 0x6d, 0x6a, 0xb7, 0x06,
    0x31, 0xa6, 0x9d, 0xe5, 0xb2, 0xd8, 0x7b, 0xf9, 0xa6, 0xf2, 0xe1, 0x63, 0xba, 0x36, 0x7a, 0x53, 0xc0, 0x22, 0x40, 0x3e,
    0xa1, 0x95, 0xe5, 0xae, 0x8c, 0x35, 0x7d, 0x34, 0xae, 0x0c, 0x69, 0x8a, 0x0e, 0xd4, 0x53, 0x19, 0xc9, 0x90, 0x78, 0x36,
    0x38, 0xc8, 0xa0, 0xc6, 0x49, 0xef, 0xe3, 0xb2, 0x1d, 0x7c, 0xd3, 0x38, 0x33, 0x51, 0x69, 0x2b, 0xd2, 0x8e, 0xa8, 0x6d,
    0x9c, 0x01, 0x06, 0x9f, 0x93, 0xa5, 0x13, 0x86, 0x67, 0x72, 0x3e, 0xc7, 0x38, 0x97, 0xe6, 0x1d, 0xce, 0x61, 0x5a, 0x57,
    0x10, 0xf7, 0x49, 0x32, 0xea, 0x8f, 0xce, 0x37, 0xea, 0x38, 0x65, 0x76, 0x86, 0xf4, 0x63, 0x9c, 0x44, 0xe6, 0x4c, 0x7b,
    0x3f, 0x9b, 0x61, 0xf7, 0xb4, 0x2a, 0x7a, 0x9c, 0x03, 0x29, 0xf3, 0x6f, 0xff, 0x76, 0x1a, 0xb8, 0x21, 0xd7, 0xd0, 0xbf,
    0x10, 0x1e, 0x82, 0x96, 0x9f, 0xe3, 0xc4, 0xcf, 0x19, 0x39, 0xd0, 0xff, 0x4b, 0x7d, 0xa4, 0x7e, 0xd7, 0x4e, 0x3b, 0x2c,
    0x0c, 0x35, 0xe1, 0xd4, 0x45, 0x10, 0x28, 0x67, 0x18, 0xdc, 0x2d, 0x77, 0x68, 0xf2, 0x39, 0x87, 0x52, 0xd7, 0xd1, 0x50,
    0x04, 0x45, 0xf6, 0x1d, 0xb8, 0x9d, 0xfa, 0xe8, 0xd8, 0x94, 0xe0, 0xae, 0x31, 0x66, 0xc7, 0xee, 0xa9, 0xa8, 0x51, 0xe9,
    0x14, 0x67, 0x53, 0x4c, 0x40, 0xf6, 0xee, 0x50, 0x67, 0x57, 0xf4, 0xd8, 0xd1, 0x95, 0x17, 0x2c, 0x5a, 0x72, 0x8f, 0xdc,
    0xeb, 0x6c, 0x9b, 0x24, 0xfc, 0x67, 0x11, 0x07, 0x82, 0x2a, 0xae, 0xfb, 0xd9, 0xfd, 0x89, 0x75, 0x71, 0x75, 0x74, 0x08,
    0x0d, 0xf7, 0xba, 0x5d, 0x56, 0xdf, 0x7e, 0x52, 0x5b, 0xce, 0xef, 0xeb, 0xf6, 0x32, 0x11, 0x93, 0x5f, 0xab, 0xfe, 0x08,
    0xec, 0x20, 0x18, 0x0a, 0x4e, 0xc5, 0xb6, 0x42, 0xe4, 0x70, 0xbb, 0xba, 0x1b, 0xec, 0x01, 0x9c, 0xe5, 0x1c, 0xf4, 0xd2,
    0x7d, 0x68, 0x0b, 0x30, 0x2d, 0xec, 0xda, 0x23, 0xff, 0xbd, 0x95, 0x46, 0x55, 0x59, 0xab, 0x03, 0x0f, 0xe4, 0xeb, 0x8c,
    0xca, 0xc1, 0x13, 0x28, 0x49, 0x12, 0x66, 0xd7, 0x85, 0xcb, 0x21, 0x8d, 0x8a, 0x34, 0x4c, 0x2e, 0x28, 0x25, 0x79, 0xbf,
    0x98, 0x04, 0xcd, 0x8a, 0x3e, 0xad, 0x08, 0xd3, 0x21, 0x32, 0xed, 0x54, 0x7e, 0x17, 0xfe, 0x89, 0xe9, 0x09, 0x18, 0xac,
    0x96, 0xda, 0x51, 0x61, 0x85, 0x44, 0xd2, 0xdf, 0xd9, 0x61, 0xa2, 0x44, 0x07, 0x29, 0xa5, 0xdc, 0x16, 0xac, 0x3d, 0x6e,
    0x27, 0xb6, 0x5a, 0x15, 0x87, 0x6c, 0x48, 0x3e, 0x34, 0xdd, 0xec, 0xf2, 0x7c, 0xc6, 0x87, 0x0f, 0xcf, 0xac, 0xde, 0x42,
    0x02, 0x93, 0xff, 0x4f, 0x10, 0x0f, 0x03, 0xf3, 0x1b, 0x1e, 0xaf, 0x94, 0x8e, 0x77, 0x7c, 0x66, 0x65, 0xfb, 0xd9, 0x0d,
    0xf8, 0x36, 0x0c, 0xac, 0xdc, 0x8e, 0xd1, 0x2d, 0x1e, 0x19, 0xa8, 0x87, 0x97, 0x39, 0x77, 0x98, 0x6c, 0xfb, 0x5d, 0xc1,
    0x09, 0x7e, 0x5d, 0xab, 0xde, 0xdc, 0x1f, 0x21, 0xd0, 0x7a, 0xaa, 0xeb, 0x96, 0x60, 0xf9, 0x95, 0xed, 0x13, 0xf4, 0x82,
    0xa9, 0x94, 0x7e, 0xa8, 0x5e, 0xe2, 0xca, 0x08, 0xed, 0xfe, 0x9d, 0x94, 0x3f, 0x08, 0xff, 0xd8, 0x5f, 0x9d, 0xb0, 0x84,
    0xd6, 0x6a, 0xcb, 0xc1, 0xd4, 0xd0, 0xe8, 0xe7, 0xae, 0x44, 0x28, 0x47, 0x75, 0xa6, 0x2a, 0x40, 0x0b, 0x82, 0x57, 0xd5,
    0x49, 0x40, 0x85, 0x5c, 0x92, 0x95, 0x6b, 0x7d, 0xbe, 0xcc, 0xb6, 0x2d, 0x2b, 0x71, 0x85, 0x63, 0x39, 0xfa, 0xbc, 0x19,
    0x7c, 0xe2, 0x3a, 0x81, 0xf1, 0x86, 0x9e, 0x46, 0x3d, 0x5f, 0x71, 0xa4, 0xb7, 0xd4, 0x2c, 0x98, 0xfa, 0xb5, 0x95, 0xac,
    0x33, 0x1a, 0x53, 0x9d, 0xe3, 0x4e, 0x5d, 0x4f, 0x54, 0x0d, 0xe8, 0xa4, 0x32, 0x25, 0x4d, 0x5e, 0x4c, 0xe1, 0xae, 0x46,
    0x85, 0x1e, 0xc4, 0x20, 0xdf, 0x73, 0x1e, 0x81, 0x40, 0xa2, 0x33, 0x75, 0x1e, 0xff, 0x1d, 0x79, 0x7c, 0x1f, 0x07, 0xe5,
    0x9e, 0x76, 0x9f, 0x42, 0xeb, 0xc4, 0xb4, 0xd3, 0xcb, 0x55, 0xe8, 0xbf, 0xf7, 0xeb, 0x7c, 0x2d, 0xa5, 0xfd, 0xda, 0xb4,
    0x5b, 0x1f, 0xa8, 0xd2, 0xeb, 0xdf, 0xd9, 0xf2, 0x35, 0xc6, 0x0c, 0x36, 0x66, 0x28, 0x95, 0xe2, 0x34, 0x64, 0xea, 0xaf,
    0x3a, 0x36, 0x3c, 0x38, 0x31, 0x99, 0x8d, 0x80, 0x78, 0x18, 0x48, 0x2c, 0xe9, 0xe5, 0xb7, 0xe6, 0x2e, 0x73, 0xf1, 0x7c,
    0x84, 0x21, 0x04, 0x00, 0x8e, 0x12, 0x66, 0x6c, 0x9b, 0x09, 0x88, 0xe2, 0xd5, 0xec, 0xb4, 0xdd, 0x2b, 0xda, 0x29, 0x6f,
    0x1b, 0x81, 0x21, 0xe4, 0x5c, 0x36, 0x73, 0x2d, 0x9d, 0xdd, 0xb6, 0x69, 0x21, 0x40, 0x4e, 0x70, 0xde, 0x9d, 0x0c, 0x17,
    0x1d, 0x15, 0x29, 0x5b, 0xd0, 0x66, 0x72, 0xb8, 0x38, 0x80, 0xbe, 0x9e, 0xd7, 0x5e, 0xb5, 0x72, 0x22, 0xbc,
};

const uint8_t SawyerCodingTest::rlecompresseddata[] = {
    0x02, 0x98, 0x07, 0x00, 0x00, 0x7d, 0xff, 0x3a, 0xff, 0x97, 0xff, 0x63, 0xff, 0x8b, 0xff, 0xbf, 0xff, 0xe5, 0xff, 0x6e,
    0xff, 0x0e, 0xff, 0xc4, 0xff, 0xac, 0xff, 0xdc, 0xff, 0x84, 0xff, 0xd7, 0xff, 0x68, 0xff, 0xf1, 0xff, 0x4d, 0xff, 0xcb,
    0xff, 0xaf, 0xff, 0x1e, 0xff, 0x5a, 0xff, 0x29, 0xff, 0x40, 0xff, 0x87, 0xff, 0x80, 0xff, 0x3f, 0xff, 0xf9, 0xff, 0xb8,
    0xff, 0xad, 0xff, 0x01, 0xff, 0xd3, 0xff, 0x79, 0xff, 0x3d, 0xff, 0xe9, 0xa8, 0xff, 0xa8, 0xff, 0x95, 0x48, 0xff, 0xc0,
    0xff, 0xc2, 0xc0, 0xff, 0x15, 0x68, 0xff, 0xdb, 0xff, 0xa6, 0xff, 0x90, 0xff, 0x8c, 0xff, 0x26, 0xff, 0x98, 0xff, 0x2a,
    0x38, 0xff, 0x2e, 0xff, 0x0c, 0xff, 0x82, 0xff, 0x43, 0xff, 0x00, 0xff, 0x10, 0xff, 0x6d, 0xff, 0x60, 0xff, 0xb9, 0xff,
    0xd4, 0xff, 0xed, 0xff, 0xf1, 0xff, 0x49, 0xff, 0xbb, 0xff, 0xf6, 0xff, 0x7d, 0x7f, 0xff, 0x21, 0xff, 0x24, 0xff, 0xc3,
    0xff, 0xfb, 0xff, 0x42, 0xff, 0xe1, 0xff, 0xfc, 0xff, 0xb8, 0x50, 0xff, 0x5e, 0xff, 0x01, 0xff, 0x5d, 0xff, 0x96, 0xff,
    0x2d, 0xff, 0x0f, 0xff, 0x48, 0xff, 0x12, 0xff, 0xdf, 0xff, 0x4b, 0xff, 0x6c, 0xff, 0x7e, 0xff, 0x99, 0xa0, 0xff, 0xfa,
    0xff, 0x46, 0xff, 0x7d, 0xff, 0x2c, 0xff, 0xeb, 0xff, 0xd6, 0xff, 0xf3, 0xff, 0x77, 0xff, 0xa3, 0xff, 0x85, 0xff, 0x8e,
    0xff, 0x00, 0xff, 0x34, 0xff, 0xee, 0xff, 0x73, 0xff, 0x31, 0xff, 0x76, 0xff, 0x53, 0xff, 0x17, 0xff, 0x5a, 0xff, 0x2b,
    0xff, 0x19, 0xff, 0x65, 0xff, 0x2f, 0xff, 0x32, 0x28, 0xff, 0x57, 0xff, 0xc4, 0xff, 0xf8, 0xff, 0x05, 0xff, 0x0d, 0xff,
    0xc3, 0xff, 0x59, 0xff, 0x29, 0xff, 0x9a, 0x90, 0xff, 0x3c, 0xff, 0x33, 0xff, 0x58, 0xff, 0x8c, 0xff, 0xd1, 0xff, 0x33,
    0x1f, 0xff, 0x3b, 0xff, 0xa5, 0xff, 0x66, 0xff, 0xa7, 0xff, 0xfc, 0xff, 0x7c, 0x48, 0xff, 0xa9, 0xff, 0x1a, 0xff, 0x5e,
    0x10, 0xff, 0x92, 0xff, 0x9e, 0xff, 0x71, 0xff, 0x27, 0xff, 0x07, 0xff, 0xbc, 0xe0, 0xff, 0xed, 0xff, 0xe5, 0xff, 0xc6,
    0xff, 0x0e, 0xff, 0x74, 0xff, 0xdb, 0xff, 0x15, 0xff, 0x91, 0xff, 0xd7, 0xfe, 0xff, 0x63, 0xf7, 0xff, 0x30, 0xff, 0x6e,
    0xff, 0x6a, 0xff, 0x00, 0xff, 0x40, 0x98, 0xff, 0x76, 0xff, 0x8a, 0xff, 0xb2, 0xff, 0x31, 0xff, 0xa3, 0xff, 0x81, 0xff,
    0x83, 0xff, 0xc8, 0x38, 0xff, 0x08, 0xff, 0x1f, 0x28, 0xff, 0x8d, 0xff, 0x13, 0xff, 0x3e, 0xff, 0x9a, 0xff, 0xca, 0xff,
    0x9c, 0xff, 0xd4, 0xff, 0x01, 0xff, 0x28, 0xff, 0xe6, 0xff, 0xaf, 0xff, 0xd6, 0xff, 0x2f, 0xff, 0xbc, 0xff, 0xe1, 0xff,
    0xb7, 0xff, 0xbb, 0xff, 0x48, 0xff, 0x88, 0xff, 0x86, 0xff, 0x19, 0xd8, 0xff, 0x52, 0xff, 0xfc, 0xff, 0x18, 0xff, 0xc9,
    0xff, 0xe3, 0xff, 0x90, 0xb0, 0xff, 0x6c, 0xff, 0x54, 0xff, 0x53, 0xff, 0x08, 0xff, 0x26, 0xfe, 0xff, 0x57, 0x38, 0xb0,
    0xff, 0x5e, 0xff, 0x06, 0xff, 0xa1, 0xff, 0xac, 0xff, 0xc2, 0xff, 0x83, 0xff, 0x23, 0xff, 0x0d, 0xff, 0x42, 0xff, 0xe6,
    0xff, 0xa9, 0xff, 0x81, 0x90, 0xff, 0x0b, 0xff, 0x4f, 0xff, 0x16, 0xff, 0x2a, 0xff, 0x77, 0xff, 0x3b, 0xff, 0x24, 0xff,
    0xfb, 0xff, 0x94, 0x20, 0xff, 0x78, 0xff, 0x6a, 0xff, 0xf3, 0xff, 0x17, 0xe8, 0xff, 0x60, 0xff, 0x44, 0xff, 0x58, 0xff,
    0x5c, 0xff, 0x50, 0xff, 0x80, 0xff, 0xc8, 0xff, 0x0f, 0xff, 0x87, 0xff, 0x7b, 0xff, 0x95, 0xff, 0x0a, 0xff, 0xdc, 0xff,
    0xf1, 0xff, 0x1a, 0x68, 0xff, 0xa3, 0xfe, 0xff, 0x7d, 0x79, 0xff, 0xda, 0xff, 0x14, 0xff, 0xd4, 0xff, 0x6c, 0xff, 0x7a,
    0xff, 0xfd, 0xff, 0x7e, 0xff, 0x1e, 0xff, 0xfe, 0xff, 0xd2, 0xff, 0x28, 0xff, 0xd0, 0xff, 0x06, 0x50, 0xff, 0x40, 0x18,
    0xff, 0xa1, 0xff, 0x84, 0xff, 0xc9, 0xff, 0xc0, 0xff, 0x4b, 0xff, 0xa5, 0xff, 0x97, 0xff, 0xb2, 0xff, 0xb9, 0xff, 0x20,
    0xff, 0x27, 0xff, 0x94, 0xff, 0x87, 0xff, 0x91, 0xff, 0xba, 0xc0, 0xff, 0xc6, 0xff, 0xdb, 0xff, 0x78, 0xff, 0x19, 0x50,
    0xff, 0x35, 0xff, 0x9f, 0xff, 0xbd, 0xff, 0xfb, 0xff, 0xc1, 0xff, 0x45, 0xff, 0xbe, 0xff, 0xdc, 0xff, 0x30, 0xff, 0xad,
    0xff, 0x38, 0xff, 0xd4, 0xff, 0x14, 0xff, 0x46, 0xff, 0x98, 0xff, 0x2a, 0xff, 0x90, 0xff, 0xc8, 0xff, 0x85, 0xff, 0x5e,
    0xff, 0x7e, 0xff, 0x76, 0xff, 0x0f, 0xff, 0xc4, 0xff, 0x20, 0xff, 0x07, 0xff, 0x54, 0xff, 0x7d, 0x1a, 0xff, 0x60, 0xff,
    0x3a, 0xff, 0x8c, 0xff, 0x75, 0xff, 0xb0, 0xff, 0x43, 0xff, 0xab, 0xff, 0xbb, 0xff, 0x05, 0xff, 0xca, 0xff, 0x93, 0xff,
    0xeb, 0xff, 0x18, 0xff, 0x94, 0xff, 0xf1, 0xff, 0x64, 0xff, 0x8a, 0xff, 0x01, 0xff, 0xe0, 0xff, 0xe6, 0xff, 0x24, 0xff,
    0x78, 0xff, 0x50, 0x98, 0xff, 0xd3, 0xff, 0xb9, 0xff, 0xe1, 0xff, 0xbf, 0xff, 0x1c, 0x38, 0xff, 0x5c, 0xff, 0xad, 0x68,
    0xff, 0x17, 0x70, 0xff, 0xd1, 0xff, 0xef, 0xff, 0x66, 0xff, 0x08, 0xff, 0xa7, 0xf0, 0xff, 0x6d, 0xff, 0x6a, 0xff, 0xb7,
    0xff, 0x06, 0xff, 0x31, 0xff, 0xa6, 0xff, 0x9d, 0xff, 0xe5, 0xff, 0xb2, 0xff, 0xd8, 0xff, 0xf9, 0xd0, 0xff, 0xf2, 0x20,
    0xff, 0x63, 0xff, 0xba, 0xff, 0x36, 0xff, 0x7a, 0xff, 0x53, 0xff, 0xc0, 0xff, 0x22, 0xff, 0x40, 0xff, 0x3e, 0xff, 0xa1,
    0xff, 0x95, 0x7d, 0x70, 0xff, 0xae, 0xff, 0x8c, 0xff, 0x35, 0xff, 0x7d, 0xff, 0x34, 0xd8, 0xff, 0x0c, 0xff, 0x69, 0xff,
    0x8a, 0xff, 0x0e, 0xff, 0xd4, 0x68, 0xff, 0x19, 0xff, 0xc9, 0xff, 0x90, 0xff, 0x78, 0x30, 0xff, 0x38, 0xff, 0xc8, 0xff,
    0xa0, 0xff, 0xc6, 0xff, 0x49, 0xff, 0xef, 0xff, 0xe3, 0xff, 0xb2, 0xff, 0x1d, 0xff, 0x7c, 0xff, 0xd3, 0xa8, 0xff, 0x33,
    0xff, 0x51, 0x40, 0xff, 0x2b, 0xff, 0xd2, 0xff, 0x8e, 0xff, 0xa8, 0xff, 0x6d, 0xff, 0x9c, 0xff, 0x01, 0xff, 0x06, 0xff,
    0x9f, 0xff, 0x93, 0xff, 0xa5, 0xff, 0x13, 0xff, 0x86, 0xff, 0x67, 0xff, 0x72, 0xff, 0x3e, 0xff, 0xc7, 0x00, 0xff, 0x97,
    0xff, 0xe6, 0x28, 0xff, 0xce, 0xff, 0x61, 0xff, 0x5a, 0xff, 0x57, 0xff, 0x10, 0xff, 0xf7, 0xff, 0x49, 0xff, 0x32, 0xff,
    0xea, 0xff, 0x8f, 0xb0, 0xff, 0x37, 0xe0, 0x78, 0xff, 0x25, 0x65, 0xff, 0x76, 0x38, 0xff, 0xf4, 0xff, 0x63, 0xff, 0x9c,
    0xff, 0x44, 0x48, 0xff, 0x4c, 0xff, 0x7b, 0xff, 0x3f, 0xff, 0x9b, 0x38, 0x50, 0xff, 0xb4, 0xff, 0x2a, 0xff, 0x7a, 0xa0,
    0xff, 0x03, 0xff, 0x29, 0xff, 0xf3, 0xff, 0x6f, 0xff, 0xff, 0x7d, 0x50, 0xff, 0x1a, 0xff, 0xb8, 0xff, 0x21, 0xff, 0xd7,
    0xff, 0xd0, 0xff, 0xbf, 0xff, 0x10, 0xff, 0x1e, 0xff, 0x82, 0xff, 0x96, 0xff, 0x9f, 0xff, 0xe3, 0xff, 0xc4, 0xff, 0xcf,
    0xff, 0x19, 0xff, 0x39, 0xa0, 0xff, 0x4b, 0xf8, 0xff, 0xa4, 0xff, 0x7e, 0x70, 0xff, 0x4e, 0xff, 0x3b, 0xff, 0x2c, 0xff,
    0x0c, 0xff, 0x35, 0xff, 0xe1, 0xff, 0xd4, 0xff, 0x45, 0x40, 0xff, 0x28, 0xff, 0x67, 0xff, 0x18, 0xff, 0xdc, 0xff, 0x2d,
    0xff, 0x77, 0xff, 0x68, 0xff, 0xf2, 0x40, 0xff, 0x87, 0xff, 0x52, 0x58, 0xff, 0xd1, 0xff, 0x50, 0xff, 0x04, 0x78, 0xff,
    0xf6, 0xff, 0x1d, 0xff, 0xb8, 0xff, 0x9d, 0xff, 0xfa, 0xff, 0xe8, 0xff, 0xd8, 0xff, 0x94, 0xff, 0xe0, 0xff, 0xae, 0xff,
    0x31, 0xff, 0x66, 0xff, 0xc7, 0xff, 0xee, 0xff, 0xa9, 0xff, 0xa8, 0xff, 0x51, 0xff, 0xe9, 0xff, 0x14, 0x7d, 0xff, 0x67,
    0xff, 0x53, 0xff, 0x4c, 0xff, 0x40, 0x48, 0xa8, 0x20, 0xc8, 0xff, 0x57, 0xff, 0xf4, 0x48, 0xff, 0xd1, 0xff, 0x95, 0xff,
    0x17, 0xff, 0x2c, 0xff, 0x5a, 0xff, 0x72, 0xff, 0x8f, 0xff, 0xdc, 0xff, 0xeb, 0xff, 0x6c, 0xff, 0x9b, 0xff, 0x24, 0xff,
    0xfc, 0x40, 0xff, 0x11, 0xff, 0x07, 0xff, 0x82, 0xff, 0x2a, 0xff, 0xae, 0xff, 0xfb, 0xff, 0xd9, 0xff, 0xfd, 0xff, 0x89,
    0xff, 0x75, 0xff, 0x71, 0xf0, 0xff, 0x74, 0xff, 0x08, 0xff, 0x0d, 0xff, 0xf7, 0xff, 0xba, 0xff, 0x5d, 0xff, 0x56, 0xff,
    0xdf, 0xff, 0x7e, 0xff, 0x52, 0xff, 0x5b, 0xff, 0xce, 0xff, 0xef, 0x08, 0xff, 0xf6, 0xff, 0x32, 0x20, 0xff, 0x93, 0xff,
    0x5f, 0xff, 0xab, 0xff, 0xfe, 0x60, 0xff, 0xec, 0xff, 0x20, 0xff, 0x18, 0xff, 0x0a, 0xff, 0x4e, 0xff, 0xc5, 0xff, 0xb6,
    0xff, 0x42, 0xff, 0xe4, 0x20, 0xff, 0x70, 0xff, 0xbb, 0x18, 0xff, 0x1b, 0x98, 0xff, 0x01, 0xff, 0x9c, 0xff, 0xe5, 0xff,
    0x1c, 0xff, 0xf4, 0xff, 0xd2, 0xff, 0x68, 0xff, 0x0b, 0xff, 0x30, 0xff, 0x2d, 0x40, 0xff, 0xda, 0xff, 0x23, 0xfe, 0xff,
    0x7d, 0xbd, 0xff, 0x95, 0xff, 0x46, 0xff, 0x55, 0xff, 0x59, 0xff, 0xab, 0xff, 0x03, 0xff, 0x0f, 0x20, 0xff, 0xeb, 0xff,
    0x8c, 0xff, 0xca, 0xff, 0xc1, 0xff, 0x13, 0xff, 0x28, 0xff, 0x49, 0xff, 0x12, 0xff, 0x66, 0xff, 0xd7, 0xff, 0x85, 0xff,
    0xcb, 0xff, 0x21, 0xff, 0x8d, 0xff, 0x8a, 0xff, 0x34, 0xff, 0x4c, 0xff, 0x2e, 0x98, 0xff, 0x25, 0xff, 0x79, 0xff, 0xbf,
    0xff, 0x98, 0xff, 0x04, 0xff, 0xcd, 0xa8, 0xff, 0x3e, 0xff, 0xad, 0xff, 0x08, 0xff, 0xd3, 0x70, 0xff, 0x32, 0xff, 0xed,
    0xff, 0x54, 0xff, 0x7e, 0xff, 0x17, 0xff, 0xfe, 0xff, 0x89, 0xff, 0xe9, 0xff, 0x09, 0xff, 0x18, 0xff, 0xac, 0xff, 0x96,
    0xff, 0xda, 0xff, 0x51, 0xff, 0x61, 0xff, 0x85, 0xff, 0x44, 0xff, 0xd2, 0xff, 0xdf, 0xff, 0xd9, 0xd0, 0xff, 0xa2, 0xd0,
    0xff, 0x07, 0xff, 0x29, 0xff, 0xa5, 0xff, 0x31, 0xdc, 0xff, 0x16, 0x70, 0xff, 0x3d, 0xff, 0x6e, 0xff, 0x27, 0xff, 0xb6,
    0xff, 0x5a, 0xff, 0x15, 0xff, 0x87, 0xff, 0x6c, 0xff, 0x48, 0xff, 0x3e, 0xff, 0x34, 0xff, 0xdd, 0xff, 0xec, 0xff, 0xf2,
    0xff, 0x7c, 0xff, 0xc6, 0xb0, 0xff, 0x0f, 0xff, 0xcf, 0x60, 0xff, 0xde, 0xff, 0x42, 0xff, 0x02, 0xff, 0x93, 0xfe, 0xff,
    0x7d, 0x4f, 0xff, 0x10, 0xb0, 0xff, 0x03, 0xff, 0xf3, 0xff, 0x1b, 0xff, 0x1e, 0xff, 0xaf, 0xff, 0x94, 0xff, 0x8e, 0xff,
    0x77, 0x50, 0xff, 0x66, 0xff, 0x65, 0xff, 0xfb, 0xff, 0xd9, 0xff, 0x0d, 0xff, 0xf8, 0xff, 0x36, 0xff, 0x0c, 0x30, 0xff,
    0xdc, 0x98, 0xff, 0xd1, 0x70, 0xff, 0x19, 0xff, 0xa8, 0xff, 0x87, 0xff, 0x97, 0xff, 0x39, 0x60, 0xff, 0x98, 0xff, 0x6c,
    0x68, 0xff, 0x5d, 0xff, 0xc1, 0xff, 0x09, 0xff, 0x7e, 0xe0, 0xff, 0xab, 0xff, 0xde, 0x60, 0xff, 0x1f, 0xff, 0x21, 0xff,
    0xd0, 0xff, 0x7a, 0xff, 0xaa, 0xff, 0xeb, 0xff, 0x96, 0xff, 0x60, 0xff, 0xf9, 0xff, 0x95, 0xff, 0xed, 0xff, 0x13, 0xff,
    0xf4, 0xff, 0x82, 0xff, 0xa9, 0xff, 0x94, 0x58, 0xff, 0xa8, 0xff, 0x5e, 0xff, 0xe2, 0xff, 0xca, 0xff, 0x08, 0xa0, 0xff,
    0xfe, 0xff, 0x9d, 0xb0, 0xff, 0x3f, 0xd0, 0x7d, 0xff, 0xd8, 0xf8, 0xd0, 0xff, 0xb0, 0xff, 0x84, 0xff, 0xd6, 0xff, 0x6a,
    0xff, 0xcb, 0xff, 0xc1, 0xff, 0xd4, 0xff, 0xd0, 0xff, 0xe8, 0xff, 0xe7, 0xff, 0xae, 0xff, 0x44, 0xff, 0x28, 0xff, 0x47,
    0xff, 0x75, 0xff, 0xa6, 0xff, 0x2a, 0xff, 0x40, 0xff, 0x0b, 0xff, 0x82, 0xff, 0x57, 0xff, 0xd5, 0xff, 0x49, 0xd0, 0xff,
    0x85, 0xff, 0x5c, 0xff, 0x92, 0xff, 0x95, 0xff, 0x6b, 0xff, 0x7d, 0xff, 0xbe, 0xff, 0xcc, 0xff, 0xb6, 0xff, 0x2d, 0xff,
    0x2b, 0xff, 0x71, 0xa0, 0xff, 0x63, 0xff, 0x39, 0xff, 0xfa, 0xff, 0xbc, 0xff, 0x19, 0xff, 0x7c, 0xff, 0xe2, 0xff, 0x3a,
    0xff, 0x81, 0xff, 0xf1, 0xff, 0x86, 0xff, 0x9e, 0xff, 0x46, 0xff, 0x3d, 0xff, 0x5f, 0x78, 0xff, 0xa4, 0xff, 0xb7, 0xff,
    0xd4, 0xff, 0x2c, 0xff, 0x98, 0x68, 0xff, 0xb5, 0xff, 0x95, 0xff, 0xac, 0xff, 0x33, 0x7d, 0xff, 0x1a, 0xff, 0x53, 0xff,
    0x9d, 0xff, 0xe3, 0xff, 0x4e, 0xff, 0x5d, 0xff, 0x4f, 0xff, 0x54, 0xff, 0x0d, 0xff, 0xe8, 0x60, 0xff, 0x32, 0xff, 0x25,
    0xff, 0x4d, 0xff, 0x5e, 0xff, 0x4c, 0xff, 0xe1, 0xff, 0xae, 0x00, 0xff, 0x85, 0xff, 0x1e, 0xff, 0xc4, 0xff, 0x20, 0xff,
    0xdf, 0xff, 0x73, 0xd8, 0xff, 0x81, 0xff, 0x40, 0xff, 0xa2, 0x10, 0xff, 0x75, 0xa8, 0xff, 0x1d, 0xf8, 0xff, 0x7c, 0xff,
    0x1f, 0xff, 0x07, 0xff, 0xe5, 0xff, 0x9e, 0xff, 0x76, 0xff, 0x9f, 0xff, 0x42, 0xff, 0xeb, 0x50, 0xff, 0xb4, 0xff, 0xd3,
    0xff, 0xcb, 0xff, 0x55, 0xff, 0xe8, 0xff, 0xbf, 0xff, 0xf7, 0xb8, 0x70, 0xff, 0x2d, 0xff, 0xa5, 0xff, 0xfd, 0xff, 0xda,
    0x98, 0xff, 0x5b, 0x40, 0xff, 0xa8, 0xff, 0xd2, 0x60, 0xff, 0xdf, 0xff, 0xd9, 0xff, 0xf2, 0xff, 0x35, 0xff, 0xc6, 0xff,
    0x0c, 0x7d, 0xff, 0x36, 0xff, 0x66, 0xff, 0x28, 0xff, 0x95, 0xff, 0xe2, 0xff, 0x34, 0xff, 0x64, 0xff, 0xea, 0xff, 0xaf,
    0xff, 0x3a, 0xb0, 0xff, 0x3c, 0xff, 0x38, 0xff, 0x31, 0xff, 0x99, 0xff, 0x8d, 0xff, 0x80, 0xff, 0x78, 0xff, 0x18, 0xff,
    0x48, 0xff, 0x2c, 0xff, 0xe9, 0xff, 0xe5, 0xff, 0xb7, 0xff, 0xe6, 0xff, 0x2e, 0xff, 0x73, 0xff, 0xf1, 0xff, 0x7c, 0xff,
    0x84, 0xff, 0x21, 0xff, 0x04, 0xff, 0x00, 0xff, 0x8e, 0xff, 0x12, 0xff, 0x66, 0xff, 0x6c, 0xff, 0x9b, 0xff, 0x09, 0xff,
    0x88, 0xff, 0xe2, 0xff, 0xd5, 0xff, 0xec, 0xff, 0xb4, 0xff, 0xdd, 0xff, 0x2b, 0xff, 0xda, 0xff, 0x29, 0xff, 0x6f, 0xff,
    0x1b, 0xff, 0x81, 0x58, 0xff, 0xe4, 0xff, 0x5c, 0xff, 0x36, 0x18, 0xff, 0x2d, 0xff, 0x9d, 0x90, 0xff, 0xb6, 0xff, 0x69,
    0x08, 0xff, 0x40, 0xff, 0x4e, 0xff, 0x70, 0xff, 0x27, 0xde, 0xb8, 0xff, 0x0c, 0xff, 0x17, 0xff, 0x1d, 0xff, 0x15, 0x40,
    0xff, 0x5b, 0xff, 0xd0, 0xff, 0x66, 0xff, 0x72, 0xff, 0xb8, 0xff, 0x38, 0xff, 0x80, 0xff, 0xbe, 0xff, 0x9e, 0xff, 0xd7,
    0xff, 0x5e, 0xff, 0xb5, 0xb8, 0xff, 0x22, 0xff, 0xbc,
};

const uint8_t SawyerCodingTest::rotatedata[] = {
    0x03, 0x00, 0x04, 0x00, 0x00, 0x74, 0xbc, 0x6c, 0xc5, 0x7f, 0x2f, 0xcd, 0x07, 0x89, 0x65, 0x9b, 0x42, 0xaf, 0x43, 0x3e,
    0xa6, 0x97, 0x7d, 0xc3, 0x2d, 0x52, 0x02, 0xf0, 0x40, 0x7e, 0xcf, 0x17, 0xd6, 0x02, 0x9e, 0x2f, 0x9e, 0xd3, 0x3c, 0x15,
    0xca, 0xd0, 0x06, 0x58, 0x9e, 0x2a, 0x3c, 0x7b, 0x53, 0x21, 0x64, 0xc4, 0x4c, 0x54, 0xf9, 0xc5, 0x06, 0x05, 0x1a, 0x00,
    0x08, 0xda, 0x03, 0x37, 0x6a, 0xdb, 0x8f, 0x29, 0xdd, 0xed, 0xfb, 0x24, 0x12, 0x87, 0xdf, 0x48, 0xf0, 0xf9, 0xc5, 0x50,
    0x2f, 0x02, 0xea, 0xd2, 0x96, 0x1e, 0x42, 0x42, 0xef, 0x96, 0x63, 0xcf, 0xcc, 0x02, 0xd7, 0xc8, 0xbe, 0x58, 0x5f, 0xda,
    0xf9, 0xee, 0x1d, 0xb0, 0x47, 0x00, 0xa1, 0xdd, 0xb9, 0x62, 0xb3, 0x6a, 0x8b, 0xb4, 0x59, 0x23, 0xb2, 0x5e, 0x91, 0x33,
    0xab, 0x89, 0xc7, 0xa0, 0x86, 0x87, 0xca, 0x25, 0x4d, 0x32, 0xe1, 0x66, 0x2c, 0x19, 0x8e, 0xe3, 0x9d, 0x4b, 0x33, 0xf4,
    0x7e, 0xf8, 0xcc, 0x35, 0x0d, 0xbc, 0x2b, 0x52, 0x4f, 0xe2, 0x39, 0xe0, 0x5e, 0xe2, 0x6f, 0xbc, 0x63, 0x1c, 0xa3, 0x7b,
    0x8a, 0x23, 0xbe, 0xff, 0xfb, 0x60, 0x73, 0x4d, 0x00, 0x80, 0x70, 0xce, 0x45, 0x65, 0x89, 0x74, 0xc0, 0x07, 0x46, 0xbd,
    0x04, 0x3e, 0x2f, 0xb1, 0x89, 0x7c, 0xd4, 0x59, 0x4e, 0xa9, 0x08, 0x05, 0x73, 0x5f, 0xb6, 0xe5, 0x5e, 0xc3, 0xbd, 0x77,
    0x24, 0x11, 0x34, 0x23, 0xdd, 0xa4, 0xe7, 0x03, 0xe4, 0xc7, 0x84, 0x11, 0x36, 0xa8, 0x9a, 0x01, 0x13, 0xff, 0xc1, 0x7c,
    0x2f, 0x0c, 0x0d, 0x95, 0x61, 0x07, 0x19, 0xa1, 0x21, 0xcd, 0x4d, 0x30, 0x1c, 0x16, 0x7a, 0xc2, 0x15, 0xee, 0xd9, 0x84,
    0xfd, 0x29, 0x9a, 0x0f, 0x35, 0xe7, 0xb8, 0x4d, 0x30, 0x88, 0xc2, 0x8b, 0x28, 0x01, 0x46, 0xe1, 0xc3, 0xf6, 0xac, 0x41,
    0x6e, 0xe3, 0xd0, 0x4d, 0xd1, 0xff, 0xcb, 0x5b, 0x0a, 0xa9, 0x63, 0x4f, 0xfe, 0xfc, 0xf0, 0xdf, 0x69, 0x50, 0x86, 0xc0,
    0xca, 0x80, 0x04, 0x34, 0x42, 0x93, 0x06, 0x69, 0xd2, 0x2f, 0x95, 0x37, 0x10, 0x4e, 0xa4, 0xf0, 0xc8, 0x75, 0x95, 0xd8,
    0xed, 0xf0, 0xc8, 0x08, 0x9a, 0x3f, 0xed, 0x7f, 0xe0, 0x8a, 0xf5, 0x9b, 0x18, 0x5b, 0xc1, 0x9a, 0x0a, 0x8c, 0xc4, 0x45,
    0x48, 0x91, 0x2c, 0xcb, 0x3f, 0xec, 0x78, 0x98, 0x10, 0x0e, 0xa2, 0x43, 0x30, 0x74, 0x64, 0xae, 0x58, 0x86, 0x5d, 0x77,
    0x82, 0x95, 0x9c, 0x7d, 0x0c, 0x29, 0x8f, 0x8c, 0x45, 0x02, 0x07, 0xdc, 0x12, 0xf0, 0x82, 0x72, 0xe9, 0x73, 0x0f, 0xf7,
    0x0e, 0x61, 0xe2, 0xb5, 0x4a, 0x2e, 0x54, 0x3a, 0xf7, 0xcc, 0x40, 0xf4, 0x04, 0xda, 0x53, 0xf6, 0x03, 0x62, 0x35, 0xb3,
    0xf2, 0x65, 0xc6, 0x3f, 0x53, 0xe5, 0x0f, 0x6c, 0x5d, 0x6c, 0xd3, 0x6a, 0x60, 0x44, 0x02, 0xc7, 0xd0, 0x2b, 0x2f, 0xd5,
    0x46, 0x6a, 0xeb, 0x86, 0x57, 0x18, 0x4b, 0x51, 0x07, 0xa9, 0x9a, 0x23, 0xe4, 0x21, 0xc3, 0xc6, 0x1c, 0x91, 0x05, 0xd8,
    0xa4, 0xdf, 0x1f, 0x56, 0x8e, 0xf8, 0x9e, 0x07, 0x99, 0xa2, 0x4b, 0x65, 0x69, 0x1d, 0x45, 0xad, 0x4e, 0x02, 0x30, 0xf3,
    0xc9, 0x4b, 0x98, 0xd0, 0xb3, 0xe4, 0xf1, 0xf8, 0x1c, 0x2f, 0x37, 0xa3, 0x67, 0xc2, 0xd2, 0xea, 0x08, 0xef, 0x4a, 0x46,
    0x75, 0x1f, 0x76, 0xe6, 0x75, 0x70, 0x2b, 0xce, 0x43, 0xe9, 0x1b, 0x93, 0x22, 0xcd, 0x62, 0x6f, 0x9f, 0x37, 0x0b, 0xfe,
    0x5a, 0x54, 0xd3, 0x93, 0x81, 0x52, 0x9f, 0xed, 0xff, 0xec, 0xd0, 0x17, 0x90, 0xaf, 0x86, 0xf7, 0x08, 0x3c, 0x14, 0xd2,
    0xcf, 0xc7, 0x26, 0xf9, 0x8c, 0x72, 0x86, 0x69, 0xa5, 0x49, 0xf3, 0xfa, 0x27, 0x76, 0x61, 0x81, 0x9a, 0xc3, 0xa6, 0xa8,
    0x08, 0x50, 0x3b, 0x03, 0x6e, 0x5a, 0xbb, 0x0d, 0x79, 0x72, 0x3c, 0x4a, 0xeb, 0xa3, 0x82, 0x80, 0xa2, 0xed, 0xe8, 0x17,
    0xce, 0xf5, 0x47, 0x1b, 0x4a, 0xc1, 0x75, 0x26, 0x33, 0x8f, 0x77, 0x35, 0x54, 0xa2, 0x4f, 0x82, 0xb3, 0xa6, 0x62, 0x08,
    0x7b, 0xdd, 0x82, 0xec, 0xab, 0xe9, 0xc6, 0x3a, 0xca, 0x2e, 0x61, 0x4b, 0x39, 0x1f, 0xe6, 0x7d, 0x36, 0x37, 0x21, 0x9f,
    0xb3, 0x22, 0x38, 0x50, 0x15, 0x5d, 0xdf, 0x3b, 0xfe, 0x13, 0xab, 0x2e, 0xba, 0xe8, 0x40, 0xa1, 0xfb, 0x75, 0xea, 0xca,
    0xef, 0xfc, 0x92, 0x6b, 0x67, 0xdf, 0x5f, 0xde, 0x19, 0x22, 0x9c, 0xeb, 0xd5, 0xfd, 0x40, 0x9d, 0x10, 0x30, 0x50, 0xc9,
    0xe2, 0x6d, 0x12, 0x9c, 0x38, 0x77, 0xd5, 0x63, 0x76, 0x02, 0xe4, 0xbc, 0x0e, 0xe9, 0x96, 0x0d, 0x85, 0x60, 0x69, 0x9d,
    0x6d, 0x46, 0xff, 0xb7, 0xca, 0x8c, 0xaa, 0x2b, 0xd5, 0x06, 0x78, 0x9c, 0xf5, 0x19, 0x56, 0x38, 0x89, 0x50, 0x4a, 0x42,
    0x33, 0xaf, 0x2c, 0x79, 0x90, 0x1b, 0x54, 0x86, 0x26, 0x5c, 0x41, 0xa4, 0xbc, 0x7f, 0xc4, 0x80, 0xe6, 0x15, 0xf1, 0xb5,
    0x04, 0xa7, 0x09, 0x46, 0xf6, 0xa8, 0xf3, 0xe2, 0x7f, 0x13, 0x4f, 0x21, 0x0c, 0x59, 0xb4, 0x5b, 0xa8, 0xc2, 0x2c, 0x88,
    0x69, 0xbf, 0xce, 0x2c, 0x51, 0x88, 0x38, 0x25, 0xd2, 0xb9, 0xb0, 0x95, 0x9e, 0xdc, 0x39, 0xd6, 0x2d, 0x2a, 0x3c, 0x8d,
    0x24, 0x7c, 0xa1, 0xbb, 0x76, 0xe5, 0xe3, 0xd8, 0xc3, 0x1e, 0x7e, 0x95, 0x6f, 0x84, 0x10, 0x72, 0xff, 0x9e, 0x80, 0xe1,
    0x81, 0xe7, 0xd8, 0xc3, 0xd7, 0x29, 0x74, 0xee, 0x3e, 0xcc, 0x2b, 0x7f, 0xec, 0x1a, 0xc7, 0xc6, 0x06, 0x59, 0xe6, 0xd1,
    0xe8, 0x3c, 0xc8, 0x15, 0xc3, 0x2f, 0xc9, 0xee, 0x4c, 0xd8, 0xdf, 0xab, 0xe0, 0x12, 0xf3, 0xab, 0xd5, 0xbd, 0xe6, 0xe3,
    0x90, 0xa1, 0xd3, 0x55, 0xf5, 0x2d, 0x03, 0x3f, 0xca, 0xdb, 0x98, 0x9e, 0x41, 0x53, 0xa4, 0xcf, 0x54, 0xbc, 0x17, 0x59,
    0x04, 0xdb, 0xf7, 0xb3, 0x4a, 0x7e, 0x40, 0x1b, 0x6c, 0x3b, 0x85, 0x90, 0x6b, 0xd4, 0x5e, 0x38, 0x6a, 0xa1, 0x47, 0xfc,
    0x57, 0x88, 0x41, 0xe8, 0xba, 0x4d, 0x51, 0x08, 0x85, 0x05, 0xba, 0xba, 0xa4, 0x80, 0x2c, 0x8b, 0x49, 0x2b, 0x5b, 0xaf,
    0x5f, 0x99, 0xb5, 0xa5, 0x95, 0xe2, 0x2c, 0x6c, 0x9c, 0xf5, 0xe5, 0x23, 0x3e, 0xc5, 0xd1, 0x30, 0xf8, 0x0d, 0xf4, 0xc8,
    0x9e, 0xbe, 0x8b, 0x94, 0xdb, 0xa9, 0x61, 0x13, 0x7d, 0x6b, 0xac, 0x95, 0x99, 0x34, 0x9a, 0xb3, 0xf1, 0x9c, 0xea, 0xe9,
    0x2a, 0x1a, 0x47, 0x94, 0x19, 0x4a, 0x6a, 0xcb, 0x26, 0xc3, 0x75, 0xc8, 0xc2, 0x3c, 0x26, 0x04, 0xef, 0xe6, 0xf0, 0x30,
    0x20, 0x45, 0x99, 0xae, 0x0f, 0x3a, 0xe8, 0x8f, 0x8f, 0x0e, 0x2f, 0xd3, 0x3b, 0x3f, 0x12, 0x7d, 0x62, 0x69, 0x9e, 0x79,
    0xaa, 0xd1, 0xfd, 0xfe, 0xf5, 0xf8, 0x69, 0xb4, 0xfe, 0xb5, 0xa5, 0x6b, 0x8f, 0x51, 0x96, 0x7d, 0xef, 0xb3, 0x97, 0xa6,
    0x63, 0x18, 0xb1, 0xcc, 0x14, 0x2b, 0x17, 0x86, 0x32, 0xd5, 0x7d, 0x47, 0x1b, 0x78, 0xc1, 0x26, 0xcc, 0x1b, 0x04, 0x0f,
    0x0c, 0x90, 0x61, 0x3d, 0xf2, 0x6f, 0x37, 0xc5, 0xb9, 0xe3, 0xe3, 0x90, 0x90, 0x08, 0x00, 0xd1, 0x09, 0xcc, 0x63, 0x73,
    0x84, 0x11, 0x17, 0xba, 0x76, 0x69, 0xee, 0x65, 0x6d, 0x52, 0x7b, 0x63, 0xc0, 0x42, 0x27, 0x8b, 0x1b, 0xe6, 0x69, 0xb3,
    0xee, 0x6d, 0x4b, 0x24, 0x20, 0x9c, 0x83, 0xdb, 0xce, 0x18, 0xb8, 0xa3, 0x8a, 0x52, 0xda, 0x1a, 0x33, 0xe4, 0xc5, 0x07,
    0x40, 0x7d, 0xf4, 0xfa, 0x2f, 0x6b, 0x93, 0x44, 0x5e,
};

const uint8_t SawyerCodingTest::invalid1[] = {
    0x02, 0x04, 0x00, 0x00, 0x00, 0xd4, 0xff, 0xce, 0xcf, 0x2e, 0x00,
};

const uint8_t SawyerCodingTest::invalid2[] = {
    0x02, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x27, 0x3f, 0x01, 0x00,
};

const uint8_t SawyerCodingTest::invalid3[] = {
    0x02, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x73,
};

const uint8_t SawyerCodingTest::invalid4[] = {
    0x02, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
};

const uint8_t SawyerCodingTest::invalid5[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0xdf, 0x00, 0x00,
};

const uint8_t SawyerCodingTest::invalid6[] = {
    0x02, 0x02, 0x00, 0x00, 0x00, 0x28, 0x7f,
};

const uint8_t SawyerCodingTest::invalid7[] = {
    0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
};

// This needs to be non-empty to satisfy MSVC. We still pass zero as length.
const uint8_t SawyerCodingTest::empty[] = {
    0x00,
};
