#!/bin/bash
#### Note: command must be run by an account or service principal with remctl wallet access
## i.e.
## service/wallet-tester-dev
## service/wallet-tester-test
## service/wallet-tester-uat
## service/wallet-tester-prod
## 
## and the group acl is group/wallet-tester

## the script does a basic reate, store, get and destroy on a wallet file object
## For example, to create the Wallet file object password/wallet-tester/something1/something2:
##
## REMOTE_USER=service/wallet-tester-dev@stanford.edu REMOTE_HOST=1.1.1.1 wallet-backend autocreate keytab service/testing-uxrg6gs
## REMOTE_USER=service/wallet-tester-dev@stanford.edu REMOTE_HOST=1.1.1.1 wallet-backend get  keytab service/testing-uxrg6gs
## REMOTE_USER=service/wallet-tester-dev@stanford.edu REMOTE_HOST=1.1.1.1 wallet-backend destroy keytab service/testing-uxrg6gs

# default thresholds go here
nagios_servers=""

passive=0
#critd=_____
# fixed conditionals

# describe your check for output and perf
description="wallet keytab crud test"
itemname="wallet_keytab_crud"

function usage () {
cat <<-USAGE
    Usage: $0 [-s <service principal>] [-i <item>] [-p <nagios_servers>] -d -h
    Nagios/Icinga check for $description including creating and deleting a keytab object
    Note: this is intended to run on the backend of the wallet server
    -s <service principal>: the service principal that has the right to create keytabs
    -i <item>: item to crud. Default is keytab service/testing-uxrg6gs
    -p <nagios_servers> Use Nagios passive mode to submit to <nagios_servers>
    -d: Set debug flag on
    -h: this help message
USAGE
}

# getops define them here, and put in usage
while getopts "s:i:p:dh" OPT; do
    case "$OPT" in
        s) principal="$OPTARG" ;;
        i) thing="$OPTARG" ;;
        p) nagios_servers="$OPTARG" ;;
        d) debug=true ;;
        h) usage
           exit 3                                     ;;
        *) echo "Unrecognized option: $OPT" >&2
           echo >&2
           usage
           exit 3                                   ;;
    esac
done

## use defaults if we didn't pass in thresholds
if [[ x"$principal" == x"" ]]; then
    usage
    echo; echo "UNKNOWN: service principal is required"
    exit 3
fi

if [[ x"$thing" == x"" ]]; then
    thing="service/testing-uxrg6gs"
fi

if [[ x"$nagios_servers" != x"" ]]; then
    passive=1
fi

export REMOTE_USER=$principal@stanford.edu
export REMOTE_HOST=1.1.1.1 

# destroy the keytab in case it is still there from a previous run
wallet-backend destroy keytab "$thing"  2>&1 > /dev/null

if [ $debug ]; then
    echo "Variables:"
    echo "principal: $principal"
    echo "objectname: $thing"
    echo "passive? $passive"
    echo "Evir: "
    env | grep "REMOTE"
    if [[ $passive -eq 1 ]]; then
        echo "nagios servers? $nagios_servers"
    fi
    echo "debug? $debug"
fi

## the way this is done is by evaluating a number of tests
## C - create 
## U - update contents
## R - read contents
## D - destroy the item
## A successful test would pass all tests == CURD
## Any failure would yield a 0 value => false

# set default fail
C=0  # create
U=0  # update the object
R=0  # get the object
F=0  # show the object
K=0  # use the object
D=0  # destroy the object

# keep a running total
total=$((C + U + R + F + K + D))

if [ $debug ]; then
    echo "Total = $total, should be 0"
fi

# create
#create=$( wallet -s "$server" autocreate file "$thing" )
create=$( wallet-backend create keytab "$thing" )
ccode=${?}

if [[ $ccode -ne 0 ]]; then
    msg="Create failed: $create"
    C=0
else
    msg="Create ok."
    C=1
fi

if [ $debug ]; then
    echo "Actual raw results:"
    echo "create = \"$create\", exit code = $ccode"
fi

total=$((C + U + R + F + K + D))
if [ $debug ]; then
    echo "Total = $total, should be 1"
fi

## now some logic:
## if you couldn't create the wallet object, there's no point in doing anything else

if [[ $total -eq 1 ]]; then

    # set the owner, needed for next step
    update=$(  wallet-backend owner keytab "$thing" "$principal"  )
    ucode=${?}

    if [[ $ucode -ne 0 ]]; then
        msg="Update failed: $update"
        U=0
    else
        msg="$msg Update ok."
        U=1
    fi

    if [ $debug ]; then
        echo "Actual raw results:"
        echo "update = \"$update\", exit code = $ucode"
    fi
fi

# this should be 2 if successful
total=$((C + U + R + F + K + D))
if [ $debug ]; then
    echo "Total = $total, should be 2"
fi

if [[ $total -eq 2 ]]; then

    # read the file
    readk=$(  wallet-backend get keytab $thing > /tmp/keytab.test )
    rcode=${?}

    if [[ $rcode -ne 0 ]]; then
        msg="Get failed: $readk"
        R=0
    else
        msg="$msg Get ok."
        R=1
    fi

    if [ $debug ]; then
        lsed=$( ls -l /tmp/keytab.test )
        echo "Actual raw results:"
        echo "keytab file \"$lsed\", exit code = $rcode"
    fi

fi

# this should be 3 if successful
total=$((C + U + R + F + K + D))
if [ $debug ]; then
    echo "Total = $total, should be 3"
fi

if [[ $total -eq 3 ]]; then

    # show
    fdiff=$(  wallet-backend show keytab "$thing" )
    fcode=${?}

    if [[ $fcode -ne 0 ]]; then
        msg="$msg Show keytab failed: $fdiff"
        F=0
    else
        msg="$msg Show keytab ok."
        F=1
    fi

    if [ $debug ]; then
        echo "Actual raw results:"
        echo "show = \"$fdiff\", exit code = $fcode"
    fi
fi

# this should be 4 if successful
total=$((C + U + R + F + K + D))
if [ $debug ]; then
    echo "Total = $total, should be 4"
fi

if [[ $total -eq 4 ]]; then

   # use the keytab
    kuse=$(  k5start -Uf /tmp/keytab.test -- sh -c "ls -lrlt /root" )
    kcode=${?}

    if [[ $kcode -ne 0 ]]; then
        msg="$msg Can't use the keytab: $kuse"
        K=0
    else
        msg="$msg Use ok."
        K=1
    fi

    if [ $debug ]; then
        echo "Actual raw results:"
        echo "use = \"$kuse\", exit code = $kcode"
    fi

    rm  /tmp/keytab.test

fi

# this should be 5 if successful
total=$((C + U + R + F + K + D))
if [ $debug ]; then
    echo "Total = $total, should be 5"
fi


if [[ $total -eq 5 ]]; then

    # destroy the keytab
    destroy=$(  wallet-backend destroy keytab "$thing" )
    dcode=${?}

    if [[ $dcode -ne 0 ]]; then
        msg="$msg Can't destroy: $destroy"
        D=0
    else
        msg="$msg Destroy ok."
        D=1
    fi

    if [ $debug ]; then
        echo "Actual raw results:"
        echo "destroy = \"$destroy\", exit code = $dcode"
    fi
fi

# this should be 6 if successful
total=$((C + U + R + F + K + D))
if [ $debug ]; then
    echo "Total = $total, should be 6"
fi

# now do the return, order matters here, note the exits
if [[ $total -ne 6 ]] ; then
    output="CRITICAL: $description failed: $msg "
    code=2
elif [[ $total -eq 6 ]] ; then
    output="OK: $description: $msg "
    code=0
else
    code=3
    output="UNKNOWN: $itemname doesn't return a valid value for $description, or something else is wrong: $msg ($code)"
fi

if [[ $passive -eq "1" ]] ; then
    if [[ $debug ]]; then
        echo "Submitting passive results"
    fi

    for nagios_server in $(tr ',' ' ' <<< "$nagios_servers")
    do 
        echo "$server~$itemname~$code~$output" | /usr/sbin/send_nsca -d "~" -H $nagios_server
        if [[ $debug ]]; then
            echo "Passive result sent to $nagios_server: $server~$itemname~$code~$output"
        fi
    done
    exit 0
fi

## unless the passives have gone and exited, we get to here
if [[ $debug ]]; then
    echo "Active result sent: $output code $code sent for $server $itemname"
fi

echo "$output"
exit $code


