#!/bin/bash

# Normal
#(root) purpleperil:~> walletl show keytab host/ldap0.stanford.edu | grep -i unchanging
#(root) purpleperil:~> echo $?
#1
#
# Unchanging
#(root) purpleperil:~> walletl show keytab service/ldap | grep -i unchanging
#          Flags: unchanging
#(root) purpleperil:~> echo $?
#0
 
#### Note: command must be run by an account or service principal get access to the 
# principal in question

## This test has two forms - normal and unchanging.  If the keytab to be downloaded
## shows as having the flag unchanging, then the test will look for both downloads 
## to be identical 


# default thresholds go here
nagios_servers=""
passive=0
#critd=_____
# fixed conditionals

# describe your check for output and perf
description="wallet keytab test"
itemname="wallet_keytab"

function usage () {
cat <<-USAGE
    Usage: $0 [-s <wallet-server>] -k <keytab> [-p <nagios_servers>] -d -h
    Nagios/Icinga check for $description.
    -s <wallet server>: wallet server name is required, default is 'hostname -s'
    -k <keytab>: keytab to test. Must be specified
    -p <nagios_servers> Use Nagios passive mode to submit to <nagios_servers>
    -d: Set debug flag on
    -h: this help message
Note: You must run this with an unchanging keytab on the read-only 
      replicas, otherwise you will get false alarms.
USAGE
}

# This is a unified report, that can be called anywhere
function report () {

    if [[ $passive -eq "1" ]] ; then
        if [[ $debug ]]; then
            echo "Submitting passive results"
        fi

        for nagios_server in $(tr ',' ' ' <<< "$nagios_servers")
        do 
            echo "$server~$itemname~$code~$output" | /usr/sbin/send_nsca -d "~" -H $nagios_server
            if [[ $debug ]]; then
                echo "Passive result sent to $nagios_server: $server~$itemname~$code~$output"
            fi
        done
        exit 0
    fi 

    ## unless the passives have gone and exited, we get to here
    if [[ $debug ]]; then
        echo "Active result sent: $output $code for $server $itemname"
    fi

    echo $output
    exit $code
}

# getops define them here, and put in usage
while getopts "s:k:p:dh" OPT; do
    case "$OPT" in
        s) server="$OPTARG" ;;
        k) keytab="$OPTARG" ;;
        p) nagios_servers="$OPTARG" ;;
        d) debug=true ;;
        h) usage
           exit 3                                     ;;
        *) echo "Unrecognized option: $OPT" >&2
           echo >&2
           usage
           exit 3                                   ;;
    esac
done

## use defaults if we didn't pass in thresholds
if [[ x"$keytab" == x"" ]]; then
    output="UNKNOWN: Keytab name not passed to script"
    code=3
    report
    
fi

if [[ x"$server" == x"" ]]; then
    server=$(hostname -s)
fi

if [[ x"$nagios_servers" != x"" ]]; then
    passive=1
fi

show=$(wallet -s $server show keytab $keytab | grep -i unchanging  )
changing=${?}

if [[ $debug ]]; then
    echo "Keytab $keytab has ($show) and changing is $changing"
fi

if [[ x"$changing" != x"0" ]]; then
    itemname="wallet_keytab_normal"
else
    itemname="wallet_keytab_unchanging"
fi

if [[ $debug ]]; then
    echo "itemname is $itemname"
fi

### download the keytab twice

# first download goes here
result1=$(wallet -s $server get keytab $keytab > /tmp/test1.keytab  )
code1=${?}

if [[ x$code1 != x"0" ]]; then
    output="CRITICAL: Can't download keytab"
    code=2
    report
fi

result1a=$( ktutil -k /tmp/test1.keytab list | sed -e "/\/tmp/d" > /tmp/test1.list)
code1a=${?}

if [[ x$code1a != x"0" ]]; then
    output="CRITICAL: Can't list the keytab of the first download"
    code=2
    report
fi

# second download goes here
result2=$(wallet -s $server get keytab $keytab > /tmp/test2.keytab  )
code2=${?}

if [[ x$code2 != x"0" ]]; then
    output="CRITICAL: Can't download keytab"
    code=2
    report
fi

result2a=$( ktutil -k /tmp/test2.keytab list | sed -e "/\/tmp/d" > /tmp/test2.list)
code2a=${?}

if [[ x$code2a != x"0" ]]; then
    output="CRITICAL: Can't list the keytab of the second download"
    code=2
    report
fi

result3=$(diff /tmp/test1.list /tmp/test2.list)
code3=${?}

if [[ x"$changing" != x"0" ]]; then

    if [[ x$code3 == x"0" ]]; then
        output="CRITICAL: Normal keytab did not change between downloads"
        code=2
        report
    elif [[ x$code3 == x"1" ]]; then
        output="OK: Normal keytab changed between downloads"
        code=0
        report
    fi
else
# else means unchanging is set
    if [[ x$code3 == x"0" ]]; then
        output="OK: Unchanging keytab did not change between downloads"
        code=2
        report
    elif [[ x$code3 == x"1" ]]; then
        output="Critical: Unchanging keytab changed between downloads"
        code=0
        report
    fi
fi

### you should never get here
output="UNKNOWN: $item doesn't return a valid value for $description, or something else is wrong: $result ($code)"
code=3
report





