#!/bin/bash -e

# Check latest wallet file object between master and replica servers
# Run this test on replica where it checks local latest wallet file object and compares
# with the file object on the master server by running a remctl command against master

# default thresholds go here
dir="/srv/wallet/files"
nagios_servers=""
passive=0

# describe your check for output and perf
description="wallet file object replication test"
itemname="wallet_file_replication"

replica_server=$( hostname -s )

function usage () {
cat <<-USAGE
    Usage: $0 -s <master-server> [-p <nagios_servers>] -d -h
    Nagios/Icinga check for $description
    -s <master server>: master server name is required
    -p <nagios_servers> Use Nagios passive mode to submit to <nagios_servers>
    -d: Set debug flag on
    -h: this help message
USAGE
}

# getops define them here, and put in usage
while getopts "s:p:dh" OPT; do
    case "$OPT" in
        s) server="$OPTARG" ;;
        p) nagios_servers="$OPTARG" ;;
        d) debug=true ;;
        h) usage
           exit 2                                     ;;
        *) echo "Unrecognized option: $OPT" >&2
           echo >&2
           usage
           exit 2                                   ;;
    esac
done

## checks
if [[ ! -d "$dir" ]]; then
    echo; echo "Wallet file object directory is Missing"
    exit 3
fi

if [[ x"$server" == x"" ]]; then
    usage
    echo; echo "UNKNOWN: master server is required"
    exit 3
fi

if [[ x"$nagios_servers" != x"" ]]; then
    passive=1
fi

if [[ $debug ]]; then
    echo "Master server is $server"
    echo "Wallet file object directory is $dir"
fi

## Check latest file object on master
master_latest_file=$( remctl "$server" wallet-latest-file )

## Check latest file object on replica
replica_latest_file=$( ls -1rt ${dir} | tail -1 )

if [[ $debug ]]; then
    echo "master latest file is $master_latest_file"
    echo "replica latest file is $replica_latest_file"
fi

## Check latest file epoch time on master
master_file_epoch=$( remctl "$server" wallet-latest-file file-epoch-time )
replica_file_epoch=$( date +%s -r "${dir}"/"${replica_latest_file}" )

#replica_file_epoch=$(find "${dir}"/"${replica_latest_file}"  -printf '%C@\n' | sort -u | awk -F'.' '{print $1}')

DIFF=$(( master_file_epoch - replica_file_epoch ))

if [[ $debug ]]; then
   echo ${master_file_epoch}
   echo ${replica_file_epoch}
   echo ${DIFF}
fi

## IF Multiple condition example
#if [[ ( "$val" -ge "36000" ) && ( "$val" -le "42000" ) ]] || [[ ( "$val" -ge "60000" ) && ( "$val" -le "66000" ) ]]

# The shell tries to interpret 08 as an octal number, as it starts with a zero. Only digits 0-7 are, however,
# allowed in octal, as decimal 8 is octal 010. Hence 08 is not a valid number, and that's the reason for the error.

## For example: file name "2f" is considered as Octal value by shell
# https://stackoverflow.com/questions/24777597/value-too-great-for-base-error-token-is-08
# put x infront of variables before checking conditions to fix this issue.

## MAIN 
if [[ ( "x${master_latest_file}" -eq "x${replica_latest_file}" ) ]] && [[ "${DIFF}" -lt "1800" ]] ; then 
    output="OK: Wallet file objects are in Sync"
    code=0
elif [[ "x${master_latest_file}" -eq "x${replica_latest_file}" ]] && [[ "${DIFF}" -lt "3600" ]] ; then
    output="WARNING: Wallet file objects epoch time not in Range "
    code=1
elif [[ "x${master_latest_file}" != "x${replica_latest_file}" ]] && [[ "${DIFF}" -gt "900" ]] ; then
    output="CRITICAL: Wallet file objects are not in SYNC"
    code=2
else 
    output="OK: Wallet file objects are not in SYNC - Waiting for master-replica sync"
    code=0
fi 


## Send Nagios Passive checks
if [[ $passive -eq "1" ]] ; then

    if [[ $debug ]]; then
        echo "Submitting passive results"
    fi

    for nagios_server in $(tr ',' ' ' <<< "$nagios_servers")
    do
        echo "$replica_server~$itemname~$code~$output" | /usr/sbin/send_nsca -d "~" -H $nagios_server
        if [[ $debug ]]; then
            echo $replica_server
            echo $itemname
            echo $output
            echo "Passive result sent to $nagios_server: $replica_server~$itemname~$code~$output"
        fi
    done
    exit 0
fi

## unless the passives have gone and exited, we get to here
if [[ $debug ]]; then
    echo "Active result sent: $output $code for $replica_server $itemname"
fi

echo "$output"
exit $code
