#!/bin/bash

#### Note: command must be run by an account or service principal with remctl wallet access
## i.e.
## service/wallet-tester-dev
## service/wallet-tester-test
## service/wallet-tester-uat
## service/wallet-tester-prod
## 
## and the group acl is group/wallet-tester

## the script does a basic reate, store, get and destroy on a wallet file object
## For example, to create the Wallet file object password/wallet-tester/something1/something2:
##
## wallet autocreate file password/wallet-tester/something1/something2
## wallet store      file password/wallet-tester/something1/something2 -f /tmp/input.txt
## wallet get        file password/wallet-tester/something1/something2 -f /tmp/output.txt
## wallet destroy    file password/wallet-tester/something1/something2

# default thresholds go here
nagios_servers=""

passive=0
#critd=_____
# fixed conditionals

# describe your check for output and perf
description="wallet crud test"
itemname="wallet_crud"

function usage () {
cat <<-USAGE
    Usage: $0 -s <wallet-server> [-i <item>] [-p <nagios_servers>] -d -h
    Nagios/Icinga check for $description including creating and deleting a file object
    -s <wallet server>: wallet server name is required
    -i <item>: item to crud. Default is password/wallet-tester/\$wallet-server/testword
    -p <nagios_servers> Use Nagios passive mode to submit to <nagios_servers>
    -d: Set debug flag on
    -h: this help message
USAGE
}

# getops define them here, and put in usage
while getopts "s:i:p:dh" OPT; do
    case "$OPT" in
        s) server="$OPTARG" ;;
        i) thing="$OPTARG" ;;
        p) nagios_servers="$OPTARG" ;;
        d) debug=true ;;
        h) usage
           exit 3                                     ;;
        *) echo "Unrecognized option: $OPT" >&2
           echo >&2
           usage
           exit 3                                   ;;
    esac
done

## use defaults if we didn't pass in thresholds
if [[ x"$server" == x"" ]]; then
    usage
    echo; echo "UNKNOWN: server is required"
    exit 3
fi

if [[ x"$thing" == x"" ]]; then
    thing="password/wallet-tester/$server/testword"
fi

if [[ x"$nagios_servers" != x"" ]]; then
    passive=1
fi

if [ $debug ]; then
    echo "Variables:"
    echo "server: $server"
    echo "objectname: $thing"
    echo "passive? $passive"
    if [[ $passive -eq 1 ]]; then
        echo "nagios servers? $nagios_servers"
    fi
    echo "debug? $debug"
fi


## the way this is done is by evaluating a number of tests
## C - create 
## U - update contents
## R - read contents
## D - destroy the item
## A successful test would pass all tests == CURD
## Any failure would yield a 0 value => false

# set default fail
C=0  # create
P=0  # generate a file
U=0  # update the object
R=0  # read the object
F=0  # get a null diff
D=0  # destroy the object

# keep a running total
total=$((C + P + U + R + F + D))

if [ $debug ]; then
    echo "Total = $total, should be 0"
fi


# create
create=$( wallet -s "$server" autocreate file "$thing" )
ccode=${?}

if [[ $ccode -ne 0 ]]; then
    msg="Create failed: $create"
    C=0
else
    msg="Create ok."
    C=1
fi

if [ $debug ]; then
    echo "Actual raw results:"
    echo "autocreate = \"$create\", exit code = $ccode"
fi

total=$((C + P + U + R + F + D))
if [ $debug ]; then
    echo "Total = $total, should be 1"
fi

## now some logic:
## if you couldn't create the wallet object, there's no point in doing anything else

if [[ $total -eq 1 ]]; then

    # generate file to store
    rm -f /tmp/input"$server".txt
    out=$( base64 /dev/urandom | base64 | tr -dc 'a-zA-Z0-9' | fold -w 32 | head -n 1 > /tmp/input"$server".txt )
    ocode=${?}

    if [[ $ocode -ne 0 ]]; then
        msg="$msg Can't generate password: $out"
        P=0
    else
        msg="$msg File made."
        P=1
    fi

    if [ $debug ]; then
        file=$( cat /tmp/input"$server".txt) 
        echo "Actual raw results:"
        echo "out = \"$out\", $file exit code = $ocode"
    fi
fi

# this should be 2 if both were successful
total=$((C + P + U + R + F + D))
if [ $debug ]; then
    echo "Total = $total, should be 2"
fi

if [[ $total -eq 2 ]]; then

    # store the file
    update=$(wallet -s "$server" store file "$thing" -f /tmp/input"$server".txt )
    ucode=${?}

    if [[ $ucode -ne 0 ]]; then
        msg="Update failed: $update"
        U=0
    else
        msg="$msg Update ok."
        U=1
    fi

    if [ $debug ]; then
        echo "Actual raw results:"
        echo "update = \"$update\", exit code = $ucode"
    fi
fi

# this should be 3 if successful
total=$((C + P + U + R + F + D))
if [ $debug ]; then
    echo "Total = $total, should be 3"
fi

if [[ $total -eq 3 ]]; then

    # read the file
    readf=$(wallet -s "$server" get file "$thing" -f /tmp/output"$server".txt )
    rcode=${?}

    if [[ $rcode -ne 0 ]]; then
        msg="Read failed: $readf"
        R=0
    else
        msg="$msg Read ok."
        R=1
    fi

    if [ $debug ]; then
        file=$( cat /tmp/output"$server".txt) 
        echo "Actual raw results:"
        echo "read = \"$readf\", $file exit code = $ocode"
    fi
fi

# this should be 4 if successful
total=$((C + P + U + R + F + D))
if [ $debug ]; then
    echo "Total = $total, should be 4"
fi

if [[ $total -eq 4 ]]; then

    # compare in with out
    fdiff=$( diff /tmp/input"$server".txt /tmp/output"$server".txt )
    fcode=${?}

    if [[ $fcode -ne 0 ]]; then
        msg="$msg File retrieved differs from stored: $fdiff"
        F=0
    else
        msg="$msg File diff ok."
        F=1
    fi

    if [ $debug ]; then
        echo "Actual raw results:"
        echo "diff = \"$fdiff\", exit code = $fcode"
    fi
fi

# this should be 5 if successful
total=$((C + P + U + R + F + D))
if [ $debug ]; then
    echo "Total = $total, should be 5"
fi

if [[ $total -eq 5 ]]; then

    # destroy the keytab
    destroy=$(wallet -s "$server" destroy file "$thing" )
    dcode=${?}

    if [[ $dcode -ne 0 ]]; then
        msg="$msg Can't destroy: $destroy"
        D=0
    else
        msg="$msg Destroy ok."
        D=1
    fi

    if [ $debug ]; then
        echo "Actual raw results:"
        echo "destroy = \"$destroy\", exit code = $dcode"
    fi
fi

# this should be 6 if successful
total=$((C + P + U + R + F + D))
if [ $debug ]; then
    echo "Total = $total, should be 6"
fi

# now do the return, order matters here, note the exits
if [[ $total -ne 6 ]] ; then
	output="CRITICAL: $description failed: $msg "
    code=2
elif [[ $total -eq 6 ]] ; then
    output="OK: $description: $msg "
    code=0
else
    code=3
	output="UNKNOWN: $itemname doesn't return a valid value for $description, or something else is wrong: $msg ($code)"
fi

if [[ $passive -eq "1" ]] ; then
    if [[ $debug ]]; then
        echo "Submitting passive results"
    fi

    for nagios_server in $(tr ',' ' ' <<< "$nagios_servers")
    do 
        echo "$server~$itemname~$code~$output" | /usr/sbin/send_nsca -d "~" -H $nagios_server
        if [[ $debug ]]; then
            echo "Passive result sent to $nagios_server: $server~$itemname~$code~$output"
        fi
    done
    exit 0
fi

## unless the passives have gone and exited, we get to here
if [[ $debug ]]; then
    echo "Active result sent: $output code $code sent for $server $itemname"
fi

echo "$output"
exit $code
