#!/usr/bin/perl
#
# scheduler - Interface for users to request and edit cron service jobs
#
# Written by Jon Robertson <jonrober@stanford.edu>
# Copyright 2009 Board of Trustees, Leland Stanford Jr. University
# Updated by Adam H. Lewenberg <adamhl@stanford.edu> in 2022.

## no critic (Modules::RequireNoMatchVarsWithUseEnglish);

#############################################################################
# Modules and declarations
#############################################################################

use Carp;
use CGI::FormBuilder;
use Data::Dumper;
use English;
use POSIX qw(strftime);
use Readonly;
use Time::HiRes qw(gettimeofday tv_interval);
use YAML::Tiny;

use Stanford::WWWScheduler qw(
  call_remctl
  db_connect
  make_kerberos_cache
  read_config
  read_password
);

#These packages are no longer needed:
#use AFS::PAG qw(setpag unlog);# Only used for Stanford::Infrared::AFS::pts_exists?
#use Stanford::Infrared::General qw(get_db_setup);
#use Stanford::Infrared::AFS qw(pts_exists);

use strict;
use warnings;
use autodie;
use vars qw(@RANGE_MINUTES @RANGE_HOURS @RANGE_DAYS_WEEK @RANGE_DAYS_MONTH
  @RANGE_MONTHS @RANGE_DAYS_WEEK_SHORT @RANGE_MONTHS_SHORT
  @RANGE_DAYS_MONTH_SHORT @RANGE_DAYS_MONTH_COMMON @INTERVALS $REMCTL
  $AKLOG %TYPES);

### CONFIGURATION
my $config_file = '/etc/www-scheduler/www-scheduler-config.yaml';
my %CONFIG      = read_config($config_file);

### Some DEBUG configuration
my $DEBUG;
if (exists($CONFIG{'debug'}) && ($CONFIG{'debug'} =~ m{true}ixsm)) {
    $DEBUG = 1;
} else {
    $DEBUG = 0;
}

my $TEMPL_DIR = '/usr/share/www-scheduler/templates/';

# We generate a short description from the full description.
# $SHORT_DESCRIPTION_LENGTH is the cut-off length.
Readonly my $SHORT_DESCRIPTION_LENGTH => 20;

# Database setup values.
my $DBH = db_connect(\%CONFIG);

# We get the cgi_principals _once_ (since it is unlikely to change
# during the execution of the script).
my @CGI_PRINCIPALS = ();

## Some URLs
my %URL;
$URL{'cgi-princ-setup'}  = 'https://tools.stanford.edu/cgi-bin/cgi-request';
$URL{'cron-princ-setup'} = 'https://tools.stanford.edu/cgi-bin/cron-request';

# Types of cronjobs
%TYPES = (
    'QuarterHourly' => 'Every Quarter Hour',
    'HalfHourly'    => 'Every Half Hour',
    'Hourly'        => 'Every Hour at %s',
    'Daily'         => 'Every Day at %s',
    'Weekly'        => 'Every %s at %s',
    'Monthly'       => 'Every Month on the %s at %s',
    'Yearly'        => 'Every %s at %s',
    'Custom'        => 'Custom Schedule',
);

# Set the ranges of minutes, hours, days/week, days/month, months.
## no critic (ValuesAndExpressions::ProhibitMagicNumbers);
## no critic (ValuesAndExpressions::ProhibitNoisyQuotes);
@RANGE_MINUTES = (
    [0,  'at :00'],
    [5,  'at :05'],
    [10, 'at :10'],
    [15, 'at :15'],
    [20, 'at :20'],
    [25, 'at :25'],
    [30, 'at :30'],
    [35, 'at :35'],
    [40, 'at :40'],
    [45, 'at :45'],
    [50, 'at :50'],
    [55, 'at :55'],
);

@RANGE_HOURS = (
    ['*', 'Every Hour', q{}],
    [0,   '00',         'Or, at specific hours:'],
    [1,   '01',         'Or, at specific hours:'],
    [2,   '02',         'Or, at specific hours:'],
    [3,   '03',         'Or, at specific hours:'],
    [4,   '04',         'Or, at specific hours:'],
    [5,   '05',         'Or, at specific hours:'],
    [6,   '06',         'Or, at specific hours:'],
    [7,   '07',         'Or, at specific hours:'],
    [8,   '08',         'Or, at specific hours:'],
    [9,   '09',         'Or, at specific hours:'],
    [10,  '10',         'Or, at specific hours:'],
    [11,  '11',         'Or, at specific hours:'],
    [12,  '12',         'Or, at specific hours:'],
    [13,  '13',         'Or, at specific hours:'],
    [14,  '14',         'Or, at specific hours:'],
    [15,  '15',         'Or, at specific hours:'],
    [16,  '16',         'Or, at specific hours:'],
    [17,  '17',         'Or, at specific hours:'],
    [18,  '18',         'Or, at specific hours:'],
    [19,  '19',         'Or, at specific hours:'],
    [20,  '20',         'Or, at specific hours:'],
    [21,  '21',         'Or, at specific hours:'],
    [22,  '22',         'Or, at specific hours:'],
    [23,  '23',         'Or, at specific hours:'],
);

@RANGE_DAYS_WEEK = (
    ['*', 'Every Day of Week', q{}],
    [0,   'Sunday',            'Or, on specific days:'],
    [1,   'Monday',            'Or, on specific days:'],
    [2,   'Tuesday',           'Or, on specific days:'],
    [3,   'Wednesday',         'Or, on specific days:'],
    [4,   'Thursday',          'Or, on specific days:'],
    [5,   'Friday',            'Or, on specific days:'],
    [6,   'Saturday',          'Or, on specific days:'],
);

@RANGE_DAYS_MONTH = (
    ['*', 'Every Day of Month', q{}],
    [1,   '1st',                'Or, on specific days:'],
    [2,   '2nd',                'Or, on specific days:'],
    [3,   '3rd',                'Or, on specific days:'],
    [4,   '4th',                'Or, on specific days:'],
    [5,   '5th',                'Or, on specific days:'],
    [6,   '6th',                'Or, on specific days:'],
    [7,   '7th',                'Or, on specific days:'],
    [8,   '8th',                'Or, on specific days:'],
    [9,   '9th',                'Or, on specific days:'],
    [10,  '10th',               'Or, on specific days:'],
    [11,  '11st',               'Or, on specific days:'],
    [12,  '12nd',               'Or, on specific days:'],
    [13,  '13rd',               'Or, on specific days:'],
    [14,  '14th',               'Or, on specific days:'],
    [15,  '15th',               'Or, on specific days:'],
    [16,  '16th',               'Or, on specific days:'],
    [17,  '17th',               'Or, on specific days:'],
    [18,  '18th',               'Or, on specific days:'],
    [19,  '19th',               'Or, on specific days:'],
    [20,  '20th',               'Or, on specific days:'],
    [21,  '21st',               'Or, on specific days:'],
    [22,  '22nd',               'Or, on specific days:'],
    [23,  '23rd',               'Or, on specific days:'],
    [24,  '24th',               'Or, on specific days:'],
    [25,  '25th',               'Or, on specific days:'],
    [26,  '26th',               'Or, on specific days:'],
    [27,  '27th',               'Or, on specific days:'],
    [28,  '28th',               'Or, on specific days:'],
    [29,  '29th',               'Or, on specific days:'],
    [30,  '30th',               'Or, on specific days:'],
    [31,  '31st',               'Or, on specific days:'],
);

@RANGE_MONTHS = (
    ['*', 'Every Month', q{}],
    [1,   'January',     'Or, on specific months:'],
    [2,   'February',    'Or, on specific months:'],
    [3,   'March',       'Or, on specific months:'],
    [4,   'April',       'Or, on specific months:'],
    [5,   'May',         'Or, on specific months:'],
    [6,   'June',        'Or, on specific months:'],
    [7,   'July',        'Or, on specific months:'],
    [8,   'August',      'Or, on specific months:'],
    [9,   'September',   'Or, on specific months:'],
    [10,  'October',     'Or, on specific months:'],
    [11,  'November',    'Or, on specific months:'],
    [12,  'December',    'Or, on specific months:'],
);

## use critic
## no critic (CodeLayout::ProhibitParensWithBuiltins);
## no critic (ErrorHandling::RequireCheckingReturnValueOfEval);

# Shorter versions for the simple interface.
@RANGE_MONTHS_SHORT = @RANGE_MONTHS;
shift @RANGE_MONTHS_SHORT;
@RANGE_DAYS_MONTH_SHORT = @RANGE_DAYS_MONTH;
shift @RANGE_DAYS_MONTH_SHORT;
@RANGE_DAYS_MONTH_COMMON = @RANGE_DAYS_MONTH;
shift @RANGE_DAYS_MONTH_COMMON;
pop @RANGE_DAYS_MONTH_COMMON;
pop @RANGE_DAYS_MONTH_COMMON;
pop @RANGE_DAYS_MONTH_COMMON;
@RANGE_DAYS_WEEK_SHORT = @RANGE_DAYS_WEEK;
shift @RANGE_DAYS_WEEK_SHORT;

@INTERVALS = qw(Hourly Daily Weekly Monthly Yearly Custom);

# Initialize the form, template, and define all the form fields.
our $FORM = CGI::FormBuilder->new(
    method     => 'POST',
    name       => 'installer',
    stylesheet => 1,
    header     => 1,
    sticky     => 1,
);

#############################################################################
# Form Setup
#############################################################################

sub fields_main {
    $FORM->field(
        name => 'cronjob',
        type => 'hidden',
    );

    return;
}

# Fields to define for the edit form.
sub fields_edit {
    my ($user) = @_;

    # The field we're editing.
    $FORM->field(
        name => 'cronjob',
        type => 'hidden',
    );

    # Default values.
    my @jobs = ();
    if ($FORM->field('cronjob')) {
        @jobs = get_user_jobs(uid(), $FORM->field('cronjob'));
    }
    @jobs = textify_times(@jobs);
    my $job = $jobs[0];


    ## no critic (ValuesAndExpressions::RequireInterpolationOfMetachars);
    my $email  = $job->{'cr_email'}  || uid() . '@stanford.edu';
    my $active = $job->{'cr_active'} || 'Yes';

    # Principals we can run as.
    my @principals = cgi_principals($user);
    $FORM->field(
        name     => 'cr_principal',
        label    => 'Run this command as this principal:',
        options  => \@principals,
        type     => 'select',
        validate => \&validate_principal,
        required => 1,
        value    => $job->{'cr_principal'},
        message  => 'Please select a principal to run this command',
    );

    # Command to run.
    $FORM->field(
        name     => 'cr_command',
        label    => 'Command:',
        type     => 'text',
        size     => 50,
        required => 1,
        validate => \&validate_command,
        value    => $job->{'cr_command'},
        message  => 'Please fill in a command',
    );

    # Active or not?
    $FORM->field(
        name    => 'cr_active',
        label   => 'Make the Job Active?',
        type    => 'select',
        options => [
            'Yes',
            'No'
        ],
        labels => {
            'Yes' => 'Yes, execute as directed',
            'No'  => 'No, save for future use',
        },
        value      => $active,
        selectname => 0,
    );

    # Email fields
    $FORM->field(
        name     => 'cr_email',
        label    => 'Send email to:',
        type     => 'text',
        required => 1,
        validate => 'EMAIL',
        value    => $email,
        message  => 'Please enter a valid email addresss',
    );

    $FORM->field(
        name    => 'cr_email_output',
        label   => 'Mail command output?',
        type    => 'select',
        options => [
            1,
            0,
        ],
        labels => {
            1 => 'Yes, email both output and errors',
            0 => 'No, email only errors',
        },
        value      => $job->{'cr_email_output'} || 0,
        selectname => 0,
    );

    # Interval to run at.
    $FORM->field(
        name => 'cr_description',
        label =>
          'Description:<br> For future reference, what does the command do?',
        type  => 'textarea',
        value => $job->{'cr_description'},
        rows  => 5,
        cols  => 50,
    );

    # Interval to run at.
    my $type = $job->{'cr_type'} || 'Daily';
    $FORM->field(
        name    => 'cr_type',
        label   => 'Run Interval',
        options => \@INTERVALS,
        type    => 'select',
        value   => $type,
    );

    # Months, days of month, days of week, hours, minutes

    my (@months, @days_month, @days_week, @hours);

    if ($job->{'cr_months'}) {
        @months = split(/,/xsm, $job->{'cr_months'});
    }
    if ($job->{'cr_days_month'}) {
        @days_month = split(/,/xsm, $job->{'cr_days_month'});
    }
    if ($job->{'cr_hours'}) {
        @days_week = split(/,/xsm, $job->{'cr_days_week'});
        @hours     = split(/,/xsm, $job->{'cr_hours'});
    }
    $FORM->field(
        name       => 'cr_months',
        label      => 'Months',
        options    => \@RANGE_MONTHS,
        type       => 'select',
        value      => \@months,
        multiple   => 1,
        size       => 5,
        optgroups  => 1,
        selectname => 0,
    );

    $FORM->field(
        name       => 'cr_days_month',
        label      => 'Days each month',
        options    => \@RANGE_DAYS_MONTH,
        type       => 'select',
        value      => \@days_month,
        multiple   => 1,
        size       => 5,
        optgroups  => 1,
        selectname => 0,
    );

    $FORM->field(
        name       => 'cr_days_week',
        label      => 'Days each week',
        options    => \@RANGE_DAYS_WEEK,
        type       => 'select',
        value      => \@days_week,
        multiple   => 1,
        size       => 5,
        optgroups  => 1,
        selectname => 0,
    );

    $FORM->field(
        name       => 'cr_hours',
        label      => 'Hours',
        options    => \@RANGE_HOURS,
        type       => 'select',
        value      => \@hours,
        multiple   => 1,
        size       => 5,
        optgroups  => 1,
        selectname => 0,
    );

    $FORM->field(
        name       => 'cr_minutes',
        label      => 'Minutes',
        options    => \@RANGE_MINUTES,
        type       => 'select',
        value      => $job->{'cr_minutes'},
        selectname => 0,
    );

    # Special refields for the simple types.
    $FORM->field(
        name       => 'cr_months_yearly',
        options    => \@RANGE_MONTHS_SHORT,
        type       => 'select',
        value      => $job->{'cr_months'},
        selectname => 0,
    );

    $FORM->field(
        name       => 'cr_days_month_monthly',
        options    => \@RANGE_DAYS_MONTH_COMMON,
        type       => 'select',
        value      => $job->{'cr_days_month'},
        selectname => 0,
    );

    $FORM->field(
        name       => 'cr_days_month_yearly',
        options    => \@RANGE_DAYS_MONTH_SHORT,
        type       => 'select',
        value      => $job->{'cr_days_month'},
        selectname => 0,
    );

    $FORM->field(
        name       => 'cr_days_week_weekly',
        label      => 'Days each week',
        options    => \@RANGE_DAYS_WEEK_SHORT,
        type       => 'select',
        value      => $job->{'cr_days_week'},
        selectname => 0,
    );

    return;
}

#############################################################################
# Form validation
#############################################################################

# Validate a user principal.  Make sure that the user has access to it.
sub validate_principal {
    my $princ = shift;

    my @principals = cgi_principals(uid());
    if (defined $princ && $princ ne q{}) {
        foreach (@principals) {
            return 1 if $_ eq $princ;
        }
    }

    # Error.
    return;
}

# Validate a command.  We only care that it's not empty.
sub validate_command {
    my $command = shift;
    return 1 if defined $command && $command ne q{};

    # Error.
    return;
}

#############################################################################
# User identity and impersonation
#############################################################################

# Return $ENV{'uid'} unless we are impersonating someone.
sub uid {
    my $impersonate = get_impersonate();
    if ($impersonate) {
        return $impersonate;
    }

    # If we get here there is no impersonation.
    if (!exists($ENV{'uid'})) {
        croak 'ENV{uid} not defined';
    } else {
        return $ENV{'uid'};
    }
}


# We are impersonating another user if
#
#  1. the environment variable IMPERSONATE_USER is set to "yes" or "true"
#     (case is ignored), AND
#  2. there is a file /etc/www-scheduler/impersonate.txt.
#
# The impersonated user id is the first line of /etc/www-scheduler/impersonate.txt.

sub is_impersonating {
    my $impersonate_flag  = 0;
    if (exists($ENV{'IMPERSONATE_USER'})) {
        $impersonate_flag = $ENV{'IMPERSONATE_USER'};
    }

    if ($impersonate_flag =~ m{^(yes|true)$}ixsm) {
        $impersonate_flag = 1;
    }
    return $impersonate_flag;
}

sub get_impersonate {
    if (is_impersonating()) {
        return read_impersonate_file()
    } else {
        return;
    }
}

sub read_impersonate_file {
    my $impersonate_file = '/etc/www-scheduler/impersonate.txt';
    my $first_line;
    if (-f $impersonate_file) {
        open(my $FH, '<', $impersonate_file);
        $first_line = <$FH>;
        chomp($first_line);
        close $FH;
    }
    return $first_line;
}

# This returns the formatted version of the authenticated user, either
# "sunetid" (if not impersonation) or "sunetid (impersonation)" (if
# impersonating).
sub user_formatted {
    my $impersonate = get_impersonate();
    if ($impersonate) {
        return uid() . ' (impersonation)';
    } else {
        return uid();
    }
}


#############################################################################
# Database interfaces
#############################################################################

sub abbreviate_text {
    my ($text) = @_;

    if (!$text) {
        return $text;
    } else {
        if (length($text) > $SHORT_DESCRIPTION_LENGTH) {
            return substr($text, 0, $SHORT_DESCRIPTION_LENGTH) . '...';
        } else {
            return $text;
        }
    }
}

# Given a username, return the entries for all jobs belonging to that user.
# Returns an array of hashrefs, each hashref an entry in the table.
sub get_user_jobs {
    my ($user, $id) = @_;

    # Get the principals that we allow the user to see.
    my @principals = cgi_principals($user);
    my %principals;
    foreach (@principals) { $principals{$_} = 1 }
    $principals{ $user . '/cgi' }  = 1;
    $principals{ $user . '/cron' } = 1;

    # Get all cronjobs, as we filter them later.
    my ($query, $sth);
    if (defined $id && $id) {
        $query = 'SELECT * from cronjobs WHERE cr_id=?';
        $sth   = $DBH->prepare($query);
        $sth->execute($id);
    } else {
        $query = 'SELECT * from cronjobs';
        $sth   = $DBH->prepare($query);
        $sth->execute();
    }

    # Go through the results and only save those owned by a principal we
    # can see.
    my @jobs = ();
    while (my $ref = $sth->fetchrow_hashref()) {

        # Add an abbreviated description.
        $ref->{'cr_description_short'} =
          abbreviate_text($ref->{'cr_description'});

        # If job is not active define a text indicator that we will
        # use in the output template.
        if ($ref->{'cr_active'} eq 'No') {
            $ref->{'inactive_indicator'} = '(inactive)';
        } else {
            $ref->{'inactive_indicator'} = q{};
        }

        if (exists($principals{ $ref->{'cr_principal'} })) {
            push(@jobs, $ref);
        }
    }

    return @jobs;
}

# Given a row id and a username, return 1 if that job belongs to that user,
#  0 if not.
sub is_owned {
    my ($job, $user) = @_;

    my $query = 'SELECT * from cronjobs where cr_requester=? AND cr_id=?';
    my $sth   = $DBH->prepare($query);
    $sth->execute($user, $job);

    return 1 if (my $ref = $sth->fetchrow_hashref());
    return 0;
}

# Given a record (hash mapping fieldname to field value), insert a new cron
# entry into the database.  Return 0 for failure, otherwise the record id.
sub create_job {
    my (%job_record) = @_;
    my ($count, @values);

    start_timer();

    # Add in the requester.
    $job_record{'cr_requester'} = uid();

    # Before inserting, set cr_pts_group to the empty string if not
    # already set.
    if (!$job_record{'cr_pts_group'}) {
        $job_record{'cr_pts_group'} = q{};
    }

    # Get the correct number of ? marks for subbing in.
    $count = keys %job_record;
    for (1 .. $count) { push(@values, q{?}) }

    # Add the record.  Create the insert statement in-place from the record
    # given us.
    my $insert_id;
    $DBH->{'AutoCommit'} = 0;
    eval {
        my $jr_keys   = join(', ', keys %job_record);
        my $jr_values = join(', ', @values);
        my $sql       = "INSERT INTO cronjobs ($jr_keys) VALUES ($jr_values)";

        my $do_result = $DBH->do($sql, undef, values %job_record);
        $insert_id = $DBH->last_insert_id();
        if ($insert_id) {
            progress("insertid found: $insert_id");
        } else {
            progress('insertid NOT found');
        }
        $DBH->commit;
    };
    if ($EVAL_ERROR) {
        $DBH->rollback;
        progress('had to rollback the transaction');
        return 0;
    }
    $DBH->{'AutoCommit'} = 1;

    show_elapsed_time('create_job');

    return $insert_id;
}

# Given a record (hash of the data for a single row of the cronjobs table),
#  update that record in the database.  Returns 0 for failure, 1 for success.
sub edit_job {
    my ($id, %job_record) = @_;
    my (@values);

    start_timer();

    # Get the correct number of ? marks for subbing in.
    foreach my $key (keys %job_record) { push(@values, $key . q{=?}) }

    #my @v2;
    #foreach my $key (keys %job_record) { push(@v2, $key.'='.$record{$key}) }
    #warn 'UPDATE cronjobs SET ' . join(', ', @v2) . " WHERE cr_id=$id\n";

    # Edit the record, creating the update statement from the values given
    #  us.
    $DBH->{'AutoCommit'} = 0;
    eval {
        my $values_joined = join(', ', @values);
        my $sql           = "UPDATE cronjobs SET $values_joined WHERE cr_id=?";
        $DBH->do($sql, undef, values %job_record, $id);
        $DBH->commit;
    };
    if ($EVAL_ERROR) {
        $DBH->rollback;
        return 0;
    }
    $DBH->{'AutoCommit'} = 1;

    show_elapsed_time('edit_job');

    return 1;
}

# Given a job id, update the last touched time of that job to now and
# set cr_active to 'Yes'.
sub renew_job {
    my ($id) = @_;

    start_timer();

    my $now = strftime('%Y-%m-%d %T', localtime(time));
    $DBH->{'AutoCommit'} = 0;
    eval {
        my $sql = 'UPDATE cronjobs SET cr_modified=?, cr_active = ? WHERE cr_id=?';
        $DBH->do($sql, undef, $now, 'Yes', $id);
        $DBH->commit;
    };
    if ($EVAL_ERROR) {
        $DBH->rollback;
        return 0;
    }
    $DBH->{'AutoCommit'} = 1;

    show_elapsed_time('renew_job');

    return 1;
}

# Given the id of a specific row in the cronjobs table, delete that row.
#  Return 0 if failure, 1 if success.
sub remove_job {
    my ($job) = @_;

    $DBH->{'AutoCommit'} = 0;
    eval {
        my $sql = 'DELETE FROM cronjobs WHERE cr_id=?';
        $DBH->do($sql, undef, $job);
        $DBH->commit;
    };
    if ($EVAL_ERROR) {
        $DBH->rollback;
        return 0;
    }
    $DBH->{'AutoCommit'} = 1;
    return 1;
}

#############################################################################
# Misc functions
#############################################################################

# Check to see if a PTS group exists. Makes a remctl call to a server that
# has access to AFS/PTS. Replaces Stanford::Infrared::AFS::pts_exists
# since we want to run this code in a container that will not have access
# to AFS.
#
# $group - PTS group name
#
# Returns: 1 if the group exists
#          0 if it does not
#          0 and a warning on any other errors from the PTS command
sub pts_exists {
    my ($group) = @_;

    my $remctl_host = $CONFIG{'remctl'}->{'afstools'}->{'host'};
    my $service     = $CONFIG{'remctl'}->{'afstools'}->{'service'};

    my ($stdout, $stderr, $rc) =
      call_remctl($remctl_host, $service, \%CONFIG, 'pts-examine', $group);
    if ($rc == 0) {
        return 1;
    } else {
        return 0;
    }
}

# Return a list of cgi principals this user has access to.  We have to
# cheat a bit and guess the names, so might miss some.  Check to make sure
# each guessed name does actually have a PTS entry to verify.
sub cgi_principals {
    my ($sunet) = @_;

    if (@CGI_PRINCIPALS) {
        progress('using cached value of cgi_principals');
        return @CGI_PRINCIPALS;
    }

    start_timer();

    # Check for principals for the sunetid itself.
    my (@principals);
    for my $type ('cgi', 'cron') {
        if (pts_exists($sunet . q{.} . $type)) {
            push(@principals, $sunet . q{/} . $type);
        }
    }

    # Get the group and department directories the user is admin for.
    my $remctl_host = $CONFIG{'remctl'}->{'afsdir'}->{'host'};
    my $service     = $CONFIG{'remctl'}->{'afsdir'}->{'service'};

    my ($stdout, $stderr, $status) =
      call_remctl($remctl_host, $service, \%CONFIG, 'afsdir', 'directories', $sunet);
    if ($status != 0) {
        return ();
    }

    # Convert $stdout into an array of directories.
    my @dirs = split(/\n/xsm, $stdout);

    # Find principals on those directories.

    # (2022-03-31) Note from adamhl: The old code replaced the first
    # underscore in the principal with a hyphen before looking for the
    # principal. Why does it do this? I do not know. Maybe Jon Robertson
    # knows. In any event, there _are_ valid PTS groups that _do_ have an
    # underscore so to work around this we check for both the altered and
    # unaltered PTS group.

    # Example: if one of the directories is "/afs/ir/group/kipac_teabot"
    # the corresponding cgi principal _before_ replacement is
    # "group-kipac_teabot". Looking at AFS we see that the actual group
    # that owns "/afs/ir/group/kipac_teabot" is "group-kipac_teabot". So
    # far so good. But if we did what the old code did and blindly
    # replaced the underscore with a hyphen we would think the owning
    # group is "group-kipac-teabot" which is not even a valid PTS group.
    foreach my $dir (@dirs) {
        if ($dir =~ m{^/afs/ir/(dept|group)/([^/]+)}xsm) {
            my $princ = lc($1 . q{-} . $2);
            if (pts_exists($princ . '.cgi')) {
                push(@principals, $princ . '/cgi');
            } else {
                $princ =~ s{_}{-}xsm;
                if (pts_exists($princ . '.cgi')) {
                    push(@principals, $princ . '/cgi');
                }
            }
        }
    }

    @CGI_PRINCIPALS = @principals;
    show_elapsed_time('cgi_principals');

    return (@principals);
}

# Given a sunetid, return 1 if the /cron principal for that sunetid exists,
# 0 otherwise.  We actually check to see if the PTS user for the cron entry
# exists rather than the entry itself, just for consistency with gathering
# the valid principals.
sub cronprinc_exists {
    my ($sunet) = @_;

    start_timer();

    my $rv;
    if (pts_exists($sunet . '.cron')) {
        $rv = 1;
    } else {
        $rv = 0;
    }
    show_elapsed_time('cronprinc_exists');

    return $rv;
}

# Given a sunetid, return 1 if the /cgi principal for that sunetid exists,
# 0 otherwise.  We actually check to see if the PTS user for the cgi entry
# exists rather than the entry itself, just for consistency with gathering
# the valid principals.
sub cgiprinc_exists {
    my ($sunet) = @_;

    start_timer();

    my $rv;
    if (pts_exists($sunet . '.cgi')) {
        $rv = 1;
    } else {
        $rv = 0;
    }
    show_elapsed_time('cgiprinc_exists');

    return $rv;
}

#############################################################################
# Time calculations
#############################################################################

# Find the time to run the next hourly cronjob request at.  We want to find
#  the earliest time with the fewest existing hourly cronjobs for that time.
#  Returns the time in minutes that the cronjob should run at every hour.
sub next_hourly {
    #<<<  perltidy ignore this
    my $query = q(SELECT cr_minutes, count(cr_minutes) FROM cronjobs ) .
                q(WHERE cr_type='Hourly' GROUP BY cr_minutes);
    #>>>
    my $sth = $DBH->prepare($query);
    $sth->execute();

    # Initialize the counts of any valid item.
    my (%counts, @valid);
    ## no critic (ValuesAndExpressions::ProhibitMagicNumbers);
    @valid = (0, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55);
    foreach (@valid) { $counts{$_} = 0 }

    # Overwrite with the ones in the database.
    while (my $ref = $sth->fetchrow_hashref()) {
        if (
            !(
                defined $ref->{'cr_minutes'}
                || ($ref->{'cr_minutes'} !~ /^\d+$/xsm)
            )
          )
        {
            next;
        }
        my $key = $ref->{'cr_minutes'};
        $counts{$key} = $ref->{'count(cr_minutes)'};
    }

    return show_lowest(%counts);
}

sub next_daily {
    #<<<  perltidy ignore this
    my $query = q(SELECT cr_minutes, cr_hours ) .
                q(FROM cronjobs WHERE cr_type='Daily');
    #>>>
    my $sth = $DBH->prepare($query);
    $sth->execute();

    # Initialize the counts of any valid item.  Keep each time in military
    #  format to track hours and minutes both.
    ## no critic (ValuesAndExpressions::ProhibitMagicNumbers);
    my (%counts, @minutes, @hours);
    @minutes = (0, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55);
    @hours   = (0, 1, 2,  3,  4,  5,  6,  7,  8,  9,  10, 11);
    foreach my $h (@hours) {
        foreach my $m (@minutes) { $counts{ $h . q{:} . $m } = 0 }
    }

    # Overwrite with the ones in the database.
    while (my $ref = $sth->fetchrow_hashref()) {
        if (!(defined $ref->{'cr_minutes'} && defined $ref->{'cr_hours'})) {
            next;
        }
        my $key = $ref->{'cr_hours'} . q{:} . $ref->{'cr_minutes'};
        $counts{$key}++;
    }

    return split(/:/xsm, show_lowest(%counts));
}

sub next_weekly {
    my ($day) = @_;

    #<<<  perltidy ignore this
    my $query = q(SELECT cr_minutes, cr_hours FROM cronjobs ) .
                q(WHERE cr_type='Weekly' AND cr_days_week=?);
    #>>>
    my $sth = $DBH->prepare($query);
    $sth->execute($day);

    # Initialize the counts of any valid item.  Keep each time in military
    #  format to track hours and minutes both.
    ## no critic (ValuesAndExpressions::ProhibitMagicNumbers);
    my (%counts, @minutes, @hours);
    @minutes = (0,  5,  10, 15, 20, 25, 30, 35, 40, 45, 50, 55);
    @hours   = (12, 13, 14, 15);
    foreach my $h (@hours) {
        foreach my $m (@minutes) { $counts{ $h . q{:} . $m } = 0 }
    }

    # Overwrite with the ones in the database.
    while (my $ref = $sth->fetchrow_hashref()) {
        if (!(defined $ref->{'cr_minutes'} && defined $ref->{'cr_hours'})) {
            next;
        }
        my $key = $ref->{'cr_hours'} . q{:} . $ref->{'cr_minutes'};
        $counts{$key}++;
    }

    return split(/:/xsm, show_lowest(%counts));
}

sub next_monthly {
    my ($day_of_month) = @_;

    #<<<  perltidy ignore this
    my $query = q(SELECT cr_minutes, cr_hours FROM cronjobs ) .
                q(WHERE cr_type='Monthly' AND cr_days_month=?);
    #>>>
    my $sth = $DBH->prepare($query);
    $sth->execute($day_of_month);

    # Initialize the counts of any valid item.  Keep each time in military
    #  format to track hours and minutes both.
    ## no critic (ValuesAndExpressions::ProhibitMagicNumbers);
    my (%counts, @minutes, @hours);
    @minutes = (0,  5,  10, 15, 20, 25, 30, 35, 40, 45, 50, 55);
    @hours   = (16, 17, 18, 19, 20, 21);
    foreach my $h (@hours) {
        foreach my $m (@minutes) { $counts{ $h . q{:} . $m } = 0 }
    }

    # Overwrite with the ones in the database.
    while (my $ref = $sth->fetchrow_hashref()) {
        if (!(defined $ref->{'cr_minutes'} && defined $ref->{'cr_hours'})) {
            next;
        }
        my $key = $ref->{'cr_hours'} . q{:} . $ref->{'cr_minutes'};
        $counts{$key}++;
    }

    return split(/:/xsm, show_lowest(%counts));
}

sub next_yearly {
    my ($month, $day_of_month) = @_;

    #<<<  perltidy ignore this
    my $query = q(SELECT cr_minutes, cr_hours FROM cronjobs ) .
                q(WHERE cr_type='Yearly' AND cr_months=? AND cr_days_month=?);
    #>>>
    my $sth = $DBH->prepare($query);
    $sth->execute($month, $day_of_month);

    # Initialize the counts of any valid item.  Keep each time in military
    #  format to track hours and minutes both.
    ## no critic (ValuesAndExpressions::ProhibitMagicNumbers);
    my (%counts, @minutes, @hours);
    @minutes = (0,  5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55);
    @hours   = (22, 23);
    foreach my $h (@hours) {
        foreach my $m (@minutes) { $counts{ $h . q{:} . $m } = 0 }
    }

    # Overwrite with the ones in the database.
    while (my $ref = $sth->fetchrow_hashref()) {
        if (!(defined $ref->{'cr_minutes'} && defined $ref->{'cr_hours'})) {
            next;
        }
        my $key = $ref->{'cr_hours'} . q{:} . $ref->{'cr_minutes'};
        $counts{$key}++;
    }

    return split(/:/xsm, show_lowest(%counts));
}

# Replace '*' with 0 (needed for strftime).
sub unasterisk {
    my ($x) = @_;

    if ($x eq q{*}) {
        return 0;
    } else {
        return $x;
    }
}

# Given an array of references to database record rows, add in and return
#  a version with one additional 'column', a version of the time at which
#  the job runs suited for readability.
sub textify_times {
    my (@jobs) = @_;
    my (@textified);

    foreach my $job (@jobs) {
        ## no critic (ControlStructures::ProhibitCascadingIfElse);
        # Figure out what data we need for each type.
        my @fields = ();
        my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $dst) =
          localtime();

        my $unasterisked_mins  = unasterisk($job->{'cr_minutes'});
        my $unasterisked_hours = unasterisk($job->{'cr_hours'});

        if ($job->{'cr_type'} eq 'Hourly') {

            # 10 minutes
            my $time = strftime(
                '%M minutes', $sec,  $unasterisked_mins, $hour,
                $mday,        $mon,  $year,
                $wday,        $yday, $dst
            );
            push(@fields, $time);
        } elsif ($job->{'cr_type'} eq 'Daily') {

            # 12:10 AM
            my $time = strftime(
                '%I:%M %p', $sec,  $unasterisked_mins, $unasterisked_hours,
                $mday,      $mon,  $year,
                $wday,      $yday, $dst
            );
            push(@fields, $time);
        } elsif ($job->{'cr_type'} eq 'Weekly') {

            # Sunday      12:10AM
            my $time = strftime(
                '%A',                   $sec,  $min, $hour,
                $mday,                  $mon,  $year,
                $job->{'cr_days_week'}, $yday, $dst
            );
            push(@fields, $time);
            $time = strftime(
                '%I:%M %p', $sec,  $unasterisked_mins, $unasterisked_hours,
                $mday,      $mon,  $year,
                $wday,      $yday, $dst
            );
            push(@fields, $time);
        } elsif ($job->{'cr_type'} eq 'Monthly') {

            # 1st         12:10AM
            push(@fields, parse_dom($job->{'cr_days_month'}));
            my $time = strftime(
                '%I:%M %p', $sec,  $unasterisked_mins, $unasterisked_hours,
                $mday,      $mon,  $year,
                $wday,      $yday, $dst
            );
            push(@fields, $time);
        } elsif ($job->{'cr_type'} eq 'Yearly') {

            # July 1st    12:10AM
            my $time = $RANGE_MONTHS[$job->{'cr_months'}][1];
            push(@fields, $time . q{ } . parse_dom($job->{'cr_days_month'}));
            $time = strftime(
                '%I:%M %p', $sec,  $unasterisked_mins, $unasterisked_hours,
                $mday,      $mon,  $year,
                $wday,      $yday, $dst
            );
            push(@fields, $time);
        }

        # Generate the message and add the new record to the filtered array.
        $job->{'pretty_type'} = sprintf($TYPES{ $job->{'cr_type'} }, @fields);
        push(@textified, $job);
    }
    return (@textified);
}

# Given a day of the month, add the sub-bit to it (1 to 1st, 2 to 2nd, etc).
sub parse_dom {
    my ($day) = @_;
    for my $i (0 .. $#RANGE_DAYS_MONTH) {
        return $RANGE_DAYS_MONTH[$i][1] if $day eq $RANGE_DAYS_MONTH[$i][0];
    }
    return q{};
}

#############################################################################
# Misc routines
#############################################################################

# Given a hash of values, find and return the key containing the lowest
# value.  If multiple keys have the same value, return the smallest key.
sub show_lowest {
    my (%counts) = @_;

    # Start with $lowcount set to a number less than all possible
    # comparisons.
    ## no critic (ValuesAndExpressions::ProhibitMagicNumbers);
    my ($lowest, $lowcount) = (q{}, -1);
    foreach my $key (sort keys %counts) {
        if ($lowcount > $counts{$key} || $lowest eq q{}) {
            $lowest   = $key;
            $lowcount = $counts{$key};
        }
    }
    return $lowest;
}

# Create a record from a page submission.  Since we have places where
# the simple schedule commands are remade into specifics, this needs to
# combine and do some field override logic.
sub create_record {
    my %job_record = ();

    my @fields = qw(
      cr_requester
      cr_pts_group
      cr_command
      cr_principal
      cr_type
      cr_email
      cr_email_output
      cr_active
      cr_description
      cr_type
    );

    foreach my $f (@fields) {
        if (defined $FORM->fields($f)) {
            $job_record{$f} = $FORM->fields($f);
        }
    }

    # Make sure email address is set.
    ## no critic (ValuesAndExpressions::RequireInterpolationOfMetachars);
    if (!$job_record{'cr_email'}) {
        $job_record{'cr_email'} = uid() . '@stanford.edu';
    }

    # For all of the types, fill in the values to the right fieldname.  For
    # the simple types, we only change them if either the type has changed
    # or, if there is user input, that user input has changed.
    progress('cronjob id is ' . $FORM->field('cronjob'));
    my @jobs = get_user_jobs(uid(), $FORM->field('cronjob'));
    @jobs = textify_times(@jobs);
    progress('number of jobs found for ' . uid() . ': ' . scalar(@jobs));

    my $job;
    if (scalar(@jobs) > 0) {
        $job = $jobs[0];
    } else {

        # No jobs, so set some reasonable defaults.
        $job = {};
        $job->{'cr_type'} = q{};
    }

    ## no critic (ValuesAndExpressions::ProhibitNoisyQuotes);
    ## no critic (ControlStructures::ProhibitCascadingIfElse);
    my $type = $FORM->fields('cr_type');
    if ($type eq 'QuarterHourly') {
        $job_record{'cr_minutes'}    = '*,15';
        $job_record{'cr_hours'}      = '*';
        $job_record{'cr_days_week'}  = '*';
        $job_record{'cr_days_month'} = '*';
        $job_record{'cr_months'}     = '*';
    } elsif ($type eq 'HalfHourly') {
        $job_record{'cr_minutes'}    = '*,30';
        $job_record{'cr_hours'}      = '*';
        $job_record{'cr_days_week'}  = '*';
        $job_record{'cr_days_month'} = '*';
        $job_record{'cr_months'}     = '*';
    } elsif ($type eq 'Hourly' && $job->{'cr_type'} ne 'Hourly') {
        my $minutes = next_hourly();
        $job_record{'cr_minutes'}    = $minutes;
        $job_record{'cr_hours'}      = '*';
        $job_record{'cr_days_week'}  = '*';
        $job_record{'cr_days_month'} = '*';
        $job_record{'cr_months'}     = '*';
    } elsif ($type eq 'Daily' && ($job->{'cr_type'} ne 'Daily')) {
        my ($hours, $minutes) = next_daily();
        $job_record{'cr_minutes'}    = $minutes;
        $job_record{'cr_hours'}      = $hours;
        $job_record{'cr_days_week'}  = '*';
        $job_record{'cr_days_month'} = '*';
        $job_record{'cr_months'}     = '*';
    } elsif (
        $type eq 'Weekly'
        && (   $job->{'cr_type'} ne 'Weekly'
            || $job->{'cr_days_week'} ne $FORM->field('cr_days_week_weekly'))
      )
    {
        my ($hours, $minutes) = next_weekly();
        $job_record{'cr_minutes'}    = $minutes;
        $job_record{'cr_hours'}      = $hours;
        $job_record{'cr_days_week'}  = $FORM->field('cr_days_week_weekly');
        $job_record{'cr_days_month'} = '*';
        $job_record{'cr_months'}     = '*';
    } elsif (
        $type eq 'Monthly'
        && (   $job->{'cr_type'} ne 'Monthly'
            || $job->{'cr_days_month'} ne $FORM->field('cr_days_month_monthly'))
      )
    {
        my ($hours, $minutes) = next_monthly();
        $job_record{'cr_minutes'}    = $minutes;
        $job_record{'cr_hours'}      = $hours;
        $job_record{'cr_days_week'}  = '*';
        $job_record{'cr_days_month'} = $FORM->field('cr_days_month_monthly');
        $job_record{'cr_months'}     = '*';
    } elsif (
        $type eq 'Yearly'
        && (   $job->{'cr_type'} ne 'Yearly'
            || $job->{'cr_days_month'} ne $FORM->field('cr_days_month_yearly')
            || $job->{'cr_months'} ne $FORM->field('cr_months_yearly'))
      )
    {
        my ($hours, $minutes) = next_yearly();
        $job_record{'cr_minutes'}    = $minutes;
        $job_record{'cr_hours'}      = $hours;
        $job_record{'cr_days_week'}  = '*';
        $job_record{'cr_days_month'} = $FORM->field('cr_days_month_yearly');
        $job_record{'cr_months'}     = $FORM->field('cr_months_yearly');
    } elsif ($type eq 'Custom') {
        $job_record{'cr_minutes'}    = join(q{,}, $FORM->field('cr_minutes'));
        $job_record{'cr_hours'}      = join(q{,}, $FORM->field('cr_hours'));
        $job_record{'cr_days_week'}  = join(q{,}, $FORM->field('cr_days_week'));
        $job_record{'cr_days_month'} = join(q{,}, $FORM->field('cr_days_month'));
        $job_record{'cr_months'}     = join(q{,}, $FORM->field('cr_months'));
    }

    return %job_record;
}

sub progress {
    my ($msg) = @_;

    if ($DEBUG) {
        my $current_time = strftime('%Y-%m-%dT%H:%M:%S', localtime(time()));
        warn "progress: [$current_time] $msg";
    }
    return;
}

my $START_TIME;

sub start_timer {
    $START_TIME = [gettimeofday];
    return;
}

sub end_timer {
    return tv_interval($START_TIME, [gettimeofday]);
}

sub show_elapsed_time {
    my ($msg) = @_;

    my $elapsed = end_timer();
    progress("'$msg' took $elapsed seconds");
    return;
}

sub note_text {
    return <<~'EOD';
    Jobs are run by the scheduling service on behalf of either a CGI principal
    (yours or one belonging to a department/group that you administer) or on
    behalf of a special cron principal. For security reasons, jobs cannot be
    run under your regular user name. Jobs will continue to run for one year
    from the date of last modification.
    EOD
}

#############################################################################
# Main routine
#############################################################################

# Get errors and output in the same order.
local $OUTPUT_AUTOFLUSH = 0;

# Clean up the path name.
my $fullpath = $PROGRAM_NAME;
$PROGRAM_NAME =~ s{^.*/}{}xsm;

# Set up tokens
#setpag or die "$PROGRAM_NAME: unable to setpag: $ERRNO\n";
#system ($AKLOG) == 0 or die "$PROGRAM_NAME: unable to obtain tokens\n";

# Load the fieldsets.  Load the edit form here if needed so we can validate
# saves.
fields_main();

if (
    (
        $FORM->submitted eq 'Create New Job'
        || (   $FORM->submitted eq 'Edit Job'
            && $FORM->fields('cronjob'))
        || $FORM->submitted eq 'Save Job'
    )
  )
{
    fields_edit(uid());
}
$FORM->tmpl_param('note-text'            => note_text());
$FORM->tmpl_param('cgi-princ-setup-url'  => $URL{'cgi-princ-setup'});
$FORM->tmpl_param('cron-princ-setup-url' => $URL{'cron-princ-setup'});
$FORM->tmpl_param('uid'                  => user_formatted());

# The edit screen.
if (   $FORM->submitted eq 'Create New Job'
    || ($FORM->submitted eq 'Edit Job' && $FORM->fields('cronjob'))
    || ($FORM->submitted eq 'Save Job' && !$FORM->validate()))
{

    # Set the screen name, and the specific type field to set.
    my $type = $FORM->fields('cr_type');
    $FORM->tmpl_param('edit'            => 1);
    $FORM->tmpl_param($type . '_Select' => 'checked="checked" ');
    $FORM->template("$TEMPL_DIR/scheduler-details.tmpl");
    print $FORM->render(submit => ['Save Job']);

} elsif ($FORM->submitted eq 'Delete Job' && $FORM->fields('cronjob')) {

    # Delete page.

    # Button for Confirm Deletion
    # Table with:
    #       Program
    #       Principal
    #       Type

    # ID for main.
    $FORM->field(name => 'cronjob', type => 'hidden');
    my $id = $FORM->fields('cronjob');
    $FORM->field(name => 'deleteid', value => $id, type => 'hidden');

    # Fields for display only.
    my (@jobs) = get_user_jobs(uid(), $id);
    @jobs = textify_times(@jobs);
    my $job = $jobs[0];
    $FORM->tmpl_param('cr_command'   => $job->{'cr_command'});
    $FORM->tmpl_param('cr_principal' => $job->{'cr_principal'});
    $FORM->tmpl_param('pretty_type'  => $job->{'pretty_type'});
    $FORM->tmpl_param('cr_id'        => $job->{'cr_id'});

    $FORM->tmpl_param('delete' => 1);
    $FORM->template("$TEMPL_DIR/scheduler-delete.tmpl");
    print $FORM->render(submit => ['Confirm Deletion', 'Cancel']);

} else {

    # Main index page.

    # Button for New Cronjob
    # Button for Edit Cronjob
    # Table with:
    #       Program
    #       Principal
    #       Time run (format??)
    #       Radio button for edit

    # Check for deletion confirmation and delete.
    my $status = q{};
    if ($FORM->submitted eq 'Confirm Deletion') {
        $FORM->field(name => 'deleteid', type => 'hidden');
        my $id = $FORM->fields('deleteid');
        if (is_owned($id, uid())) {
            if (remove_job($id)) {
                $status = 'Job has been deleted.';
            } else {
                $status = 'There was an error deleting your job.';
            }
        } else {
            $status = 'You do not own the given job to delete!';
        }
    }

    # If they tried to save, validate and save the updated record.
    if ($FORM->submitted eq 'Save Job' && $FORM->validate()) {

        # If we have a job id, edit that job.  Otherwise, create a new.
        my %job_record = create_record();
        if ($FORM->fields('cronjob')) {
            my $id = $FORM->fields('cronjob');
            if (edit_job($id, %job_record)) {
                $status = 'Your job was edited successfully.';
            } else {
                $status = 'There was an error editing your job.';
            }
        } else {
            if (create_job(%job_record)) {
                $status = 'Your job was created successfully.';
            } else {
                $status = 'There was an error creating your job.';
            }
        }
    }

    # Renew a job by updating the last modified time.
    if ($FORM->submitted eq 'Renew Job' && $FORM->fields('cronjob')) {
        my $id = $FORM->fields('cronjob');
        if (renew_job($id)) {
            $status = 'You have renewed this scheduling job for a year.';
        } else {
            $status = 'Error renewing, please try again or file a HelpSU.';
        }
    }

    # Check for a few error cases and bring up a status message if met.
    if ($FORM->submitted eq 'Delete Job' && !$FORM->fields('cronjob')) {
        $status = 'You must select a job to delete!';
    } elsif ($FORM->submitted eq 'Edit Job' && !$FORM->fields('cronjob')) {
        $status = 'You must select a job to edit!';
    }
    $FORM->tmpl_param('status' => $status);

    # Bring up messages about creating a cron or cgi principal if needed.
    if (!cronprinc_exists(uid())) {
        $FORM->tmpl_param('needcronprinc' => 1);
    }
    if (!cgiprinc_exists(uid())) {
        $FORM->tmpl_param('needcgiprinc' => 1);
    }

    # Grab the jobs to display.
    my (@jobs) = get_user_jobs(uid(), q{});
    @jobs = textify_times(@jobs);
    $FORM->tmpl_param('cronjobs', \@jobs);

    $FORM->tmpl_param('select' => 1);
    $FORM->template("$TEMPL_DIR/scheduler-index.tmpl");

    # If we had jobs, display the table and add buttons for Edit and Delete.
    if (@jobs) {
        $FORM->tmpl_param('tabledata' => 1);
        print $FORM->render(
            submit => [
                'Create New Job', 'Renew Job',
                'Edit Job',       'Delete Job',
            ]
        );
    } else {
        print $FORM->render(submit => ['Create New Job']);
    }
}

exit(0);

__END__

=for stopwords Adam Lewenberg cgi TBD

##############################################################################
# Documentation
##############################################################################

=head1 NAME

name - description

=head1 USAGE

scheduler.cgi

=head1 DESCRIPTION

Manage personal cron jobs.

=head1 REQUIRED ARGUMENTS

None.

=head1 OPTIONS

N/A.

=head1 DIAGNOSTICS

None.

=head1 EXIT STATUS

Exits with 0 on success, 1 on failure.

=head1 CONFIGURATION

The main configuration file in in F</etc/www-scheduler/www-scheduler-config.yaml>.

=head1 DEPENDENCIES

TBD.

=head1 INCOMPATIBILITIES

None known.

=head1 BUGS AND LIMITATIONS

None known.

=head1 AUTHOR

Jon Robertson <jonrober@stanford.edu>
Adam H. Lewenberg <adamhl@stanford.edu>

=head1 LICENSE AND COPYRIGHT

TBD.



=cut
