package Stanford::WWWScheduler;

## no critic (CodeLayout::ProhibitParensWithBuiltins);

use warnings;
use strict;
use autodie;

use Carp;
use DBI;
use Net::Remctl;
use YAML::Tiny;

use Stanford::Orange::Util qw(
  kerberos_ticket_cache_no_afs
  run_command_improved
);

use base qw( Exporter );
our @EXPORT_OK = qw (
  call_remctl
  db_connect
  dump_mysql
  make_kerberos_cache
  read_config
  read_password
);

sub read_config {
    my ($config_yaml_file) = @_;

    my $config_yaml = YAML::Tiny->read($config_yaml_file);
    return %{ $config_yaml->[0] };
}

# Connect to the database.
sub db_connect {
    my ($config_href) = @_;

    my $username      = $config_href->{'db'}->{'username'};
    my $host          = $config_href->{'db'}->{'host'};
    my $dbname        = $config_href->{'db'}->{'dbname'};
    my $port          = $config_href->{'db'}->{'port'};
    my $password_file = $config_href->{'db'}->{'password-file-path'};

    #<<<  perltidy ignore this
    my $dsn = sprintf ('DBI:MariaDB:database=%s;host=%s;port=%d',
                       $dbname, $host, $port);
    #>>>

    my $password = read_password($password_file);

    my $dbh = DBI->connect($dsn, $username, $password);

    return $dbh;
}

sub read_password {
    my ($password_file) = @_;

    open(my $PWF, '<', $password_file);
    my $password = <$PWF>;
    close($PWF);

    chomp($password);
    return $password;
}

sub make_kerberos_cache {
    my ($config_href) = @_;

    my $keytab_file    = $config_href->{'kerberos'}->{'keytab-file-path'};
    my $cache_location = $config_href->{'kerberos'}->{'cache-file-path'};
    my $principal      = $config_href->{'kerberos'}->{'principal'};

    #<<<  perltidy ignore this
    return kerberos_ticket_cache_no_afs($keytab_file, $cache_location, $principal);
    #>>>

}

# Issue a command to a given server using remctl and make sure that it is
# successful.  Returns the status of remctl in a scalar context and a list
# containing the status and any output in a list context.
#
# $service is the Kerberos principal name that the remctl service on $server
# runs under. Often this is "host/$server$ but it can also be a service principal.
sub call_remctl {
    my ($server, $service, $config_href, @command) = @_;

    make_kerberos_cache($config_href);

    my $result = remctl($server, 0, $service, @command);
    my $error  = $result->error;
    if ($error) {
        my $cmd_fmted = join(q{ }, @command);
        croak "remctl command '$cmd_fmted' failed with error $error";
    } else {
        my $stdout      = $result->stdout;
        my $child_error = $result->status;
        my $stderr      = $result->stderr;

        return ($stdout, $stderr, $child_error);
    }
}

sub dump_mysql {
    my ($config_href, $dumpfile) = @_;

    my $username      = $config_href->{'db'}->{'username'};
    my $host          = $config_href->{'db'}->{'host'};
    my $dbname        = $config_href->{'db'}->{'dbname'};
    my $port          = $config_href->{'db'}->{'port'};
    my $password_file = $config_href->{'db'}->{'password-file-path'};

    my $password = read_password($password_file);

    my @cmd = (
        'mysqldump',
        '-h', $host,
        '-P', $port,
        '-u', $username,
        "--password=$password",
        '-r', $dumpfile,
        $dbname,
    );

    my ($stout, $stderr, $rc) = run_command_improved(@cmd);

    if ($stderr) {
        my $cmd_formatted = join(q{ }, @cmd);

        # Redact the password.
        $cmd_formatted =~ s{--password=(\S)+[ ]}{--password=REDACTED[ ]}xsm;

        croak "error running command '$cmd_formatted': $stderr";
    }

    return;
}

1;

__END__
