#!/usr/bin/perl

## no critic (InputOutput::RequireCheckedSyscalls);
## no critic (Modules::RequireNoMatchVarsWithUseEnglish);

use strict;
use warnings;
use autodie;

our $VERSION = 2.0;

use CGI;
use Config::Simple;
use DBI;
use Email::Valid;
use English;
use Readonly;
use Template;
use Try::Tiny;

use Stanford::Schema::WebApps::SharedEmail;
use Stanford::WebApps::SharedEmail::Config qw(%CONFIG);
use Stanford::WebApps::SharedEmail::RequestForm;
use Stanford::WebApps::SharedEmail::Logger qw(get_logger);
use Stanford::WebApps::SharedEmail::Util qw(
  lookup_orgid
  refresh_krb5_cache
  send_email
  split_entries
  white_space_cleanup
);

my $REMOTE_USER = $ENV{'REMOTE_USER'};

# Setup LOGGER
my $LOGGER = get_logger() ;

# Set our Kerberos credentials cache
refresh_krb5_cache($CONFIG{'krb5_keytab'}, $CONFIG{'ccache'});
local $ENV{'KRB5CCNAME'} = $CONFIG{'ccache'};

# First, connect to the database
my $SCHEMA;
eval { $SCHEMA = Stanford::Schema::WebApps::SharedEmail->connect(\%CONFIG); };
my $at_error = $EVAL_ERROR;
if ($at_error) {

    # If we have a problem, we need to tell the user, so make an "incomplete"
    # form object, just to send out an error.
    my %FORM_DATA;
    my $FORM =
      Stanford::WebApps::SharedEmail::RequestForm->new(undef, \%FORM_DATA);

    $LOGGER->fatal("Database connection error: $at_error (request was from "
          . "$REMOTE_USER)");
    $FORM_DATA{'error'} = 'Sorry, we cannot talk to the database right now.  '
      . 'Please try again later!';
    print $FORM->render;
    exit 0;
}

# Get our form
# We need a schema connection, so we have to do this after.
my %FORM_DATA;
my $FORM = Stanford::WebApps::SharedEmail::RequestForm->new($SCHEMA, \%FORM_DATA);

# Now, handle our action!
if ($FORM->submitted) {
    # We need to do a bit of extra verification, above what CGI::FormBuilder does for us.

    # Combine forwarding addresses into a list
    # Also, make sure at least one forwarding address is set
    my @forwards;
    foreach my $i (1 .. $CONFIG{'shared_email_max_forwards'}) {
        my $address = $FORM->field("forwardaddr$i");
        if (defined($address) && (length($address) > 0)) {
            push @forwards, lc($address);
        }
    }
    if (scalar(@forwards) == 0) {
        my $message = 'Please enter at least one forwarding address.';
        $FORM_DATA{'error'} = $message;
    }

    # Now that we're done with our verification, we can check CGI::FormBuilder's work!
    if (($FORM->validate) && (scalar(@forwards) > 0)) {
        # Prepare the request to go to the DB
        my %request;
        $request{'account'}     = lc($FORM->field('acctname'));
        $request{'requestor'}   = $REMOTE_USER;
        $request{'description'} = white_space_cleanup($FORM->field('description'));
        $request{'org'}         = uc($FORM->field('orgid'));
        $request{'sponsor'}     = lc($FORM->field('sponsor'));
        if (defined($FORM->field('addlinfo'))) {
            $request{'additional'} = $FORM->field('addlinfo');
        }

        # Add a note to the log.
        $LOGGER->info(
            $REMOTE_USER
              . ' is requesting account '
              . $request{account}
              . ' for Org '
              . $request{org}
              . ' with owners '
              . join(q{,}, split_entries($FORM->field('owners')))
              . '; forwarding to '
              . join(q{,}, @forwards) . q{.}
        );

        # Add the request to the database
        my $db_record;
        try {
            $SCHEMA->txn_do(
                sub {
                    $db_record = $SCHEMA->resultset('Account')->create(\%request);
                    foreach my $owner (split_entries($FORM->field('owners'))) {
                        my $owner_record =
                          $SCHEMA->resultset('AccountOwner')->create(
                            {
                                account => $FORM->field('acctname'),
                                owner   => $owner,
                            }
                          );
                    }
                    foreach my $forward (@forwards) {
                        my $forward_record =
                          $SCHEMA->resultset('AccountForward')->create(
                            {
                                account => $FORM->field('acctname'),
                                forward => $forward,
                            }
                          );
                    }
                }
            );
        }
        catch {
            $LOGGER->fatal("DB transaction error: $_ (request was from $REMOTE_USER)");
            my $message;
            if (m/Rollback[ ]failed/xms) {
                $message =
                    'Due to a database problem, your request was '
                  . 'partially submitted. Please submit a HelpSU '
                  . 'before trying to resubmit your request.';
            } else {
                $message =
                    'We are having trouble talking to the database '
                  . 'right now. Please wait a while and try your '
                  . 'request again later!';
            }
            $FORM_DATA{'error'} = $message;
            print $FORM->render;
            exit 0;
        };

        # Now that the database stuff is in, send an email
        try {
            send_email(
                'request-submitted.txt.tmpl',
                {
                    request => \%request,
                },
                $request{'requestor'} . '@stanford.edu',
                'Shared email request submitted for ' . $request{'account'}
            );
        }
        catch {
            $LOGGER->error(
                'Error sending request-submitted email for account to '
                  . $request{'requestor'} . ": $_"
            );
            my $message =
                'We had problems sending you an email, but everything else '
              . 'worked OK!';
            $FORM_DATA{'error'} = $message;
        };

        # Hopefully everything went OK!
        my $template = Template->new(
            {
                INCLUDE_PATH => [
                    qw(
                      /usr/share/shared-email/templates
                      /usr/share/stanford-web-template/html
                      )
                ],
            }
        );
        my %template_data;
        $template_data{'lookup_orgid'} = \&lookup_orgid;
        $template_data{'target'}       = $db_record;
        $template_data{'adminpage'}    = "";
        print CGI->new()->header();
        $template->process('confirm.html.tmpl', \%template_data)
          or croak $template->error;

        exit 0;
    }

    # If any of the validations failed, then don't take any action
    else {
        print $FORM->render;
        exit 0;
    }
} else {
    print $FORM->render;
    exit 0;
}
