#!/usr/bin/perl

## no critic (ErrorHandling::RequireCheckingReturnValueOfEval);
## no critic (Modules::RequireNoMatchVarsWithUseEnglish);
## no critic (CodeLayout::ProhibitParensWithBuiltins);

use strict;
use warnings FATAL => 'all';

use Carp;
use CGI;
use DBI;
use Email::Valid;
use English;
use Net::Remctl;
use Readonly;
use Template;
use Try::Tiny;

use Stanford::Schema::WebApps::SharedEmail;
use Stanford::WebApps::SharedEmail::Config qw(%CONFIG);
use Stanford::WebApps::SharedEmail::Logger qw(get_logger);
use Stanford::WebApps::SharedEmail::RequestForm;
use Stanford::WebApps::SharedEmail::Util qw(
  lookup_orgid
  refresh_krb5_cache
  send_email
  white_space_cleanup
);
use Stanford::WorkgroupXML;
use Stanford::WorkgroupXML::Workgroup;

# Setup LOGGER
my $LOGGER = get_logger() ;

# Set our Kerberos credentials cache
refresh_krb5_cache($CONFIG{'krb5_keytab'}, $CONFIG{'ccache'});
local $ENV{'KRB5CCNAME'} = $CONFIG{'ccache'};

# Set the number of results per page
Readonly my $PAGE_MAX => 10;

# Start connecting to outside stuff

# First, prep CGI and Template
my $CGI = CGI->new;
my $TEMPLATE = Template->new({
    INCLUDE_PATH => [qw(
        /usr/share/shared-email/templates
        /usr/share/stanford-web-template/html
    )],
});
my $TEMPLATE_PAGE = 'manage.html.tmpl';
my %TEMPLATE_DATA;
$TEMPLATE_DATA{'adminpage'} = 1;

# Next, connect to the database.
our $SCHEMA;
our $TRANSACTION;
eval {
    $SCHEMA = Stanford::Schema::WebApps::SharedEmail->connect(\%CONFIG);
};
if ($EVAL_ERROR) {
    $LOGGER->fatal("Database connection error: $EVAL_ERROR");
    $TEMPLATE_DATA{error} =   'Sorry, we cannot talk to the database right now.'
                            . 'Please try again later!';
    print $CGI->header;
    print $TEMPLATE->process($TEMPLATE_PAGE, \%TEMPLATE_DATA);
    exit 0;
}

# First we need to perform our action
# (Some actions modify search results, so this must be first)

# Let's check what we're being asked to do
my $action = $CGI->param('action') || q{};
my $target;

# If we are doing something that will make a change, start a transaction first
if (   ($action eq 'approve')
    || ($action eq 'really_reject')
    || ($action eq 'modify')
    || ($action eq 'really_deletedb')
) {
    eval {
        $TRANSACTION = $SCHEMA->txn_scope_guard;
    };
    if ($EVAL_ERROR) {
        $LOGGER->fatal("Database transaction-start error: $EVAL_ERROR");
        $TEMPLATE_DATA{error} =   'Sorry, we can\'t talk to the database right now.'
                                . 'Please try again later!';
        print $CGI->header;
        print $TEMPLATE->process($TEMPLATE_PAGE, \%TEMPLATE_DATA);
        exit 0;
    }
}

# If we have a valid action, then see if we have a valid target.
# The default action is 'show', so we don't need to test for that.
if (   ($action eq 'approve')
    || ($action eq 'modify')
    || ($action eq 'reject')
    || ($action eq 'really_reject')
    || ($action eq 'deletedb')
    || ($action eq 'really_deletedb')
) {
    # Check for our target.
    # For all non-show actions, we'll likely do some sort of update,
    # so get a lock as well.
    $target = $CGI->param('target');
    $target = $SCHEMA->resultset('Account')->search({
        account => $target,
    }, {
        for => 'update',
    });

    # If we can't find the target, display a message and change our action.
    if ($target->count == 0) {
        my $message =   'An action was requested, but with an unrecognized '
                      . 'account name.';
        $TEMPLATE_DATA{error} = $message;
        $action = q{};
        undef $target;
    }
    else {
        # If we did find the target, make sure we aren't changing a target
        # that cannot be changed.
        $target = $target->first;
        if (    defined($target->when_reviewed())
            and ($action ne 'deletedb')
            and ($action ne 'really_deletedb')
        ) {
            my $message =   'The account ' . $target->account
                      . ' has already been processed, and can no longer be changed.';
            $TEMPLATE_DATA{error} = $message;
            $action = q{};
            undef $target;
        }
    }
}

# Now we can take action!
if ($action eq 'approve') {
    # Approving involves a lot of code, so it's been refactored into several
    # subroutines.
    my $can_continue = 1;

    # Create the workgroup
    $can_continue = approve_request_workgroup(
        $CONFIG{'shared_email_wxml_urlbase'},
        $CONFIG{'shared_email_wxml_key'},
        $CONFIG{'shared_email_wxml_cert'},
        $target,
        \%TEMPLATE_DATA
    );

    # TODO: Link workgroup:
    # The remctl is:
    # remctl tools linkage link-existing office365:grouptest-full
    # “WIN Active Directory” office365-grouptest

    # Create the functional account, and update the database
    if ($can_continue) {
        $can_continue = approve_request_remctl_db($CONFIG{'acctsponsor_server'},
                                                  $CONFIG{'acctsponsor_server_principal'},
                                                  $SCHEMA, $target,
                                                  \%TEMPLATE_DATA
        );
    }

    # Finally, send our emails!
    if ($can_continue) {
        $can_continue = approve_request_email($target, \%TEMPLATE_DATA,
                                              $CONFIG{shared_email_backend_list},
        );
    }

    # If everything worked, then commit
    if ($can_continue) {
        $TRANSACTION->commit;
    }
}
elsif ($action eq 'modify') {
    #TODO: Request-modification code
    $TEMPLATE_DATA{error} =   'The Shared Email Robot is sorry; the Shared '
                            . 'Shared Email Robot does not currently know '
                            . 'how to modify requests, because the puny '
                            . 'humans have not told it how to do so.  While '
                            . 'the puny humans get their act together, you '
                            . 'should email or IM akkornel@stanford.edu, so '
                            . 'he can put in the modification. Bleep bloop.';
    $TRANSACTION->commit;
}
elsif ($action eq 'reject') {
    $TEMPLATE_DATA{reject} = $target;
    $TEMPLATE_DATA{message} = 'Please confirm your rejection request';
}
elsif ($action eq 'really_reject') {
    my $can_continue = 1;

    # Make sure a reason is provided
    my $reason = $CGI->param('reason');
    if (   (!defined($reason))
        || (length($reason) == 0)
    ) {
        my $message =   'You tried to reject account ' . $target->account
                      . ', but without providing a reason.';
        $TEMPLATE_DATA{error} = $message;
        $can_continue = 0;
    }

    # If a reason is provided, update the database
    if ($can_continue) {
        $can_continue = reject_request_db($SCHEMA, $target, $reason, \%TEMPLATE_DATA);
    }

    # If the database updated, send the emails!
    if ($can_continue) {
        $can_continue = reject_request_email($target, \%TEMPLATE_DATA);
    }

    # If everything worked, then commit
    if ($can_continue) {
        $TRANSACTION->commit;
    }
}
elsif ($action eq 'deletedb') {
    $TEMPLATE_DATA{deletedb} = $target;
    $TEMPLATE_DATA{message} = 'Please confirm your request';
}
elsif ($action eq 'really_deletedb') {
    $LOGGER->info('Deleting rejected request for account '
                   . $target->account . ', which was for Org '
                   . $target->org . q{.}
    );

    # Deleting from the database is pretty easy to do!
    my $deletion_successful = 1;
    try {
        $target->delete;
        $TRANSACTION->commit;
    } catch {
        $deletion_successful = 0;
        $LOGGER->error("DB error on deleting rejected request: $_");
        my $message =   'We were unable to delete the rejected request from '
                      . 'the request DB.  Please file a HelpSU.  Sorry!';
        $TEMPLATE_DATA{error} = $message;
    };
    if ($deletion_successful) {
        $TEMPLATE_DATA{message} =   'Request for ' . $target->account
                                  . ' has been deleted.';
    }
}
# If we don't have a known action, then just display results
# (which is what we do anyway).

# Now that action has been taken, we can do our search

# Check what we're going to show
my @show;
if (!$CGI->param('show')) {
    $CGI->param('show', 'pending');
}
foreach my $show_option ($CGI->multi_param('show')) {
    if (    ($show_option ne 'pending')
        and ($show_option ne 'approved')
        and ($show_option ne 'rejected')
    ) {
        my $message = "The option '$show_option' is not recognized.";
        $TEMPLATE_DATA{error} = $message;
        next;
    }
    push @show, $show_option;
}

# Pass the list of selected options through to the template.
$TEMPLATE_DATA{show} = \@show;

# Convert each "show" into a search fragment
# Remember: Items in arrayed are ORed; items in hashes are ANDed
my $search = [];
foreach my $show_item (@show) {
    if ($show_item eq 'pending') {
        push @{$search}, { when_reviewed => undef };
    }
    elsif ($show_item eq 'approved') {
        push @{$search}, {
            is_approved => 1,
            when_reviewed => { q{!=}, undef },
        };
    }
    elsif ($show_item eq 'rejected') {
        push @{$search}, {
            is_approved => 0,
            when_reviewed => { q{!=}, undef },
        };
    }
}

# If we were given an account name to search for, add that to the search.
my $search_name = scalar $CGI->param('search');
if (defined($search_name)) {
    $TEMPLATE_DATA{search} = $search_name;

    # Construct the search fragment that finds our name
    my $name_search = {
        account => { -like => q{%} . $search_name . q{%} },
    };

    # Add the fragment into our main search
    $search = [ -and => [
        $name_search,
        [
            @{$search}
        ],
    ]];
}

# Check what page we're on
my $page = $CGI->param('page') || 1;
if ($page !~ m/\A\d+\z/xms) {
    $page  = 1;
    my $message = 'An invalid page number was specified.  Going to page 1.';
    $TEMPLATE_DATA{error} = $message;
}
$TEMPLATE_DATA{page} = $page;

# Do the search!
my $search_rs_unpaged = $SCHEMA->resultset('Account')->search($search, {
    order_by => { -desc => 'when_requested' },
});
my $search_rs_paged = $SCHEMA->resultset('Account')->search($search, {
    order_by => { -desc => 'when_requested' },
    rows     => $PAGE_MAX,
    page     => $page,
});
my @records = $search_rs_paged->all;
$TEMPLATE_DATA{total_count} = $search_rs_unpaged->count;
$TEMPLATE_DATA{page_count} = $search_rs_paged->count;
$TEMPLATE_DATA{records} = \@records;

# lookup_orgid is a function that we're passing to the template
$TEMPLATE_DATA{lookup_orgid} = \&lookup_orgid;

# Show the page!
print $CGI->header;
$TEMPLATE->process($TEMPLATE_PAGE, \%TEMPLATE_DATA)
or croak $TEMPLATE->error;

exit 0;

###########################################################################
###########################################################################

#
# ACTION SUBROUTINES
#

# approve_request_email: Sends all of the emails for an approved request.
# The parameters we take are:
#   $account: A DBIx::Class row, the one that's just been approved
#   $template: A hashref of variables to go out to the template.
#   $list_recipient: The email address that should receive the backend
# approval notification, so that manual tasks can be done.
# We return a 1 if everything is OK, or a 0 if there was a problem.
# We also update $template->{error} if there was an error.
sub approve_request_email {
    my ($account, $template, $list_recipient) = @_;

    ## no critic (ValuesAndExpressions::RequireInterpolationOfMetachars);

    # Try to send each email, and fail if any one of them fails
    try {
        # First, notify the requestor
        send_email('request-approved.txt.tmpl', {
                       request => $account,
                   },
                   $account->requestor . '@stanford.edu',
                   'Shared email request approved for ' . $account->account
        );
        # Next, notify the sponsor
        send_email('request-sponsor.txt.tmpl', {
                       request => $account,
                   },
                   $account->sponsor . '@stanford.edu',
                   (  'Sponsorship required for shared email account '
                    . $account->account)
        );
        # Finally, notify the team doing the backend work
        send_email('windows-infra-notice.txt.tmpl', {
                       request => $account,
                   },
                   $list_recipient,
                   'Approved Shared Email request: ' . $account->account
        );
    } catch {
        $LOGGER->error('Error sending one of the approved emails for '
                        . ' account to ' . $account->requestor . ": $_"
        );
        my $message =   'We had problems sending out at least one of the '
                      . 'emails.  Please email ' . $account->requestor
                      . ', letting them know the request has been approved.  '
                      . 'Please also email ' . $account->sponsor . ', asking '
                      . 'them to sponsor the new account.  Finally, please '
                      . 'email windows-infrastructure@lists.stanford.edu, '
                      . 'and ask them to provision the account.';
        $template->{error} = $message;
        return 0;
    };
    return 1;
}

# approve_request_workgroup: Create the workgroup for this Shared Email account.
# The workgroup name is "office365:ACCOUNT_NAME", and the owners are added as
# workgroup members AND workgroup admins.
# The parameters we take are:
#   $account: The specific DBIx::Class row that has been approved.
#   $template: A hashref of variables that are going to the template.
# We return a 1 if everything worked, or a 0 if there was a problem.
# We also set $template->{message} and $template->{error} appropriately.
#
# The steps that happen in this function (each of which is wrapped in a try/catch):
#
#   1. Create a WorkgroupXML::Workgroup object (no Workgroups created yet).
#
#   2. Create an _empty_ Stanford Workgroup.
#
#   3. Add each owner as a Workgroup admin and member.
sub approve_request_workgroup {
    my ($wgroup_urlbase, $wgroup_key, $wgroup_cert, $account, $template) = @_;

    # Some workgroup-related variables
    my ($wgroup, $wgroup_admins, $wgroup_members);

    # Track if things are OK to continue
    my $ok_to_continue = 1;

    ##
    ## Step 1. Create an empty WorkgroupXML object
    try {
        $wgroup = Stanford::WorkgroupXML::Workgroup->new({
            url_base => $wgroup_urlbase,
            key      => $wgroup_key,
            cert     => $wgroup_cert,
        });
    } catch {
        $LOGGER->fatal("Failure setting up WorkgroupXML object: $_");
        my $message =   'We are having trouble getting your Workgroup ready.'
                      . '  Please try again later, or open a HelpSU.  Sorry!';
        $template->{error} = $message;
        $ok_to_continue = 0;
    };
    return 0 if !$ok_to_continue;

    ##
    ## Step 2. Find out if the workgroup already exists. If it does already
    ## exist, set $skip_workgroup_create to 1 so we skip the create step in
    ## subsequent code.
    my $wresult ;
    my $wgroup_name = 'office365:' . $account->account ;
    my $skip_workgroup_create = 0;
    try {
        $wresult = $wgroup->fetch($wgroup_name);
        $skip_workgroup_create = 1;
    } catch {
        my $error_message = $_->error() ;
        my $regex = qr/No.*workgroup.found.with.name.*${wgroup_name}/ixsm;
        if ($error_message =~ $regex) {
            # The workgroup does not exist so we don't skip the create.
            $skip_workgroup_create = 0;
        }  else {
            # This is a _different_ error so raise an exception.
            $LOGGER->error('Failure querying Workgroup API:'
                        . $account->account . ": $_"
                );
            my $message =   'We are having trouble creating your Workgroup. '
                          . 'Please try again later, or open a HelpSU.  Sorry!';
            $template->{error} = $message;
            $ok_to_continue = 0;
        }
    };
    return 0 if !$ok_to_continue;

    ##
    ## Step 3. Set basic attributes before creating the workgroup.
    $wgroup->description(  'Read/write access to the ' . $account->account
                         . ' Shared Email account'
    );

    if ($wgroup->privgroup() != 1) {
        $wgroup->privgroup(1);
    }

    if ($wgroup->reusable() != 0) {
        $wgroup->reusable(0);
    }

    if ($wgroup->visibility() != 1) {
        $wgroup->visibility(1);
    }

    ##
    ## Step 4. If the workgroup does not already exist, create it. Later we get
    ## our admin and member lists.
    try {
        if (! $skip_workgroup_create) {
            $wgroup->create($wgroup_name);
        }
        $wgroup_admins  = $wgroup->administrators();
        $wgroup_members = $wgroup->members();
    } catch {
        $LOGGER->error('Failure creating Workgroup office365:'
                        . $account->account . ": $_"
        );
        my $message =   'We are having trouble creating your Workgroup. '
                      . 'Please try again later, or open a HelpSU.  Sorry!';
        $template->{error} = $message;
        $ok_to_continue = 0;
    };
    return 0 if !$ok_to_continue;

    ##
    ## Step 5. Add each owner as an admin & member
    foreach my $owner ($account->owners) {
        my $sunetid = $owner->owner;
        try {
            $wgroup_admins->add_member($sunetid);
            $wgroup_members->add_member($sunetid);
        } catch {
            $LOGGER->error("Failure adding $sunetid to Workgroup office365:"
                            . $account->account . ": $_"
            );
            my $message =   'We are having trouble creating your Workgroup '
                          . 'right now.  Account creation was incomplete.  '
                          . 'Please open a HelpSU, so we can finish setting '
                          . 'up your account.  Thanks!';
            $template->{error} = $message;
            $ok_to_continue = 0;
        };
        return 0 if !$ok_to_continue;
    }

    ##
    ## Step 6. Add the stem owner as a workgroup admin
    try {
        $wgroup_admins->add_workgroup('workgroup:office365-owners');
    } catch {
                $LOGGER->error('Failure adding workgroup:office365-'
                                . 'to Workgroup office365:'
                                . $account->account . ": $_"
                );
                my $message =   'We are having trouble creating your Workgroup '
                              . 'right now.  Account creation was incomplete.  '
                              . 'Please open a HelpSU, so we can finish setting'
                              . ' up your account.  Thanks!';
                $template->{error} = $message;
                $ok_to_continue = 0;
    };
    return 0 if !$ok_to_continue;

    # Workgroup creation & population is done!
    return 1;
}

# approve_request_workgroup_restful: Create the workgroup for this Shared
# Email account using the Workgroup RESTful Web Services API (replaces
# previous function approve_request_workgroup).
#
# The workgroup name is "office365:ACCOUNT_NAME", and the owners are added as
# workgroup members AND workgroup admins.
# The parameters we take are:
#   $account: The specific DBIx::Class row that has been approved.
#   $template: A hashref of variables that are going to the template.
# We return a 1 if everything worked, or a 0 if there was a problem.
# We also set $template->{message} and $template->{error} appropriately.
sub approve_request_workgroup_restful {
    my ($wgroup_urlbase, $wgroup_key, $wgroup_cert, $account, $template) = @_;

    # Some workgrou-related variables
    my ($wgroup, $wgroup_admins, $wgroup_members);

    # Track if things are OK to continue
    my $ok_to_continue = 1;

    # Create an empty WorkgroupXML object
    try {
        $wgroup = Stanford::WorkgroupXML::Workgroup->new({
            url_base => $wgroup_urlbase,
            key      => $wgroup_key,
            cert     => $wgroup_cert,
        });
    } catch {
        $LOGGER->fatal("Failure setting up WorkgroupXML object: $_");
        my $message =   'We are having trouble getting your Workgroup ready.'
                      . '  Please try again later, or open a HelpSU.  Sorry!';
        $template->{error} = $message;
        $ok_to_continue = 0;
    };
    return 0 if !$ok_to_continue;

    # Set basic attributes before creating the workgroup
    $wgroup->description(  'Read/write access to the ' . $account->account
                         . ' Shared Email account'
    );
    $wgroup->privgroup(1);
    $wgroup->reusable(0);
    $wgroup->visibility(1);

    # Actually create the workgroup, then get our admin and member lists
    try {
        $wgroup->create('office365:' . $account->account);
        $wgroup_admins = $wgroup->administrators();
        $wgroup_members = $wgroup->members();
    } catch {
        $LOGGER->error('Failure creating Workgroup office365:'
                        . $account->account . ": $_"
        );
        my $message =   'We are having trouble creating your Workgroup. '
                      . 'Please try again later, or open a HelpSU.  Sorry!';
        $template->{error} = $message;
        $ok_to_continue = 0;
    };
    return 0 if !$ok_to_continue;

    # Add each owner as an admin & member
    foreach my $owner ($account->owners) {
        my $sunetid = $owner->owner;
        try {
            $wgroup_admins->add_member($sunetid);
            $wgroup_members->add_member($sunetid);
        } catch {
            $LOGGER->error("Failure adding $sunetid to Workgroup office365:"
                            . $account->account . ": $_"
            );
            my $message =   'We are having trouble creating your Workgroup '
                          . 'right now.  Account creation was incomplete.  '
                          . 'Please open a HelpSU, so we can finish setting '
                          . 'up your account.  Thanks!';
            $template->{error} = $message;
            $ok_to_continue = 0;
        };
        return 0 if !$ok_to_continue;
    }

    # Add the stem owner as a workgroup admin
    try {
        $wgroup_admins->add_workgroup('workgroup:office365-owners');
    } catch {
                $LOGGER->error('Failure adding workgroup:office365-'
                                . 'to Workgroup office365:'
                                . $account->account . ": $_"
                );
                my $message =   'We are having trouble creating your Workgroup '
                              . 'right now.  Account creation was incomplete.  '
                              . 'Please open a HelpSU, so we can finish setting'
                              . ' up your account.  Thanks!';
                $template->{error} = $message;
                $ok_to_continue = 0;
    };
    return 0 if !$ok_to_continue;

    # Workgroup creation & population is done!
    return 1;
}

# approve_request_remctl_db: Run the remctl to create a functional account,
# and update the database.
# The parameters we take are:
#   $remctl_server: The server for the account-create-functional remctl.
#   $remctl_server_principal: The Kerberos principal name used by $remctl_server (usually
#                             "host/$remctl_server" but it might be something different).
#   $schema: A connected DBIx::Class schema.
#   $target: The specific DBIx::Class row that has been approved.
#   $template: A hashref of variables that are going to the template.
# We return a 1 if everything worked, or a 0 if there was a problem.
# We also set $template->{message} and $template->{error} appropriately.
sub approve_request_remctl_db {
    my ($remctl_server, $remctl_principal, $schema, $account, $template) = @_;

    # First, update the database to record request completion
    # We also run the remctl in the transaction, because they're related.
    try {
        # Get a list of UIDs from the owner SUNetID
        my @uids;
        foreach my $sunetid ($account->owners) {
            push @uids, $sunetid->owner;
        }

        # Get our list of forwarding addresses
        my @forwards;
        foreach my $forward ($account->forwards) {
            push @forwards, $forward->forward;
        }

        # Run the remctl
        my @remctl_params;
        push @remctl_params, q{-a} . $account->account;
        push @remctl_params, q{-lbase};
        push @remctl_params, q{-o} . $account->org;
        push @remctl_params, q{-x} . white_space_cleanup($account->description);
        push @remctl_params, q{-m} . join(q{,}, @uids);
        push @remctl_params, q{-f} . q{"} . join(q{,}, @forwards) . q{"};
        my $acct_remctl = remctl($remctl_server, 0, $remctl_principal,
                                 'account-create-functional', 'submit',
                                 @remctl_params
        );

        my $msg = "About to run command: 'remctl $remctl_server "
                . "-s $remctl_principal account-create-functional submit "
            . join(q{ }, @remctl_params) . q{'} ;
        $LOGGER->info($msg) ;

        if ($acct_remctl->error) {
            $LOGGER->error('Error on remctl account-create-functional '
                            . 'submit ' . join(q{ }, @remctl_params) . q{: }
                            . $acct_remctl->error);
            croak('(actually it was a remctl error)');
        }

        # After the remctl, now update the DB
        #
        # (Note from adamhl [2021-03-04]: THE original line was
        #      $account->when_reviewed(\'NOW()');
        # I don't understand why a scalar ref is being passed to
        # when_reviewed, but rather than risk breaking it I have
        # changed the constant scalar ref to a ref to a string variable
        # so that editors will not be confused by the mis-matched
        # single quotes.
        my $now_scalar = 'NOW()';
        $account->when_reviewed(\$now_scalar);

        $account->is_approved(1);
        $account->update;

        # Log the request approval
        $LOGGER->info('Account ' . $account->account . ' approved by '
                         . $ENV{REMOTE_USER} . q{.});
        $template->{message} =   'Account ' . $account->account
                               . ' has been approved!';
    } catch {
        $LOGGER->fatal("DB transaction error marking request approved: $_");
        my $message;
        if (m/Rollback[ ]failed/xms) {
            $message =   'Due to a database problem, your request was '
                       . 'partially submitted.  Please submit a HelpSU '
                       . 'before trying to resubmit your request.';
        }
        else {
            $message =   'We are having trouble talking to the database '
                       . 'right now.  Please wait a while and try you '
                       . 'request again later!';
        }
        $template->{error} = $message;
        return 0;
    };

    # Everything seemed to work!
    return 1;
}

# reject_request_email: Send an email saying that the request was rejected
# There are two parameters:
#   $target: A DBIx::Class row that is the account being rejected
#   $template_data: A hashref of Template variables.
# Returns 1 if the emails were sent OK; 0 otherwise.
sub reject_request_email {
    my ($target1, $template_data) = @_;

    ## no critic (ValuesAndExpressions::RequireInterpolationOfMetachars);

    # Let the requestor know that the request was rejected
    my $worked_ok = 1;
    try {
        send_email('request-rejected.txt.tmpl', {
                       request => $target1,
                   },
                   $target1->requestor . '@stanford.edu',
                   ('Shared email request for ' . $target1->account
                    . ' rejected')
        );
    } catch {
        $LOGGER->error('Error sending the reject email for account '
                        . ' to ' . $target1->requestor . ": $_"
        );
        my $message =   'We had a problem sending out the rejection '
                      . 'email.  Please contaact ' . $target1->requestor
                      . 'to let them know that their request was '
                      . 'rejected, and let them know why.';
        $template_data->{error} = $message;
        $worked_ok = 0;
    };

    # If the email sent, then we're good!
    return $worked_ok;
}


# reject_request_db: Update the database with notes on the rejected request
# We take three parameters:
#   $schema: A DBIx::Class schema, which we'll use for a transaction
#   $target: A DBIx::Class row, which is the account to reject
#   $reason: The reason for rejection
#   $template_data: A hashref of Template variables
sub reject_request_db {
    my ($schema, $target1, $reason, $template_data) = @_;

    # Update the account in the database
    my $worked_ok = 1;
    try {
        my $now_scalar = 'NOW()';
        $target1->when_reviewed(\$now_scalar);
        $target1->is_approved(0);
        $target1->reject_reason($reason);
        $target1->update;
        $template_data->{message} =   'Account ' . $target1->account
                                    . ' has been rejected.';

        # Before logging, remove newlines from the reason
        $reason =~ s/\n/----/xgms;
        $LOGGER->info('Account ' . $target1->account . ' rejected by '
                         . $ENV{REMOTE_USER} . ", reason '$reason'.");
    } catch {
        $LOGGER->fatal("DB transaction error: $_");
        my $message;
        if (m/Rollback[ ]failed/xms) {
            $message =   'Due to a database problem, your request was '
                       . 'partially submitted.  Please submit a HelpSU '
                       . 'before trying to resubmit your request.';
        }
        else {
            $message =   'We are having trouble talking to the database '
                       . 'right now.  Please wait a while and try you '
                       . 'request again later!';
        }
        $template_data->{error} = $message;
        $worked_ok = 0;
    };

    # If everything wen't OK, then great!
    return $worked_ok;
}

