package Stanford::WebApps::SharedEmail::Config;

## no critic (CodeLayout::ProhibitParensWithBuiltins);
## no critic (Modules::RequireNoMatchVarsWithUseEnglish);

use strict;
use warnings;
use autodie;

require Exporter;
use base qw(Exporter);

use Carp;
use YAML::Tiny;
use Readonly;
use English;

our %CONFIG = ();
our @EXPORT_OK = qw(%CONFIG);

Readonly my @CONFIG_PATHS => (qw(
    /etc/shared-email-web/config.yaml
    /etc/shared-email-web/migrate.yaml
));

# Read config files
foreach my $config_path (@CONFIG_PATHS) {
    if (-r $config_path) {
        my $yaml = YAML::Tiny->read($config_path);
        my $config = $yaml->[0];
        %CONFIG = (%CONFIG, %{ $config });
    } else {
        croak("Config file not readable at '$config_path': $ERRNO");
    }
}

# Make sure we have each of our parameters
Readonly my @CONFIG_PARAMS => (qw(
    ccache
    acctsponsor_server
    acctsponsor_server_principal
    workgroup_link_server
    workgroup_link_server_principal
    workgroup_link_environment
    krb5_keytab
    ldap_server
    platform
    shared_email_db_host
    shared_email_db_name
    shared_email_db_user
    shared_email_db_passwd_file
    shared_email_backend_list
    shared_email_max_forwards
    shared_email_wxml_urlbase
    shared_email_wxml_key
    shared_email_wxml_cert
    crdb_db_host
    crdb_db_name
    crdb_db_user
    crdb_db_passwd_file
    send_email_smtp_host
    send_email_smtp_port
    send_email_from
    send_email_use_auth_enabled
    send_email_use_auth_username
    send_email_use_auth_password_file
    workgroup_linkage_enabled
));

Readonly my @REQUIRED_CONFIG_PARAMS => (qw(
    ccache
    acctsponsor_server
    acctsponsor_server_principal
    workgroup_link_server
    workgroup_link_server_principal
    workgroup_link_environment
    krb5_keytab
    ldap_server
    platform
    shared_email_db_host
    shared_email_db_name
    shared_email_db_user
    shared_email_db_passwd_file
    shared_email_backend_list
    shared_email_max_forwards
    shared_email_wxml_urlbase
    shared_email_wxml_key
    shared_email_wxml_cert
    send_email_smtp_host
    send_email_use_auth_enabled
    workgroup_linkage_enabled
));

foreach my $config_param (@REQUIRED_CONFIG_PARAMS) {
    if (!defined($CONFIG{$config_param})) {
        croak("Required configuration parameter $config_param not set\n");
    }
}


1;

__END__

=pod

=begin stopwords

ccache
config
dev
kerberos
Kornel
MaIS
namespace
uat
webapp
whitespace

=end stopwords

=head1 NAME

Stanford::WebApps::SharedEmail::Config - Shared email webapp configuration

=head1 SYNOPSIS

use Stanford::WebApps::SharedEmail::Config qw(%CONFIG);

$ENV{KRB5CCNAME} = $CONFIG{ccache};
# And so on...

=head1 DESCRIPTION

This module reads in common configuration parameters from multiple files, making
them available in the C<%CONFIG> hash, which can be imported directly into
the client's namespace.

Configuration files have a simple form: Parameter name, whitespace, value.

Empty lines are ignored.  Lines starting with C<#> are treated as comments, and are ignored.

=head1 FILES

=over 4

=item /etc/tools-cgi/common

=item /etc/tools-cgi/deptemail

=item /etc/tools-cgi/platform

=back

The list of configuration files is defined in the C<@CONFIG_PATHS> list.
Each configuration file is read into the same namespace.

=head1 OPTIONS

The following configuration options are read from the config files:

=over 4

=item ccache

This is the path to a Kerberos 5 credentials cache, which is used for
Kerberos-authenticated connections to remote hosts.

=item acctsponsor_server

This is the name of the accounts system, used for making remctl calls related to accounts.

=item acctsponsor_server_principal

The name of kerberos principal to use when making a remctl call to the
acctsponsor_server.  If calling to the server directly set this to
C<host/"acctsponsor_server">, e.g.,
C<host/asregappuat7.stanford.edu>.

However, in those situations where the endpoint is not the server itself
(e.g., you making a the remctl through a proxy of some sort), set it to
the principal used by the server actually running the remctl service.  For
example, if B<acctsponsor_server> is proxy such as
B<asregappuat7.stanford.edu-lb> while the server running the remctl
service is using B<host/asregappuat7.stanford.edu> for its remctl service
principal, set B<acctsponsor_server_principal> to
B<host/asregappuat7.stanford.edu>.

=item workgroup_link_server

The name of the server that will respond to a remctl call to link a Stanford
Workgroup to a Windows Active Directory group.

=item workgroup_link_server_principal

Kerberos name to use when making remctl call to B<workgroup_link_server>.

=item platform

This is the name of the platform the server is on: Either "prod" or "dev".

=item sharedemail_db_host

This is the name of the MySQL database server.

=item sharedemail_db_name

This is the name of the MySQL database.

=item sharedemail_db_user

This is the username to use for connecting to the MySQL database server.

=item sharedemail_db_passwd-file

This is the path to a file which contains the password for
C<sharedemail_db_user>.  The first line of the file is read and used as
the password.

=item sharedemail_backend_list

The email address, typically a Windows Team address, that should get a
notification when a request is approved.

=item sharedemail_max_forwards

This is the maximum number of forwarding addresses allowed.  The limit is
essentially set by MaIS.

=item sharedemail_orgid_db

This is the name of the Org ID database to read from.  This is either "prod" or "uat".

=back

=head1 AUTHOR

A. Karl Kornel - akkornel@stanford.edu
Updated by Adam H. Lewenberg - adamhl@stanford.edu
