import unittest
import re
import tempfile
from subprocess import PIPE, Popen, STDOUT

# Add ../lib to the python path so we can find the incommon.py module
import sys
import os
sys.path.append(os.path.dirname(os.path.abspath(__file__)) + os.sep + ".." + os.sep + "lib")
sys.path.append('/usr/share/python-stanford-incommoncert/lib')

import yaml

from incommon import Certificate, InCommonWS
from csr import CSR

class TestIncommonClass(unittest.TestCase):
  # Read yml file to get username and password
  connect_file = os.path.dirname(os.path.abspath(__file__)) + os.sep + "connect.yml"
  f = open(connect_file)
  connect_yml = yaml.load(f)
  f.close()

  mgr_bin_path = \
    os.path.dirname(os.path.abspath(__file__)) + os.sep + ".." + os.sep + 'incommon-mgr'

  username = connect_yml['username']
  password = connect_yml['password']


class TestCommandLine(TestIncommonClass):

  # This runs at the start of each test below.
  def setUp(self):
    self.certificate = Certificate()
    self.certificate.username = self.username
    self.certificate.password = self.password
    self.certificate.set_test()

  def test_show_help(self):
    args = [
             self.mgr_bin_path,
             'help',
           ]

    p = Popen(args, stdout=PIPE, stderr=STDOUT)
    output = p.communicate()[0]

    re1 = re.compile("Display help on an action")
    self.assertTrue(re1.search(output),
                      "missing expected text on help screen")


  # Check the help text for 'help request' and 'help status'.
  def test_show_help_request(self):
    args = [
             self.mgr_bin_path,
             'help',
             'request',
           ]

    p = Popen(args, stdout=PIPE, stderr=STDOUT)
    output = p.communicate()[0]

    re1 = re.compile("Usage:.*request.*csr_file.*username.*password")
    self.assertTrue(re1.search(output),
                      "missing expected text on request help screen")

    args = [
             self.mgr_bin_path,
             'help',
             'status',
           ]

    p = Popen(args, stdout=PIPE, stderr=STDOUT)
    output = p.communicate()[0]

    re1 = re.compile("Usage:.*status.*order_number.*username.*password")
    self.assertTrue(re1.search(output),
                      "missing expected text on status help screen")


  # Request a certificate and check its status
  def test_request_cert(self):
    csr = CSR()
    csr.commonName = 'testabc123.stanford.edu'
    csr.make()

    # Put the CSR string into a temporary file
    (temp_csr_fd, temp_csr_filename) = tempfile.mkstemp(suffix=".tmpcsr")
    temp_csr_fh = open(temp_csr_filename, "w")
    temp_csr_fh.write(csr.csr_string) ;
    temp_csr_fh.close()

    # Request using the manager executable
    args = [
             self.mgr_bin_path,
             'request',
             '--test',
             '--username',
             self.certificate.username,
             '--password',
             self.certificate.password,
             '--subject-alternative-names',
             'testdef123.stanford.edu,abcdef123.stanford.edu',
             temp_csr_filename,
           ]

    p = Popen(args, stdout=PIPE, stderr=STDOUT)
    output = p.communicate()[0]
    order_number = output.strip()

    re1 = re.compile("^\d+$")
    self.assertTrue(re1.search(order_number),
                      "order number not returned")



if __name__ == '__main__':
  unittest.main()
