import os
import time
import unittest
import pprint
import re

from selenium                        import webdriver
from selenium.common.exceptions      import StaleElementReferenceException

from selenium.webdriver.common.by    import By
from selenium.webdriver.common.keys  import Keys
from selenium.webdriver.support.ui   import WebDriverWait
from selenium.webdriver.support      import expected_conditions as EC
from selenium.webdriver.support.wait import WebDriverWait

from selenium.webdriver.common.action_chains import ActionChains
from selenium.webdriver.common.desired_capabilities import DesiredCapabilities

from selenium.webdriver import Firefox
from selenium.webdriver.firefox.options import Options


# I cannot get this function to work!
class wait_for_text_to_be_nonempty(object):
  def __init__(self, locator):
    self.locator = locator

  def __call__(self, driver):
    try:
      element = EC._find_element(driver, self.locator)
      element_text = element.text
      element1 = driver.find_element_by_id('server_name')
      print element
      print element1.text
      print element_text
      if (element_text):
        return True
    except StaleElementReferenceException:
      return False


class CertreqTesting():
  # IMPORTANT:
  #
  # 1. Install chromedriver in /usr/sbin
  # 2. Install Debian package libgconf-2-4

  username = 'adamhl'
  password = 'spdb-quality-assurance'
  hostname = 'certreq-test.stanford.edu'
  debug    = True

  # Set up the webdriver object. To access it, use "self.driver".
  def __init__(self):
    #self.driver = webdriver.Chrome()
    #self.driver = webdriver.Firefox()
    #self.driver = webdriver.PhantomJS()
    ## FIREFOX
    ##
    ## CHROME
    #chrome_options = Options()
    #chrome_options.add_argument("--headless")
    #chrome_options.add_argument("--window-size=1920x1080")
    #chrome_driver = "/usr/bin/chromedriver"
    #os.environ["webdriver.chrome.driver"] = chrome_driver
    #driver = webdriver.Chrome(chrome_options=chrome_options, executable_path=chrome_driver)
    self.setup_firefox()

  def setup_firefox(self):
    options = Options()
    options.add_argument('-headless')
    self.driver = Firefox(executable_path='/usr/bin/geckodriver', options=options)
    self.wait = WebDriverWait(self.driver, 10)

  def progress(self, message):
    if (self.debug):
      print message

  # We use "basic auth" to login to this service.
  def login(self):
    url = "https://{}:{}@{}/cert-request".format(self.username, self.password, self.hostname)
    self.progress("attempting to login to application at url " + url)
    result = self.driver.get(url)
    return result

  # Fill in the form values (but do not press submit).
  def fill_in_form(self, fields2values):
    self.progress("filling in form")
    for field in fields2values:
      value = fields2values[field]
      element = self.driver.find_element_by_id(field)
      element.send_keys(value)
      self.driver.execute_script("arguments[0].value = arguments[1];",
                                 element,
                                 value)
      # Force a change event on the field just changed.
      self.driver.execute_script('$("#' + field + '").change();')

  def get_text_in_warning(self):
    element = self.wait.until(EC.presence_of_element_located((By.ID, 'submission_warning')))
    return element.get_attribute('innerHTML')

  def submit(self):
    element = self.driver.find_element_by_id('_submit')
    element.click()

  def page_contents(self):
    return self.driver.find_element_by_tag_name('body').text

  def set_focus(self, element_id):
    # Find element
    element = self.driver.find_element_by_id(element_id)

    # Set focus
    action = ActionChains(self.driver)
    action.move_to_element(element).perform()

  def set_focus_on_body(self):
    # Find element
    element = self.driver.find_element_by_tag_name('body')

    # Set focus
    action = ActionChains(self.driver)
    action.move_to_element(element).perform()

  # This method cahnges the focus to the servername input box to trigger the
  # Javscript to parse the CSR and populate the servername input.
  def populate_server_name(self):
    self.set_focus_on_body()
    element = self.driver.find_element_by_id('server_name')
    element.click()
    value = element.get_attribute('value')

  def get_id_of_active_element(self):
    return self.driver.switch_to.active_element.get_attribute('id')

  def get_value_element_by_id(self, id):
    return self.driver.find_element_by_id(id).get_attribute('value')

class Util():
  @staticmethod
  def slurp(filename):
    return open(filename).read()

  @staticmethod
  def get_csr_from_file(filename):
    return Util.slurp('./csrs/' + filename)


#####################################################

#  * Page loads with correct title
#  * Correct error(s) if page submitted with no input.
#  * Wildcard in subject throws correct error.
#  * NetDB checking
#    + (test_030) regular cert, node NOT in NetDB
#    + (test_040) regular cert, node in NetDB, user not admin
#    + (test_050) regular cert, node in NetDB, user IS admin
#    + regular cert, node not in NetDB but domain is in NetDB (are there any such?)
#    + (test_060) wildcard cert, domain not in NetDB
#    + (test_070) wildcard cert, domain in NetDB, user not admin of domain
#    + (test_080) wildcard cert, domain in NetDB, user IS admin of domain
#    + regular cert with a two SANs (one SAN in NetDB, one not in NetDB)

class TestCertReqUnitTest(unittest.TestCase):

    certreq = CertreqTesting()
    debug   = False

    def progress(self, message):
      if (self.debug):
        print message

    def setUp(self):
      self.progress("running setUp")
      self.certreq.debug = self.debug
      self.certreq.login()
      self.certreq.driver.maximize_window()
      self.fields2values = {}

      # We set contact_address to something that will not bother the form.
      self.progress("defining fields2values")
      self.fields2values['contact_address'] = 'adamhl@lists.stanford.edu'

    def submit_with_csr(self, csr_file_name):
      self.fields2values['csr'] = Util.get_csr_from_file(csr_file_name)

      # Put focus on CSR field
      self.certreq.set_focus('csr')

      # Fill in CSR field
      self.certreq.fill_in_form(self.fields2values)

      # Wait until the server_name field is populated.
      found = False
      while (not found):
        found = self.certreq.get_value_element_by_id('server_name')
        time.sleep(0.5)

      # Submit
      self.certreq.submit()

    def is_successful_submission(self, label):
      success_element = self.certreq.wait.until(EC.presence_of_element_located((By.ID, 'submission_success')))
      success_contents = success_element.get_attribute('innerHTML')

      match_rx = r'Your request for .* has been submitted to\s*the\s*SSL\s*certificate\s*administrators.'
      return self.assertRegexpMatches(success_contents, match_rx, label)


    def test_005_login(self):
        title = self.certreq.driver.title
        self.assertRegexpMatches(title, 'SSL Certificate Request Form',
                         'title is correct')

    # Submit page with no input at all
    def test_010_submit_empty(self):
        self.certreq.fill_in_form(self.fields2values)
        self.certreq.submit()

        warning_text = self.certreq.get_text_in_warning()
        self.assertRegexpMatches(warning_text,
                                 'you must indicate the name of the server',
                                 'missing server error correct')
        self.assertRegexpMatches(warning_text,
                                 'must include a valid Certificate Signing Request',
                                 'missing CSR error correct')

    # Subject has a wildcard (a non-no)
    def test_020_wildcard_subject(self):
      self.submit_with_csr('wildcard_in_subject.csr')

      warning_text = self.certreq.get_text_in_warning()
      self.assertRegexpMatches(warning_text,
                               'Wildcard certificate names cannot be in the Subject',
                               'wildcard in subject error correct')

    # Subject not in NetDB
    def test_030_not_in_netdb(self):
      self.submit_with_csr('node_not_in_netdb.csr')

      warning_text = self.certreq.get_text_in_warning()
      self.assertRegexpMatches(warning_text,
                               'does not exist in NetDB as a NODE or DOMAIN object',
                               'not in NetDB error correct')

    # user does not control the node in NetDB
    def test_040_not_controlled_by_user(self):
      self.submit_with_csr('node_not_controlled_by_user.csr')

      warning_text = self.certreq.get_text_in_warning()
      self.assertRegexpMatches(warning_text,
                               'You do not have User or Admin access to the NODE',
                               'no control in NetDB error correct')

    # user DOES control the node in NetDB
    def test_050_controlled_by_user(self):
      self.submit_with_csr('node_controlled_by_user.csr')
      label = 'regular node submitted controlling user'
      self.is_successful_submission(label)

    # ECC cert, user controls the node, key is large enough
    def test_055_ecc_node_key_large_enough(self):
      self.submit_with_csr('spdb-dev-ecc-256bit.csr')
      label = 'ecc node with adequate key'
      self.is_successful_submission(label)

    # ECC cert, user controls the node, key is too small (160 bits is the
    # minimum).
    def test_057_ecc_node_key_too_small_enough(self):
      self.submit_with_csr('spdb-dev-ecc-128bit.csr')
      warning_text = self.certreq.get_text_in_warning()
      match_text = 'CSRs using ECC public-keys must have a ' \
                   'minimum key size of 160 bits. Your CSR ' \
                   'has a 128-bit key which is too small.'
      self.assertRegexpMatches(warning_text, match_text,
                              'ECC key too small error correct')


    # wildcard cert, node in NetDB but domain not in NetDB
    def test_060_wildcard_domain_not_in_netd(self):
      self.fields2values['alternatives'] = '*.spdb-dev.stanford.edu'
      self.submit_with_csr('spdb-dev.csr')

      warning_text = self.certreq.get_text_in_warning()
      regex = re.compile("^.*spdb-dev.stanford.edu.*not exist.*DOMAIN object", re.DOTALL)
      label = 'wildcard node not in NetDB as a DOMAIN object'
      self.assertRegexpMatches(warning_text, regex, label)


    # wildcard cert, domain in NetDB but user does not control domain
    def test_070_wildcard_domain_not_controlled_by_user(self):
      self.fields2values['alternatives'] = '*.itlab.stanford.edu'
      self.submit_with_csr('spdb-dev.csr')
      warning_text = self.certreq.get_text_in_warning()
      regex = re.compile("^.*do.not.have..Use.as.name..access.to.the.DOMAIN.object", re.DOTALL)
      label = 'wildcard node exists but submitted by non-controlling user'
      self.assertRegexpMatches(warning_text, regex, label)


    # wildcard cert, domain in NetDB, user IS admin of domain
    def test_080_wildcard_domain_controlled_by_user(self):
      self.fields2values['alternatives'] = '*.lemon.stanford.edu'
      self.submit_with_csr('spdb-dev.csr')

      label = 'wildcard node exists and was submitted by a controlling user'
      self.is_successful_submission(label)


def suite():
  tests = [
           'test_050_controlled_by_user',
          ]
  return unittest.TestSuite(map(TestCertReqUnitTest, tests))



if __name__ == '__main__':
    unittest.main()
    unittest.TextTestRunner(verbosity=1).run(suite())
