import unittest
import re

# Add ../lib to the python path so we can find the incommon.py module
import sys
import os
sys.path.append(os.path.dirname(os.path.abspath(__file__)) + os.sep + ".." + os.sep + "lib")

import yaml


from incommon import Certificate, InCommonWS
from csr import CSR

class TestIncommonClass(unittest.TestCase):
  # Read yml file to get username and password
  connect_file = os.path.dirname(os.path.abspath(__file__)) + os.sep + "connect.yml"
  f = open(connect_file)
  connect_yml = yaml.load(f)
  f.close()

  username  = connect_yml['username']
  password  = connect_yml['password']
  secretkey = connect_yml['secretkey']



class TestSequenceFunctions(TestIncommonClass):

  # This runs at the start of each test below.
  def setUp(self):
    self.certificate = Certificate()
    self.certificate.username  = self.username
    self.certificate.password  = self.password
    self.certificate.secretKey = self.secretkey
    self.certificate.set_test()

    self.hostname = 'e816149ai7'

    # Make some subject alternative names in case we need them
    self.subjectAltName1 = 'e816149ai7a'
    self.subjectAltName2 = 'e816149ai7b'

    csr = CSR()
    csr.commonName = self.hostname + '.stanford.edu'
    csr.make()
    self.certificate.csr = csr.csr_string

    self.certificate.serverType = self.certificate.SERVER_TYPES['Apache/ModSSL']
    self.certificate.setCert = 'InCommon SSL'
    self.certificate.term = 1

    # '*.itservices.stanford.edu'
    self.wildcard_csr = """
-----BEGIN CERTIFICATE REQUEST-----
MIIC0zCCAbsCAQAwgY0xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpDYWxpZm9ybmlh
MREwDwYDVQQHEwhTdGFuZm9yZDEcMBoGA1UEChMTU3RhbmZvcmQgVW5pdmVyc2l0
eTEUMBIGA1UECxMLSVQgU2VydmljZXMxIjAgBgNVBAMUGSouaXRzZXJ2aWNlcy5z
dGFuZm9yZC5lZHUwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDsymv0
qtvA3vX0AZ9cI8D8a38JiOffuay5KIneAZsTqf3cfiBXBgWN29uV0J908SoSutOp
j+Ea8ZQw80ucWKb8G1LILWlGxc8WiK7MSkbxASDVidcTY3Zlkv6YiiHD1DzjjjL5
J4OX6n6lb0VUdqLPLl495TPNPT1uIdnu5TDmobdKzyffy7jMYCJeEdBhVwudaIjD
TqMswyUZm4fQ6Sduv+437tSzL2PcrVcNdKPXsYAK3OtF3D3mgHJ4xJyxT1F2EvP4
TG+h7SAsOt7dr14BH/ZreIDaiQcWbVRc9ZKkEn56rvW2jLvUZzsQzMJsOoGBBwg4
+xkcJ6pG5SJinwL5AgMBAAGgADANBgkqhkiG9w0BAQUFAAOCAQEARDLsyJv2SFDz
coADFH1GRoA1sA0J3S1QE7tNnNKiE2DM4bMAbXk8ka2US/CAR4OfWpTggzoanS4e
mmhyHvzbp665nb4gRCJdKqLmjlBXhZmJyzuZvB9H/avbSqAgS3jykb6VqeSIBqBG
aPi58hc0M9fxEW6O0BxnIrwiBm2Ovhvhe9kmgem7eHzHgMGC8DIFJM+8lrCMK4Ek
2qhLuZT1iX5VpBFqj6gV+mSiDzjUV8c03sjmrMHHfa+ppq0Eq5dU0Veo+hHmEFxH
+ioGlUofExVW+n8ctfWZlGVpMfvuz53L8+2qzpErNPreix0MK5A/4KL8kxmMUJHs
hmtPXspPaw==
-----END CERTIFICATE REQUEST-----
"""


  # Create an InCommowWS object (no actual cert requests).
  def test_create_InCommonWS_object(self):
    # Create an InCommonWS object
    incommonws = InCommonWS()

    # Check to see if the default values are correct
    self.assertTrue(incommonws.test_mode, "test_mode should be true by default")
    self.assertEqual(incommonws.username, None, "username should be None by default")
    self.assertEqual(incommonws.password, None, "password should be None by default")
    self.assertEqual(incommonws.authData, None, "authData should be None by default")
    self.assertEqual(incommonws.URI,      None, "URI should be None by default")


  # Create a Certificate object (no actual cert requests).
  def test_create_Certificate_object(self):
    # Create a Certificate object
    certificate = Certificate()
    self.assertTrue(certificate.is_test_certificate, "is_test_certificate should be true by default")

    SERVER_TYPES = certificate.SERVER_TYPES
    self.assertTrue(len(SERVER_TYPES) > 20, "SERVER_TYPES should have length at least 20")

    # Look for ModSSL in the dictionary
    number_of_matches = 0
    for key in SERVER_TYPES:
      if (not(re.compile("modssl", re.IGNORECASE).search(key) is None)):
        number_of_matches = number_of_matches + 1

    self.assertTrue(number_of_matches > 0, "SERVER_TYPES should have an entry for ModSSL")


  # Can we login and get a list of certificate types?
  def test_login(self):
    certificate = self.certificate
    certificate.login()

    # Make sure the certificate types were populated.
    cert_types = certificate.cert_types
    self.assertTrue(len(cert_types) > 6, "there should be at least 6 certificate types")

  # Can we create a CSR (no actual cert requests).
  def test_csr_create(self):
    csr = CSR()
    csr.commonName = self.hostname + '.stanford.edu'

    self.assertTrue(len(csr.template()) > 50, "the template should have at least 50 characters")
    self.assertTrue(csr.make(), "the template should have at least 50 characters")

    self.assertTrue(len(csr.csr_string) > 100,
                    "there should be at least 100 characters in the CSR string")
    re1 = re.compile("^-----BEGIN CERTIFICATE REQUEST-----")
    self.assertTrue(not(re1.search(csr.csr_string) is None),
                    "CSR string should start with -----BEGIN CERTIFICATE REQUEST-----")
    re1 = re.compile("-----END CERTIFICATE REQUEST-----$")
    self.assertTrue(not(re1.search(csr.csr_string) is None),
                    "CSR string should end with -----END CERTIFICATE REQUEST-----")

  # Request a certificate but make some mistakes. Each test should throw
  # an exception.
  def test_missing_attributes(self):
    certificate = self.certificate

    certificate.login()

    # Try to request with some missing atttributes
    save = certificate.csr
    certificate.csr = None
    try:
      certificate.enroll()
    except Exception as exc:
      self.assertTrue(True)
      re1 = re.compile("missing a CSR")
      self.assertTrue(re1.search(exc.args[0]),
                      "exception message should have contained 'missing a CSR'")
    else:
      self.assertTrue(False)
    certificate.csr = save


    save = certificate.term
    certificate.term = None
    try:
      certificate.enroll()
    except Exception as exc:
      self.assertTrue(True)
      re1 = re.compile("term.*not.*defined")
      self.assertTrue(re1.search(exc.args[0]),
                      "exception message should have contained 'term not defined'")
    else:
      self.assertTrue(False)

    certificate.term = 10
    try:
      certificate.enroll()
    except Exception as exc:
      self.assertTrue(True)
      re1 = re.compile("term.*not.*recognized")
      self.assertTrue(re1.search(exc.args[0]),
                      "exception message should have contained 'term not defined'")
    else:
      self.assertTrue(False)

    certificate.term = save


  # Request a UCC certificate (finally!).
  def test_request_certificate(self):
    certificate = self.certificate

    # Set a SAN:
    certificate.add_subjAltName(self.subjectAltName1)

    certificate.login()
    certificate.enroll()

    self.assertTrue(not (certificate.order_number is None))
    re1 = re.compile("^\d\d\d+$")
    self.assertTrue(re1.search(str(certificate.order_number)),
                      "enrollment failed to return a valid order number")

  # Request a wildcard certificate
  def test_request_wildcard_certificate(self):
    certificate = self.certificate

    certificate.csr = self.wildcard_csr.strip()
    certificate.certTypeName = 'InCommon Wildcard SSL Certificate'

    certificate.login()
    certificate.enroll()

    self.assertTrue(not (certificate.order_number is None))
    re1 = re.compile("^\d\d\d+$")
    self.assertTrue(re1.search(str(certificate.order_number)),
                      "enrollment failed to return a valid order number")


  # Request a certificate (finally!).
  def test_is_available_certificate(self):
    certificate = self.certificate
    self.certificate.order_number = '462933' # This one must be available on test site
    certificate.login()                      # for the test to pass.

    self.assertTrue(certificate.isAvailable(), "certificate not available?!?")


  # Get the status of a declined certificate
  def test_status_declined_certificate(self):
    certificate = self.certificate
    self.certificate.order_number = '83204' # This one must be available on test site
    certificate.login()                     # for the test to pass.

    self.assertTrue(certificate.isDeclined(), "certificate not available?!?")


  # Add some SANs to a certificate
  def test_sans_certificate(self):
    certificate = self.certificate
    self.assertTrue(not certificate.is_multi_domain())

    # Add a SAN to the certificate.
    san1 = self.subjectAltName1
    certificate.add_subjAltName(san1)

    # Get the array of SANs.
    sans = certificate.subjAltNames

    self.assertTrue(len(sans) == 1)
    self.assertTrue(sans[0] == san1)

    # Add the same SAN (should not really get addded since already is
    # there).
    certificate.add_subjAltName(san1)
    sans = certificate.subjAltNames

    self.assertTrue(len(sans) == 1)
    self.assertTrue(sans[0] == san1)

    self.assertTrue(certificate.is_ucc())

    # print certificate.to_s()


def suite():
  tests = [
           'test_request_wildcard_certificate',
           'test_sans_certificate',
           'test_request_certificate',
           'test_create_InCommonWS_object',
           'test_create_Certificate_object',
           'test_login',
           'test_csr_create',
           'test_missing_attributes',
           'test_status_declined_certificate',
           'test_is_available_certificate',
          ]
  return unittest.TestSuite(map(TestSequenceFunctions, tests))


if __name__ == '__main__':
  # unittest.main()
  unittest.TextTestRunner(verbosity=1).run(suite())

