from suds.client import Client # for web-services (like SOAP, get it?)
import sys
import time
import hashlib
import json
import re
import string

VERSION = "0.01"

##################################################################
## caching
#
#from beaker.cache import CacheManager
#from beaker.util import parse_cache_config_options
#
#cache_opts = {
#    'cache.type': 'file',
#    'cache.data_dir': '/tmp/cache_incommon/data',
#    'cache.lock_dir': '/tmp/cache_incommon/lock',
#    'cache.expire': 10
#}
#cache = CacheManager(**parse_cache_config_options(cache_opts))
#incommon_cache = cache.get_cache('InCommonCache')
#print str(incommon_cache)
#################################################################
class InCommonWS(object):

  # The username, password, and secretKey should be set by the calling
  # program. Normally, we store them in these wallet objects:
  #   password/its-idg/certreq-test/incommon_credentials
  #   password/its-idg/certreq-prod/incommon_credentials
  #
  # Note that secretKey is only used during certificate enrollment. The other
  # functions require only the username and password.
  username  = None
  password  = None
  secretKey = None
  authData  = None

  test_mode = True

  URI      = None
  URI_test = 'InCommon_test'
  URI_prod = 'InCommon'

  WSDL_URL  = None
  WSDL_URLs = {
               'SSL':       'https://cert-manager.com/ws/EPKIManagerSSL?wsdl',
               'SMIME':      None,
               'Reporting': 'https://cert-manager.com:443/ws/ReportService?wsdl'
              }

  #################################################################
  def login(self):
    # print( 'Logging into InCommon WS...' )

    if self.WSDL_URL is None:
      raise Exception('cannot connect with an empty WSDL_URL')

    if self.username is None:
      raise Exception('cannot connect with an empty username')

    if self.password is None:
      raise Exception('cannot connect with an empty password')

    self.client = Client( self.WSDL_URL )

    self.authData = self.client.factory.create( 'authData' )
    self.authData.login            = self.username
    self.authData.password         = self.password
    self.authData.customerLoginUri = self.URI

  #################################################################
  def to_s(self):
    print('username:          ' + self.username)
    print('password (MD5):    ' + self.password_hashed())
    print('URI:               ' + self.URI)


  #################################################################
  # Set the test_mode
  def setTestMode(self, test_mode_flag):
    self.test_mode = test_mode_flag

    if (self.test_mode):
      URI = self.URI_test
    else:
      URI = self.URI_prod

    return self.test_mode
  #################################################################
  # Return a hashed version of the password suitable for printing.
  def password_hashed(self):
    if (self.password is None):
      return 'NONE'
    else:
      return hashlib.md5(str(self.password).encode('utf-8'))

#################################################################
#################################################################
class InCommonReporting(InCommonWS):

  # Currently only the 'IT Services' organization allows reporting (in
  # prod).
  organizationNames      = None
  organizationNames_prod = 'IT Services'
  organizationNames_test = 'IT Services TEST'

  #################################################################
  def __init__(self):
    super(InCommonReporting, self).__init__()
    self.WSDL_URL = self.WSDL_URLs['Reporting']

    if (self.test_mode):
        self.organizationNames = self.organizationNames_test
    else:
        self.organizationNames = self.organizationNames_prod

  #################################################################
  def login(self):
    # print( 'Logging into Comodo' )

    if self.WSDL_URL is None:
      raise Exception('cannot connect with an empty WSDL_URL')

    if self.username is None:
      raise Exception('cannot connect with an empty username')

    if self.password is None:
      raise Exception('cannot connect with an empty password')

    if self.URI is None:
      raise Exception('cannot connect with an empty URI')

    self.client = Client( self.WSDL_URL )

    self.authData = self.client.factory.create( 'authData' )
    self.authData.login            = self.username
    self.authData.password         = self.password
    self.authData.customerLoginUri = self.URI

  #################################################################
  # Strip out non-printable characters.
  @staticmethod
  def printable(string1):
    # From
    # https://www.kite.com/python/answers/how-to-remove-non-ascii-characters-in-python
    encoded_string = string1.encode("ascii", "ignore")
    return encoded_string.decode()

  #################################################################
  def reportRowSSL_to_json(self, reportRowSSL):
    f2v = {}

    fields = [
               'comments',
               'commonName',
               'id',
               'orderNumber',
               'status',
               'subject',
               'type',
               'serverType',
               'downloaded',
               'requester',
               'expires',
               'approved',
               'issued',
               'replaced',
               'requested',
               'revoked',
             ]

    for field in vars(reportRowSSL).keys():
      if (field in fields):
        value = getattr(reportRowSSL, field)
        if (value is not None):
          value = InCommonReporting.printable(value)

          f2v[field] = value

    return json.dumps(f2v).strip()

  #################################################################
  def getSSLReport(self, start, finish):
    certificateStatus = '0'
    certificateDate   = '0'

    # The requestedVia parameter is, for some reason known only
    # to the developers of the API, NOT documented, but by
    # trial-and-error it appears that setting it to the empty string
    # makes things work. I don't know what it is for. :^(
    requestedVia = ''

    result = self.client.service.getSSLReport(
                                 self.authData,
                                 start,  # e.g., '2014-04-15',
                                 finish, # e.g., '2014-04-16',
                                 self.organizationNames,
                                 certificateStatus,
                                 requestedVia,
                                 certificateDate)

    statusCode = result.statusCode

    if (statusCode != 0):
      raise Exception("failed to get SSLReport; status code returned was " +
      str(statusCode))

    reports = result.reports
    sys.stdout.write("[\n")

    counter = 0
    for report in reports:
      if (counter > 0):
        sys.stdout.write(",\n")
      sys.stdout.write(self.reportRowSSL_to_json(report))
      counter = counter + 1

    sys.stdout.write("\n]")


#################################################################
#################################################################
class CertTypes():
  id_to_name  = {}
  id_to_terms = {}

  def to_s(self):
    for id in self.id_to_name.keys():
      print('id         ' + str(id))
      print('name       ' + self.id_to_name[id])
      print('terms      ' + str(self.id_to_terms[id]))

#################################################################
#################################################################
class Certificate(InCommonWS):

  is_test_certificate = True

  SERVER_TYPES = {
              'AOL': 1,
              'Apache/ModSSL': 2,
              'Apache-SSL (Ben-SSL, not Stronghold)': 3,
              'C2Net Stronghold': 4,
              'Cisco 3000 Series VPN Concentrator': 33,
              'Citrix': 34,
              'Cobalt Raq': 5,
              'Covalent Server Software': 6,
              'IBM HTTP Server': 7,
              'IBM Internet Connection Server': 8,
              'iPlanet': 9,
              'Java Web Server (Javasoft / Sun)': 10,
              'Lotus Domino': 11,
              'Lotus Domino Go!': 12,
              'Microsoft IIS 1.x to 4.x': 13,
              'Microsoft IIS 5.x and later': 14,
              'Netscape Enterprise Server': 15,
              'Netscape FastTrack': 16,
              'Novell Web Server': 17,
              'Oracle': 18,
              'Quid Pro Quo': 19,
              'R3 SSL Server': 20,
              'Raven SSL': 21,
              'RedHat Linux': 22,
              'SAP Web Application Server': 23,
              'Tomcat': 24,
              'Website Professional': 25,
              'WebStar 4.x and later': 26,
              'WebTen (from Tenon)': 27,
              'Zeus Web Server': 28,
              'Ensim': 29,
              'Plesk': 30,
              'WHM/cPanel': 31,
              'H-Sphere': 32,
              'OTHER': -1,
                 }


  uses_ecc = False

  localType = 'single'

  # Note 1: these names are the SHA-1 certificate names; to make the
  # SHA-2 (i.e., SHA-256) names, simply append ' (SHA-2)'. See also
  # setCertTypeName below.
  #
  # Note 2: The ECC certificates only have hashing version, so do _not_
  # append 'SHA-2' to the name.
  CERTLOCALTYPE_TO_NAME = {
    'single':       'InCommon SSL',
    'multi':        'InCommon Multi Domain SSL',
    'wildcard':     'InCommon Wildcard SSL Certificate',
    'intranet':     'InCommon Intranet SSL',
    'ecc_single':   'InCommon ECC',
    'ecc_multi':    'InCommon ECC Multi Domain',
    'ecc_ucc':      'InCommon ECC Multi Domain',
    'ecc_wildcard': 'InCommon ECC Wildcard',
    'ucc':          'InCommon Unified Communications Certificate',
    'ev_sgc':       'Comodo EV SGC SSL',
    'ev_multi':     'Comodo EV Multi Domain SSL',
  }

  REVOCATION_ERROR_CODE_TO_REASON = {
    -14: 'An unknown error occurred!',
    -16: 'Permission denied!',
   -100: 'Invalid authentication data for customer',
   -101: 'Invalid authentication data for customer organization',
   -110: 'Domain is not allowed for customer',
   -111: 'Domain is not allowed for customer organization',
   -120: 'Customer configuration is not allowed the requested action',
  }

  username = None
  password = None

  authData      = None
  csr           = None
  phrase        = "I cast thee out" # revokation phrase
  subjAltNames  = []
  numberServers = 1 # Why would this ever be more than one?
  serverType    = SERVER_TYPES['Apache/ModSSL']
  term          = 2 # in years (1 or 2)
  comments      = 'requested by API'

  # There is an overall orgId ('Stanford') and one for each Deparment.
  orgName_to_orgId_prod = {
    'Stanford':                      1228,
    'IT Services':                   1331,
    'SLAC':                          5204,
    'Emerging Technology':           5176,
    'Stanford Hospital & Clinics':   3942,
    "Stanford Children's Hospital":  3941,
    'Stanford Blood Center':        15286,
  }

  orgName_to_orgId_test = {
    'Stanford':                     1481,
    'SLAC':                         5124,
    'IT Lab TEST':                  4011,
    'Ripslicer Hospital':           3506,
    'IT Services TEST':             1576,
  }

  # orgName is the department name (e.g., "IT Services")
  orgName      = None
  orgId        = None

  cert_types   = None

  certTypeId   = None
  certType     = None # This will have a type derived from a SOAP call.
  certTypeName = None

  # The hash algorithm we want InCommon to use when signing the
  # certificate request. Possible values: 'sha1', 'sha256'.
  signatureHashType = None

  WSDL_URL = None

  client = None

  order_number = None

  #################################################################
  def __init__(self):
    self.WSDL_URL = self.WSDL_URLs['SSL']

    # Default to cert local type of single.
    self.setLocalType('single')
    self.setSignatureHashType('sha256')

  #################################################################
  def set_prod(self):
    self.orgName   = 'IT Services'
    self.orgId     = self.orgName_to_orgId_prod[self.orgName]
    self.URI       = self.URI_prod

  #################################################################
  def set_test(self):
    self.orgName   = 'IT Services TEST'
    self.orgId     = self.orgName_to_orgId_test[self.orgName]
    self.URI       = self.URI_test

  #################################################################
  # Add a subject alternative name to the object.
  #
  # Note that InCommon _always_ adds the certificate common name to the
  # list of subject alternative names. Thus, if you add TWO SANs to the
  # object, then the signed certificate will have THREE SANs: the two you
  # added plus the common name.
  def add_subjAltName(self, subjAltName):
    if (subjAltName not in self.subjAltNames):
      sans_new = self.subjAltNames
      sans_new.append(subjAltName)
      self.subjAltNames = sans_new
      # Since we have a SAN we need to set the type to 'ucc'
      self.setLocalType('ucc')

  # Return a string formatted appropriately for the InCommon web services
  # call.
  def format_subjAltNames_for_InCommon(self):
    return ','.join(self.subjAltNames)

  def is_multi_domain(self):
    return (self.localType == 'multi')

  def is_ucc(self):
    return (self.localType == 'ucc')

  def wants_sha256(self):
    return (self.signatureHashType == 'sha256')

  # Use a setter to set certLocalType so we can be sure that
  # certTypeName also gets set.
  def setLocalType(self, local_type):
    self.localType = local_type
    self.setCertTypeName()

  # Use a setter for signatureHashType so we can be sure that
  # certTypeName also gets set.
  def setSignatureHashType(self, sig_hash):
    self.signatureHashType = sig_hash
    if (self.localType is not None):
      self.setCertTypeName()
    return self.signatureHashType


  # Given a (local) certificate type (e.g., 'single', 'ucc', etc.), return
  # the certType name that InCommon wants. Note that if we are requesting
  # a SHA256 cert, we need to append the string ' (SHA-2)' (except for ECC
  # certificates).
  def setCertTypeName(self):
    cert_type = self.localType
    if (self.uses_ecc):
      cert_type = 'ecc_' + cert_type

    if (cert_type in self.CERTLOCALTYPE_TO_NAME):
      certTypeName_tmp = self.CERTLOCALTYPE_TO_NAME[cert_type]
    else:
      raise Exception("no certificate with local type " + cert_type)

    if (self.wants_sha256() and (not self.uses_ecc)):
      certTypeName_tmp = certTypeName_tmp + ' (SHA-2)'
    self.certTypeName = certTypeName_tmp

    return self.certTypeName


  #################################################################
  def login(self):
    # print( 'Logging into Comodo' )

    if self.WSDL_URL is None:
      raise Exception('cannot connect with an empty WSDL_URL')

    if self.username is None:
      raise Exception('cannot connect with an empty username')

    if self.password is None:
      raise Exception('cannot connect with an empty password')

    if self.URI is None:
      raise Exception('cannot connect with an empty URI')

    self.client = Client( self.WSDL_URL )

    self.authData = self.client.factory.create( 'authData' )
    self.authData.login            = self.username
    self.authData.password         = self.password
    self.authData.customerLoginUri = self.URI

    # Set the cert type now that we are logged in
    self.populateCertTypes()
    self.setCertType() # Now we can set the cert id

  #################################################################
  def populateCertTypes(self):

    ### A closure! (In case we want to do caching at some point)
    def getCertTypes():
      certTypesRaw = self.client.service.getCustomerCertTypesByOrg(self.authData, self.orgId)
      # print(certTypesRaw)
      statusCode = certTypesRaw.statusCode
      if (statusCode < 0):
        raise Exception("failed to populate certificate types; status code returned was " +
          str(statusCode))

      return certTypesRaw
    ###

    # cert_types = incommon_cache.get(key="thingy", createfunc=getCertTypes)
    cert_types = getCertTypes()

    # Recently the cert type names are coming back looking like this:
    #   "InCommon SSL (SHA-2) (customized for IT Services [Stanford University])"
    #   "InCommon Wildcard SSL (SHA-2) (customized for IT Services [Stanford University])"
    #   ...

    # If we define certificate types to be "InCommon SSL (SHA-2)",
    # "InCommon Wildcard SSL (SHA-2)", etc., then a ceritifcate
    # "profile" is a pair (type, department). The point of this is to
    # allow different terms on each type. For example, Department A
    # can set the only possible expiration term for wildcard
    # certificates to 1 year, while Department B can set the possible
    # expirations to 1 or 2 years.

    # We need to connect a "local" type (see CERTLOCALTYPE_TO_NAME above)
    # to a certificate type id that comes back from the API
    # call. Normally, only there will only one be one profile for each
    # type. HOWEVER, currently there is one type that comes back with two
    # profiles:
    #
    #    InCommon SSL (SHA-2) (customized for IT Services [Stanford University])
    #    InCommon SSL (SHA-2) (InCommon level)
    #
    # Due to this overlap, we throw away any profile containing the string
    # "(InCommon level)".

    # The resulting certificate types returned will all have names
    # that match the values of the dict CERTLOCALTYPE_TO_NAME defined
    # above.
    valid_ct_types = []
    for c_type in cert_types.types:
      # We only want certificate types NOT containing the string "(Incommon level)"
      if (c_type.name.find('(InCommon level)') == -1):
        valid_ct_types.append(c_type)

    self.cert_types = valid_ct_types
    return self.cert_types

  #################################################################
  def showCertTypes(self):
    for certType in self.cert_types.types:
      print(str(certType.id) + ' ' + certType.name)


  #################################################################
  # From certTypeName get InCommon's cert type id and set
  # CertTypeId.
  def setCertType(self):
    self.certType = None

    # Look for a certificate type that contains the certificate type we
    # want as a substring. However, because some cert types are substrings
    # of others (e.g., "InCommon ECC" and "InCommon ECC Multi Domain"), if
    # there is more than one match we choose the SHORTEST match.
    cur_match = None
    for certType in self.cert_types:
      if (certType.name.strip().find(self.certTypeName) >= 0):
        if (cur_match is None):
          cur_match = certType
        elif (len(certType.name.strip()) < len(cur_match.name.strip())):
          cur_match = certType
        else:
          # Nothing to do.
          pass

    if (cur_match is None):
      raise Exception("could not find cert type with name '" +
                      str(self.certTypeName) + "'")
    else:
      self.certType   = cur_match
      self.certTypeId = cur_match.id

  #################################################################
  def to_s(self):
    # Reverse the dict
    server_type_id_to_name = {v: k for k, v in self.SERVER_TYPES.items()}

    if (self.csr is None):
      csr_string = 'None'
    else:
      csr_string = hashlib.md5(str(self.csr)).encode('utf-8').hexdigest()

    print('username:          ' + self.username)
    print('password (MD5):    ' + self.password_hashed().hexdigest())
    print('orgId:             ' + str(self.orgId))
    print('secretKey:         ' + str(self.secretKey))
    print('csr (MD5):         ' + csr_string)
    print('revokation phrase: ' + self.phrase)
    print('Subject Alt Names: ' + self.format_subjAltNames_for_InCommon())
    print('number of servers: ' + str(self.numberServers))
    print('term (years):      ' + str(self.term))
    print('comments:          ' + self.comments)
    print('localType          ' + str(self.localType))
    print('certTypeName       ' + str(self.certTypeName))
    print('certTypeId         ' + str(self.certTypeId))
    print('certType           ' + str(self.certType))
    print('signatureHashType  ' + str(self.signatureHashType))
    print('serverType         ' + str(server_type_id_to_name[self.serverType]))
    print('URI:               ' + str(self.URI))


  #################################################################
  def enroll(self):
    # Do some error-checking
    if (self.term is None):
      raise Exception("the certificate term has not been defined")
    elif ((self.term != 1) and (self.term != 2)):
      raise Exception("a certificate term of " + str(self.term) +
                      " is not recognized; must be set to either 1 or 2")

    if (not self.csr):
      raise Exception("this certificate is missing a CSR")

    self.setCertType() # Make sure the certificate type is correct before sending.

    order_number = self.client.service.enroll(
                                 self.authData,
                                 self.orgId,
                                 self.secretKey,
                                 self.csr,
                                 self.phrase,
                                 self.format_subjAltNames_for_InCommon(),
                                 self.certType,
                                 self.numberServers,
                                 self.serverType,
                                 self.term,
                                 self.comments
                        )
    if order_number <= 0:
        print("Error %s on enroll()" % order_number)
        sys.exit()
    else:
      self.order_number = order_number

    return order_number

  #################################################################
  # Returns ??
  def isAvailable(self):
    status = self.status()
    if (status == "fulfilled"):
      return 1
    else:
      return 0

  def isDeclined(self):
    status = self.status()
    if (status == "declined"):
      return 1
    else:
      return 0

  #################################################################
  # Returns:
  #   * "new"       (submitted but not yet approved)
  #   * "approved"  (approved but not yet fulfilled)
  #   * "fulfilled" (fulfilled)
  #   * "declined"  (an RAO/DRAO (cert admin) declined this request)
  def status(self):
    if (self.order_number is None):
      raise Exception("you cannot check whether a certificate is available " +
                      "unless you have an order number")

    else:
      return_value = self.client.service.getCollectStatus(
                                 self.authData,
                                 self.order_number
                                       )
      if (return_value < 0):
        if (return_value == -23):
          # An error code of -23 means that the certificate was not
          # approved.
          return "new"
        elif (return_value == -6):
          # An error code of -6 means "admin has declined request", i.e.,
          # that someone with administrative access to the Certificate
          # Manager web site has declined the request.
          return "declined"
        else:
          # For other error codes we just give up.
          raise Exception("attempting to get collect status returned this " +
                        "error code: " + str(return_value))
      else:
        if (return_value == 0):
          return "approved"
        else:
          return "fulfilled"

  #################################################################
  # format:
  # 0 --> certificate and intermediate certs
  # 1 --> certificate only (no intermediate or root certs)

  def collect(self, format = 0):
    if (self.order_number is None):
      raise Exception("you cannot collect a certificate " +
                      "unless you have an order number")

    else:
      return_value = self.client.service.collect(
                                 self.authData,
                                 self.order_number,
                                 format
                                       )

      # print dir(return_value)
      statusCode = return_value.statusCode

      if (statusCode == 1):
        raise Exception("your certificate is available but cannot be downloaded(?!)")
      elif (statusCode != 2):
        raise Exception("attempting to collect returned this " +
                        "error code: " + str(return_value))
      else:
        return return_value.SSL

  #################################################################
  # Return 0 on success, raise an exception in all other cases.
  def revoke(self, reason='no reason given'):
    if (self.order_number is None):
      raise Exception("you cannot revoke a certificate " +
                      "unless you have an order number")

    else:
      # Prepend to the reason
      reason = 'API: ' + reason

      return_value = self.client.service.revoke(
                                 self.authData,
                                 self.order_number
                                       )

      if (return_value == 0):
        # Success
        return 0
      else:
        if (return_value in self.REVOCATION_ERROR_CODE_TO_REASON):
          error_message = self.REVOCATION_ERROR_CODE_TO_REASON[return_value]
        else:
          error_message = "unrecognized error"

        raise Exception("attempting to revoke returned this " +
                        "error code: " + str(return_value) +
                        " [" + error_message + "]")

#################################################################
