package PuppetBackend::RepoInfo ;

use strict ;
use warnings ;
use autodie ;

use Carp ;
use Data::Dumper ;
use Date::Parse ;
use English ;
use File::Basename ;
use File::Spec ;
use File::stat ;
use IPC::Cmd ;
use POSIX 'strftime' ;
use Readonly ;

use Stanford::Orange::Nagios ;

## no critic (Subroutines::RequireArgUnpacking) ;
## no critic (CodeLayout::ProhibitParensWithBuiltins) ;
## no critic (ErrorHandling::RequireCheckingReturnValueOfEval) ;

# Staging:        /etc/puppetlabs/code-staging/environments/<environment> (Git repo)
# Live directory: /etc/puppetlabs/code/environments/<environment>         (not a Git repo)
#
# CodeManager syncs code from Staging to the live directory.
#
# The last modified date on
# /etc/puppetlabs/code/environments/<environment>/.r10k-deploy.json tells
# you when the live directory was last changed.

Readonly my $BASEGITDIR_DEFAULT  => '/etc/puppetlabs/code-staging/environments' ;
Readonly my $BASELIVEDIR_DEFAULT => '/etc/puppetlabs/code/environments' ;
Readonly my $CANARYFILE_DEFAULT  => '.r10k-deploy.json' ;

Readonly my $REGEX_DEFAULT         => q{.} ;
Readonly my $OUTPUT_FORMAT_DEFAULT => 'date-only' ;
Readonly my $GIT                   => 'git' ;

# Default number of minutes for Nagios warning and critical.
Readonly my $NAGIOS_WARN_MINUTES     => 2 ;
Readonly my $NAGIOS_CRITICAL_MINUTES => 4 ;

####################################################################

sub new {
    my ($class, %args) = @_ ;

    #<<<  perltidy please ignore this section
    # The units of nagios-warning and nagios-critical are seconds.
    my $self = {
        'basegitdir'      => $args{'basegitdir'}      || $BASEGITDIR_DEFAULT,
        'baselivedir'     => $args{'baselivedir'}     || $BASELIVEDIR_DEFAULT,
        'canaryfile'      => $args{'canaryfile'}      || $CANARYFILE_DEFAULT,
        'long'            => $args{'long'}            || 0,
        'regex'           => $args{'regex'}           || $REGEX_DEFAULT,
        'output-format'   => $args{'output-format'}   || $OUTPUT_FORMAT_DEFAULT,
        'sync-seconds'    => $args{'sync-seconds'}    || 0,
        'verbose'         => $args{'verbose'}         || 0,
        'nagios-warning'  => $args{'nagios-warning'}  || $NAGIOS_WARN_MINUTES     * 60,
        'nagios-critical' => $args{'nagios-critical'} || $NAGIOS_CRITICAL_MINUTES * 60,
    };
    #>>>

    # Some validations.
    if ($self->{'nagios-warning'} >= $self->{'nagios-critical'}) {
        croak 'Nagios critical must be greater than Nagios warning' ;
    }

    return bless $self, $class ;
}

sub verbose {
    my $self = shift ;
    my ($verbose) = @_ ;

    if (defined($verbose)) {
        $self->{'verbose'} = $verbose ;
    }
    return $self->{'verbose'} ;
}

sub long {
    my $self = shift ;
    my ($long) = @_ ;

    if (defined($long)) {
        $self->{'long'} = $long ;
    }
    return $self->{'long'} ;
}

sub basegitdir {
    my $self = shift ;
    my ($basegitdir) = @_ ;

    if (defined($basegitdir)) {
        $self->{'basegitdir'} = $basegitdir ;
    }
    return $self->{'basegitdir'} ;
}

sub baselivedir {
    my $self = shift ;
    my ($baselivedir) = @_ ;

    if (defined($baselivedir)) {
        $self->{'baselivedir'} = $baselivedir ;
    }
    return $self->{'baselivedir'} ;
}

sub canaryfile {
    my $self = shift ;
    my ($canaryfile) = @_ ;

    if (defined($canaryfile)) {
        $self->{'canaryfile'} = $canaryfile ;
    }
    return $self->{'canaryfile'} ;
}

sub regex {
    my $self = shift ;
    my ($regex) = @_ ;

    if (defined($regex)) {
        $self->{'regex'} = $regex ;
    }
    return $self->{'regex'} ;
}

sub output_format {
    my $self = shift ;
    my ($output_format) = @_ ;

    if (defined($output_format)) {
        $self->{'output-format'} = $output_format ;
    }
    return $self->{'output-format'} ;
}

sub sync_seconds {
    my $self = shift ;
    my ($sync_seconds) = @_ ;

    if (defined($sync_seconds)) {
        $self->{'sync-seconds'} = $sync_seconds ;
    }
    return $self->{'sync-seconds'} ;
}

sub nagios_critical {
    my $self = shift ;
    my ($nagios_critical) = @_ ;

    if (defined($nagios_critical)) {
        $self->{'nagios-critical'} = $nagios_critical + 0 ;
    }
    return $self->{'nagios-critical'} ;
}

sub nagios_warning {
    my $self = shift ;
    my ($nagios_warning) = @_ ;

    if (defined($nagios_warning)) {
        $self->{'nagios-warning'} = $nagios_warning + 0 ;
    }
    return $self->{'nagios-warning'} ;
}

####################################################################

sub to_s {
    my $self = shift ;

    my $verbose         = $self->verbose() ;
    my $long            = $self->long() ;
    my $regex           = $self->regex() ;
    my $output_format   = $self->output_format() ;
    my $sync_seconds    = $self->sync_seconds() ;
    my $basegitdir      = $self->basegitdir() ;
    my $baselivedir     = $self->baselivedir() ;
    my $canaryfile      = $self->canaryfile() ;
    my $nagios_warning  = $self->nagios_warning() ;
    my $nagios_critical = $self->nagios_critical() ;

    return << "EOS";
verbose:         $verbose
long:            $long
regex:           $regex
out-format:      $output_format
sync-seconds:    $sync_seconds
basegitdir:      $basegitdir
baselivedir:     $baselivedir
canaryfile:      $canaryfile
nagios_warning:  $nagios_warning
nagios_critical: $nagios_critical
EOS
}

sub exit_with_error {
    my $self = shift ;
    my ($msg) = @_ ;
    print "error: $msg\n" ;
    exit 1 ;
}

sub max_length {
    my ($names_aref) = @_ ;

    my @names = @{$names_aref} ;

    my $max_length = 0 ;
    foreach my $name (@names) {
        if (length($name) > $max_length) {
            $max_length = length($name) ;
        }
    }

    return $max_length ;
}

sub progress {
    my $self = shift ;
    my ($msg) = @_ ;
    if ($self->verbose()) {
        print "progress: $msg\n" ;
    }
    return ;
}

sub get_directories {
    my $self = shift ;

    my $basegitdir = $self->basegitdir() ;

    my @directories = () ;

    opendir(my $dir, $basegitdir) ;
    my @files = readdir $dir ;

    for my $file (@files) {
        my $fullpath = File::Spec->catfile(($basegitdir), $file) ;
        if (-d $fullpath) {
            $self->progress("$fullpath is a directory") ;
            push(@directories, $fullpath) ;
        }
    }

    return @directories ;
}

sub get_git_repositories {
    my $self = shift ;

    my @directories = $self->get_directories() ;
    my @repos       = () ;

    # Only include a directory if its name contains at least one
    # "_" (underscore).
    for my $dir (@directories) {
        if ($dir =~ m{_}xsm) {
            $self->progress("found git repo: $dir") ;
            push(@repos, $dir) ;
        }
    }

    return @repos ;
}

# Given a regular expression, return all the git repos that match.
# If no regular expression, return all directories.
sub get_matched_git_repositories {
    my $self = shift ;

    my @repos = $self->get_git_repositories() ;

    my $regex = $self->regex() ;
    if (!$regex) {
        return @repos ;
    } else {
        my @matched_repos = () ;
        foreach my $repo (@repos) {
            # Get the basename
            my $basename = basename($repo) ;
            if ($basename =~ m{$regex}xsm) {
                push(@matched_repos, $repo) ;
            }
        }
        return @matched_repos ;
    }
}

sub format_reponame {
    my $self = shift ;

    my ($reponame) = @_ ;

    my $formatted = basename($reponame) ;

    # Remove the trailing '.git'.
    $formatted =~ s{[.]git$}{}xsm ;

    # Remove all but the source Git repo name unless $self->long() is set.
    if (!$self->long()) {
        # git@uitgitaws.amazon.stanford.edu-iso-pp-iso
        if ($formatted =~ m{^.*[.][^.\-]+-([^.]+)$}xsm) {
            $self->progress("shortening repo name '$formatted'") ;
            $formatted = $1 ;
        }
    }

    return $formatted ;
}

sub last_commits {
    my $self = shift ;

    if ($self->verbose()) {
        print $self->to_s() ;
    }

    my @repos = $self->get_matched_git_repositories() ;
    if (scalar(@repos) == 0) {
        $self->exit_with_error('No matching repositories found.') ;
    }

    my $output_format = $self->output_format() ;

    my @messages = () ;

    my %repo_name_to_last_commit     = () ;
    my %repo_name_to_formatted_name  = () ;
    my %repo_name_to_git_epoch       = () ;
    my %repo_name_to_last_live_epoch = () ;
    foreach my $repo (@repos) {
        my $repo_name = basename($repo) ;
        $self->progress("formatting $repo_name") ;
        my $last_commit ;
        eval {
            $last_commit = $self->git_last_commit_message($repo, $output_format) ;
        } ;
        my $at_error = $EVAL_ERROR ;
        if ($at_error) {
            $self->progress("error when getting last commit message: $at_error") ;
            if ($at_error =~ m{no.commits}xsm) {
                $last_commit = 'problem: no commits' ;
            } else {
                $last_commit = 'problem: unknown error' ;
            }
        }

        # Add more user-friendly time-since text if we are only showing
        # the date.
        my $epoch ;
        if (($output_format =~ m{date-only}ixsm)) {
            # date-only
            if ($last_commit =~ m{^(\d+)::(.*)$}xsm) {
                $epoch       = $1 ;
                $last_commit = trim($2) ;
            }
        } else {
            # extended
            if ($last_commit =~ s{(\d+)\s*\Z}{}xsm) {
                $epoch       = $1 ;
                $last_commit = trim($last_commit) ;
            }
        }

        my $formatted_repo_name = $self->format_reponame($repo) ;

        # Get the last touched time for the corresponding live directory. We
        # wrap in an eval in case the repo is not synced to the live  directory.
        my $last_live ;
        eval { $last_live = $self->last_live_update_epoch($repo_name) ; } ;
        $at_error = $EVAL_ERROR ;
        if (!$at_error) {
            # There was no error so we can use this repo.
            $repo_name_to_last_live_epoch{$repo} = $last_live ;
            $repo_name_to_formatted_name{$repo}  = $formatted_repo_name ;
            $repo_name_to_last_commit{$repo}     = $last_commit ;
            $repo_name_to_git_epoch{$repo}       = $epoch ;
        }
    }

    my @repo_names_formatted = values %repo_name_to_formatted_name ;
    my $max_length           = max_length(\@repo_names_formatted) ;

    # Get a list of all the repositories sorted by last git commit date/time.
    my @repos_sorted =
      reverse sort { $repo_name_to_git_epoch{$a} cmp $repo_name_to_git_epoch{$b} }
      keys %repo_name_to_git_epoch ;

    my $pad_amount = $max_length + 1 ;
    foreach my $repo (@repos_sorted) {
        my $last_commit         = $repo_name_to_last_commit{$repo} ;
        my $repo_name_formatted = $repo_name_to_formatted_name{$repo} ;
        my $last_live_epoch     = $repo_name_to_last_live_epoch{$repo} ;
        my $git_epoch           = $repo_name_to_git_epoch{$repo} ;

        my ($last_live_message, $lag_time) ;
        if ($last_live_epoch >= $git_epoch) {
            $lag_time          = 0 ;
            $last_live_message = q{in sync} ;
        } else {
            # In this case the git repo is more recent than the live
            # directory, so lag_time is simply how many seconds git_epoch
            # is behind the current time.
            $lag_time = time() - $git_epoch ;
            $last_live_message = 'sync behind ' . time_since_text($lag_time, 2) ;
        }
        my $last_commit_time_formatted = time_since_text(time() - $git_epoch, 2) ;

        my $current_message = q{} ;
        if ($output_format =~ m{date-only}xsm) {
            if ($self->sync_seconds()) {
                # If we are showing only the number of seconds out of sync
                $current_message .= sprintf("%${pad_amount}s %s",
                                            $repo_name_formatted . q{:}, $lag_time) ;
            } else {
                # If we are showing normal human-readable message
                $current_message .= sprintf("%${pad_amount}s %s [%s]",
                                            $repo_name_formatted . q{:}, $last_commit_time_formatted, $last_live_message) ;
            }
            push(@messages, $current_message) ;
        } else {
            my $last_live_text = git_format_date($last_live_epoch) ;
            $current_message .= "$repo:\n" ;
            $current_message .= $last_commit . "\n" ;
            $current_message .= 'Live:   ' . $last_live_text . "\n" ;
            $current_message .= '-----------------------------------------------' ;
            push(@messages, $current_message) ;
        }
    }

    return @messages ;
}


##### GIT functions
sub git_run_command {
    my $self = shift ;

    my ($repo, $cmd_aref) = @_ ;

    my @cmd = @{$cmd_aref} ;

    chdir $repo ;

    my @git_cmd = ($GIT, @cmd) ;
    $self->progress('git command is "' . join(q{ }, @git_cmd) . q{"}) ;

    return $self->run_command(@git_cmd) ;
}

# Get the last modified date of the $CANARY_FILE.
sub last_live_update_epoch {
    my $self = shift ;
    my ($repo_name) = @_ ;

    my $fullpath = File::Spec->catfile(($self->baselivedir(), $repo_name), $self->canaryfile()) ;
    $self->progress("last_live_update: fullpath is $fullpath") ;

    my $last_modified_epoch ;
    if (-f $fullpath) {
        $last_modified_epoch = stat($fullpath)->mtime ;
    } else {
        # The file does not exist, so raise an error.
        croak "file $fullpath does not exist"
    }

    return $last_modified_epoch ;
}

# Get last commit message for a repository
sub git_last_commit_message {
    my $self = shift ;
    my ($repo) = @_ ;

    $self->progress('git_last_commit_message') ;

    my $log_format ;
    if ($self->output_format =~ m{date-only}ixsm) {
        $log_format = '%at::%ar' ;
    } else {
        $log_format = 'commit %H%nAuthor: %an%nDate:   %ad%n%at' ;
    }

    my (@cmd, $stdout, $stderr, $rc) ;
    # my @cmd = ($GIT, '-C', $repo, 'log', "--format='$log_format'", '-1') ;

    # Get all branches sorted by commit date
    # https://stackoverflow.com/questions/5188320/how-can-i-get-a-list-of-git-branches-ordered-by-most-recent-commit
    @cmd = ('for-each-ref', '--sort=-committerdate', 'refs/heads/') ;
    ($stdout, $stderr, $rc) = $self->git_run_command($repo, \@cmd) ;
    my @branches = split(/\n/xsm, $stdout) ;
    my $git_hash ;
    if (scalar(@branches) >= 1) {
        my $latest = $branches[0] ;
        $self->progress("latest commit: $latest") ;
        if ($latest =~ m{^(\S+)\s.*}xsm) {
            $git_hash = $1 ;
        }
        $self->progress("latest commit hash: $git_hash") ;
    } else {
    }

    if (!$git_hash) {
        croak 'could not find latest hash' ;
    }

    @cmd = ('log', "--format=$log_format", $git_hash, '-1') ;

    ($stdout, $stderr, $rc) = $self->git_run_command($repo, \@cmd) ;

    if ($stderr =~ m{fatal:.bad.default.revision..HEAD.}xsm) {
        croak 'no commits' ;
    }

    return trim($stdout) ;
}

# Use: pass in an array, returns ($stdout, $stderr, $exit_value)
sub run_command {
    my $self = shift ;
    my (@command) = @_ ;

    # Try to use IPC::Run. If that is not available, try using
    # IPC::Open3.
    $IPC::Cmd::USE_IPC_OPEN3 = 1 ;

    $self->progress(q{about to run command '} . join(q{ }, @command) . q{'}) ;
    my ($ok, $err, $full_buf, $stdout_buffer, $stderr_buffer) =
      IPC::Cmd::run(command => \@command) ;

    # Convert buffers to strings.
    my $stdout = join("\n", @{$stdout_buffer}) ;
    my $stderr = join("\n", @{$stderr_buffer}) ;

    # Try and extract error.
    my $return_code = 0 ;
    if ($err && ($err =~ m{exited[ ]with[ ]value[ ](\d+)$}xsm)) {
        $return_code = $1 ;
    }

    return ($stdout, $stderr, $return_code, $err) ;
}

sub trim {
    my ($s) = @_ ;

    if (!defined($s)) { return $s }

    $s =~ s{^\s*}{}xs ;  ## no critic (RequireLineBoundaryMatching) ;
    $s =~ s{\s*$}{}xs ;  ## no critic (RequireLineBoundaryMatching) ;

    return $s ;
}

### Some date and time functiona
sub plural {
    my ($x) = @_ ;
    if ($x . q{} eq '1') {
        return q{} ;
    } else {
        return q{s} ;
    }
}

sub round_down {
    ## no critic (ValuesAndExpressions::ProhibitMagicNumbers) ;
    my ($x) = @_ ;
    if ($x < 0.9) {
        return 0 ;
    } else {
        return int($x + 0.5) ;
    }
}

sub time_text {
    my ($date_string) = @_ ;

    my $now = time() ;
    return time_since_text($now - str2time($date_string)) ;
}

sub time_since_text {

    my ($seconds) = @_ ;

    if ($seconds < 0) {
        return 'a future time?!?' ;
    }

    if ($seconds == 0) {
        return 'zero seconds' ;
    }

    ## no critic (ValuesAndExpressions::ProhibitMagicNumbers) ;
    my $minutes = round_down($seconds / 60) ;
    if ($minutes == 0) {
        my $s = plural($seconds) ;
        return "$seconds second$s" ;
    }

    my $hours = round_down($seconds / (60 * 60)) ;
    if ($hours == 0) {
        my $s = plural($minutes) ;
        return "$minutes minute$s" ;
    }

    my $days = round_down($seconds / (24 * 60 * 60)) ;
    if ($days == 0) {
        my $s = plural($hours) ;
        return "$hours hour$s" ;
    }

    my $weeks = round_down($seconds / (7 * 24 * 60 * 60)) ;
    if ($weeks == 0) {
        my $s = plural($days) ;
        return "$days day$s" ;
    }

    my $months = round_down($seconds / (30 * 24 * 60 * 60)) ;
    if ($months == 0) {
        my $s = plural($weeks) ;
        return "$weeks week$s" ;
    }

    my $years = round_down($seconds / (365 * 24 * 60 * 60)) ;
    if ($years == 0) {
        my $s = plural($months) ;
        return "$months month$s" ;
    }

    my $s = plural($years) ;
    return "$years year$s" ;
}

sub git_format_date {
    my ($epoch) = @_ ;

    my $date = strftime('%Y-%m-%d %H:%M:%S %z', localtime $epoch) ;
    return $date ;
}

# Return the tuple (secs_behind, reponame) where "reopname" is the
# repository name furthest behind and secs_behind is the number of seconds
# behind this repository is. If there are more than one repository with
# the same maximum seconds behind it returns the repository whose name
# comes first alphabetically.
#
# If ALL repositories are in sync, the tuple returned is (0, undef).
sub max_seconds_behind {
    my $self = shift ;

    # Set sync_sconds to true so that last_commits returns the
    # information in the correct format ("reponame: seconds").
    #
    # Also set the regex to '.' (i.e., everything) so we cover
    # every repository.
    $self->sync_seconds(1) ;
    $self->regex(q{.}) ;

    my @messages = $self->last_commits() ;

    # Each message in @messages has the form "reponame: N" where N is the number of
    # seconds the repo is behind.
    my $max_seconds_behind      = 0 ;
    my $max_seconds_behind_repo = undef ;
    foreach my $message (@messages) {
        if ($message =~ m{^([^:]+):[ ](\d+)$}ixsm) {
            my $reponame    = $1 ;
            my $secs_behind = $2 + 0 ;
            if ($secs_behind > $max_seconds_behind) {
                $max_seconds_behind      = $secs_behind ;
                $max_seconds_behind_repo = trim($reponame) ;
            }
        }
    }

    return ($max_seconds_behind, $max_seconds_behind_repo) ;
}

sub nagios {
    my $self = shift ;

    my ($seconds_behind, $repo_most_behind) = $self->max_seconds_behind() ;

    my $nagios = Stanford::Orange::Nagios->new() ;

    my $msg ;
    if (!defined($nagios)) {
        $nagios->set_unknown() ;
        $msg = 'could not determine number of seconds behind' ;
    } elsif ($seconds_behind > $self->nagios_critical()) {
        $nagios->set_critical() ;
        $msg = "$repo_most_behind is $seconds_behind seconds behind" ;
    } elsif ($seconds_behind > $self->nagios_warning()) {
        $nagios->set_warning() ;
        $msg = "$repo_most_behind is $seconds_behind seconds behind" ;
    } else {
        $nagios->set_ok() ;
        my $repo_text ;
        if (defined($repo_most_behind)) {
            $repo_text = "$repo_most_behind is $seconds_behind seconds behind" ;
        } else {
            $repo_text = 'all repositories in sync' ;
        }
        $msg = "system is within acceptable limits ($repo_text)" ;
    }

    if (defined($seconds_behind)) {
        $msg = "$msg | puppet_run=$seconds_behind" ;
    }

    $nagios->set_exit_message($msg) ;

    # $nagios->finish() calls an "exit" so we never return from this
    # function.
    $nagios->finish() ;
}

1;
