#!/usr/bin/python

# pylint: disable=bad-whitespace,superfluous-parens
# pylint --max-line-length=90

import os
import re
import sys
import cgi
import logging
import tempfile
import subprocess

from wsgiref.handlers import CGIHandler
from flask import Flask
from flask import render_template, request, flash, redirect
from flask_wtf import Form

from wtforms import PasswordField, SubmitField, StringField
from wtforms import validators, ValidationError

import remctl

main_title          = 'Kerberos Root Principal Password Change'
pw_requirements_url = 'https://uit.stanford.edu/service/accounts/passwords/quickguide'

# The authentications are done in the context of a Kerberos environment. This
# environment is set via the environment variable "SU_KERBEROS_ENVIRONMENT". It should
# set to one "dev", "test", "uat", or "prod".
#
# See get_su_kerberos_environment() and get_su_kerberos_environment_port().


# Apache does not send a PATH_INFO for the "root" URL. For example,
# for the URL htts://kpasswd.stanford.edu the PATH_INFO is empty. So, we
# force the PATH_INFO to be "/" in those cases where no PATH_INFO is sent.
if ((not ("PATH_INFO" in os.environ)) or (os.environ["PATH_INFO"] == '')):
    os.environ["PATH_INFO"] = '/'

# Make the actual app.
app = Flask(__name__)

# To enable CRSF protection we need a secret.
app.config.update(
    SECRET_KEY='aWa0OesQgBJpfT0AbeMt',
)

# This creates the form object with the necessary fields.
class ChangePasswordForm(Form):
    oldpassword  = PasswordField('Old Password')
    newpassword1 = PasswordField('New Password')
    newpassword2 = PasswordField('New Password (again)')
    submit = SubmitField("Submit")

# Here is the main route. If the the page is post'ed to, it validates the
# posted date, submits to the change password program, and, possibly,
# generates a success page.

@app.context_processor
def inject_dict_for_all_templates():
    su_kerberos_environment = get_su_kerberos_environment()
    return {'su_kerberos_environment': su_kerberos_environment.upper()}

def render_error(flash_message):
    flash(flash_message, 'error')
    local_title = main_title
    return render_template('error.html',
                           title=local_title,
                           flash_class="flash_error")

@app.route('/',  methods = ['GET', 'POST'])
def index_page():
    ## STAGE 1. Handle serious errors first.
    remote_user = get_remote_user()

    if (remote_user is None):
        return render_error('Missing REMOTE_USER enviroment variable.')

    root_principal  = get_root_principal()

    if (root_principal is None):
        return render_error('Could not determine root principal from REMOTE_USER.')

    if (not user_has_root_principal()):
        return render_error("You ({0}) cannot change your root principal {1} password as "
                            "you do not have a root principal.".format(remote_user, root_principal))

    ## STAGE 2. Handle a post to ourselves.
    errors_found      = False
    flash_class       = None
    change_successful = False
    if request.method == 'POST':
        # Process request.
        if (('oldpassword' in request.form) and (request.form['oldpassword'] != '')):
            oldpassword = request.form['oldpassword']
        else:
            oldpassword = None
            flash('The &quot;OLD password&quot; field cannot be blank.', 'error')
            errors_found = True

        if (('newpassword1' in request.form) and (request.form['newpassword1'] != '')):
            newpassword1 = request.form['newpassword1']
        else:
            newpassword1 = None
            flash('The &quot;NEW password&quot; field cannot be blank.', 'error')
            errors_found = True

        if (('newpassword2' in request.form) and (request.form['newpassword2'] != '')):
            newpassword2 = request.form['newpassword2']
        else:
            newpassword2 = None
            flash('The &quot;NEW password (again)&quot; field cannot be blank.', 'error')
            errors_found = True

        if ((newpassword1 is not None and newpassword2 is not None) and (newpassword1 != newpassword2)):
            flash('NEW passwords do not match.', 'error')
            errors_found = True

        if (not errors_found):
            # If no errors found, try changing the password.
            successful, message = change_password(oldpassword, newpassword1)
            if (successful):
                change_successful = True
            else:
                change_successful = False
                flash("Error: {0}.".format(message))
                errors_found = True

        # End of handling the POST

    # If we get here, then we did not have a successful password change yet.
    local_title = main_title
    if (errors_found):
        flash_class = "flash_error"
    else:
        flash_class = "flash_success"

    form = ChangePasswordForm()
    return render_template('form.html', title=local_title, form=form,
                           flash_class=flash_class, remote_user=remote_user,
                           change_successful=change_successful
    )

@app.route('/error', methods = ['GET'])
def error():
    local_title = 'Encountered an error'
    return render_template('error.html', title=local_title)

@app.route('/help', methods = ['GET'])
def helpp():
    local_title = 'Help'
    return render_template('help.html', title=local_title)

#########################################################

def get_kdc_master():
    su_kerberos_environment = get_su_kerberos_environment()
    return 'kdc-master-' + su_kerberos_environment.lower() + '.stanford.edu'

def get_kdc_port():
    env_to_port = {
        'dev':   '688',
        'test':  '288',
        'uat':   '188',
        'prod':   '88',
        }
    su_kerberos_environment = get_su_kerberos_environment()
    if (su_kerberos_environment in env_to_port):
        return env_to_port[su_kerberos_environment]
    else:
        raise Exception("unrecognized environment {0}".format(su_kerberos_environment))

def get_root_principal():
    remote_user = get_remote_user()

    if (remote_user is not None):
        return remote_user + '/root'
    else:
        return None

def get_remote_user():
    if ('REMOTE_USER' in os.environ):
        remote_user = os.environ['REMOTE_USER']
    else:
        remote_user = None

    if (remote_user is not None):
        remote_user = re.sub('@.+$','', remote_user)
        return remote_user
    else:
        return None

def get_su_kerberos_environment():
    if ("SU_KERBEROS_ENVIRONMENT" not in os.environ):
        raise Exception("missing environment variable SU_KERBEROS_ENVIRONMENT")
    else:
        return os.environ["SU_KERBEROS_ENVIRONMENT"]

def user_has_principal(principal):
    if (principal is None):
        return False

    os.environ["KRB5CCNAME"] = "FILE:/run/kpasswd/kpasswd_web_principal.tkt"

    # Some stuff...
    command = ('exists', principal)
    try:
        result = remctl.remctl(host = get_kdc_master(), command = command)
    except remctl.RemctlProtocolError, err:
        app.logger.error('%s remctl error', str(err))
        sys.exit(1)
    if (result.status == 0):
        return True
    else:
        return False

def user_has_root_principal():
    root_principal = get_root_principal()
    return user_has_principal(root_principal)

def random_temp_file(suffix):
    _, temp_path = tempfile.mkstemp(suffix = suffix, dir = '/run/kpasswd')
    return temp_path

def log_f(message):
    with open('/tmp/flog.txt', 'a') as the_file:
        the_file.write(message + "\n")

# Create a ticket cache for the current user using kinit. We make the lifetime very short,
# We store the user's password in a temporary file so we can use kinit-from-file.
# We put this file in /run so that in case it does not get cleaned up properly it will
# be deleted on reboot.
def create_ticket_cache(oldpassword):
    cache_file     = random_temp_file('.krbcc')
    password_file  = random_temp_file('.password')
    root_principal = get_root_principal()

    # Put the password in the password_file
    with open(password_file, 'w') as the_file:
        the_file.write(oldpassword + "\n")

    cmd = ['/usr/sbin/kinit-from-file',
           password_file,
           '-l5m',
           '-F',
           '-S', 'kadmin/changepw',
           '-c', cache_file,
           root_principal]

    # result = Popen(cmd, stdout=PIPE, stderr=PIPE)
    # stdout, stderr = process.communicate()
    pipes = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    std_out, std_err = pipes.communicate()
    exit_code = pipes.returncode

    # Delete the password file
    os.remove(password_file)

    return (cache_file, std_out, std_err, exit_code)

# Returns:
#
# (stdout, stderr, exit code)
def make_remctl(newpassword, credentials_cache):

    # Set up Kerberos context.
    os.environ['KRB5CCNAME'] = credentials_cache

    stdout    = None
    stderr    = None
    exit_code = None

    command = ('kadmin', 'password', newpassword)
    try:
        result = remctl.remctl(host = get_kdc_master(), port = '4443',
                               principal = 'kadmin/changepw', command = command)
    except remctl.RemctlProtocolError, err:
        return (None, str(err), None)
    if result.stdout:
        stdout = result.stdout
    if result.stderr:
        stderr = result.stderr

    exit_code = result.status

    return (stdout, stderr, exit_code)

# Create a krb5.conf file pointing at the correct
# KDC environment.
def make_krb5_conf():
    krb5_string = '''
[appdefaults]
    default_lifetime      = 25hrs

[libdefaults]
    default_realm         = stanford.edu
    ticket_lifetime       = 25h
    renew_lifetime        = 7d
    forwardable           = true
    noaddresses           = true
    allow_weak_crypto     = true
    rdns                  = true

[realms]
    stanford.edu = {{
        kdc            = {kdc}:{port}
        default_domain = stanford.edu
        kadmind_port   = 749
    }}
'''.format(kdc = get_kdc_master(), port = get_kdc_port())

    krb5_conf_file = random_temp_file('.conf')
    with open(krb5_conf_file, 'w') as the_file:
        the_file.write(krb5_string + "\n")

    return krb5_conf_file

# Make a remctl call to the KDC master to change the password
# of the principal. Returns the array (success, message) where
# success is True if it worked, and False if it did not.
def change_password(oldpassword, newpassword):
    result  = None
    message = None

    # Step 1. Create the krb5.conf file and make sure we use it in
    # our subsequent Kerberos authentications.
    krb5_conf_location        = make_krb5_conf()
    os.environ['KRB5_CONFIG'] = krb5_conf_location

    # Step 2. Create the credentials cache for this user's principal.
    credentials_cache, std_out, std_err, exit_code = create_ticket_cache(oldpassword)

    if (exit_code != 0):
        result  = False
        root_principal = get_root_principal()
        message = ("problem creating ticket cache "
                   "for {0}; this could be due to an "
                   "incorrect old password"
                   )
        message = message.format(root_principal)
    else:
        # Step 3.Make the remctl call that changes the password.
        std_out, std_err, exit_code = make_remctl(newpassword, credentials_cache)
        if (exit_code != 0):
            result = False
            if (std_err is None):
                message = 'unknown problem'
            else:
                # See if this is one of the recognized errors
                pattern = r'error: (.*)$'
                matches = re.match(pattern, std_err)
                if (matches):
                    err_msg_formatted = matches.group(1)
                else:
                    err_msg_formatted = 'unknown error'

                message = ('unable to change your password: <strong>{0}</strong>. '
                           'Be sure the new password meets '
                           '<a href="{1}" target="_blank">Stanford\'s '
                           'password complexity '
                           'requirements</a>'
                          )
                message = message.format(err_msg_formatted, pw_requirements_url)
        else:
            result  = True
            message = "worked"

    # Cleanup.
    os.remove(krb5_conf_location)
    os.remove(credentials_cache)

    return (result, message)


CGIHandler().run(app)
