package edu.stanford.krb;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.security.auth.DestroyFailedException;
import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.kerberos.KerberosKey;
import javax.security.auth.kerberos.KerberosPrincipal;
import javax.security.auth.kerberos.KerberosTicket;
import javax.security.auth.login.LoginException;
import javax.security.auth.spi.LoginModule;

import sun.misc.HexDumpEncoder;
import sun.security.jgss.krb5.Krb5Util;
import sun.security.krb5.Credentials;
import sun.security.krb5.EncryptionKey;
import sun.security.krb5.KrbAsReqBuilder;
import sun.security.krb5.KrbException;
import sun.security.krb5.PrincipalName;




/**
 * A login module that makes an AS-REQ for a specific service principal. A
 * {@link CallbackHandler} needs to be used to prompt for a password and client
 * principal when requesting a key for the service principal.
 * 
 * <p>
 * Specifically, no TGT is requested.
 * </p>
 * 
 * <p>
 * This class is heavily based on Krb5LoginModule. That class may change between
 * major Java releases, and will require changes here as well
 * </p>
 * 
 * <p>
 * The following options are supported in the JAAS config file.
 * <ul>
 * <li>debug=true</li>
 * <li>serverPrincipal="service/some-principal"</li>
 * <li>refreshKrb5Config=true</li>
 * </ul>
 * <p>
 * 
 * @author pradtke
 * 
 */
public class ServiceASReqLoginModule implements LoginModule {
    // initial state
    private Subject subject;
    private CallbackHandler callbackHandler;
    private Map sharedState;
    private Map<String, ?> options;

    private boolean debug = false;

    private static boolean DEBUG = true;
    private String serverPrincName = null;
    private PrincipalName serverPrincipal = null;
    private boolean refreshKrb5Config = false;

    static final java.util.ResourceBundle rb = java.util.ResourceBundle.getBundle("sun.security.util.AuthResources");

    // FIXME: move these into login
    StringBuffer krb5PrincName = null;
    private char[] password = null;
    // FIXME: these are all internal sun classes. Do we need them?
    private EncryptionKey[] encKeys = null;
    private Credentials cred = null;
    private PrincipalName principal = null;
    private KerberosPrincipal kerbClientPrinc = null;
    private KerberosTicket kerbTicket = null;
    private KerberosKey[] kerbKeys = null;

    private boolean succeeded = false;

    @Override
    public boolean abort() throws LoginException {
        // TODO Auto-generated method stub
        return false;
    }

    @Override
    public boolean commit() throws LoginException {
        this.kerbTicket = Krb5Util.credsToTicket(this.cred);
        boolean storeKey = true;

        Set<Object> privCredSet = this.subject.getPrivateCredentials();
        Set<java.security.Principal> princSet = this.subject.getPrincipals();
        this.kerbClientPrinc = new KerberosPrincipal(this.principal.getName());

        if (storeKey) {
            if (this.encKeys == null || this.encKeys.length <= 0) {
                this.succeeded = false;
                throw new LoginException("Null Server Key ");
            }

            this.kerbKeys = new KerberosKey[this.encKeys.length];
            for (int i = 0; i < this.encKeys.length; i++) {
                Integer temp = this.encKeys[i].getKeyVersionNumber();
                this.kerbKeys[i] = new KerberosKey(this.kerbClientPrinc, this.encKeys[i].getBytes(),
                        this.encKeys[i].getEType(), (temp == null ? 0 : temp.intValue()));
            }

        }
        // Let us add the kerbClientPrinc,kerbTicket and kerbKey (if
        // storeKey is true)
        if (!princSet.contains(this.kerbClientPrinc)) {
            princSet.add(this.kerbClientPrinc);
        }

        // add the TGT
        if (this.kerbTicket != null) {
            if (!privCredSet.contains(this.kerbTicket)) {
                privCredSet.add(this.kerbTicket);
            }
        }

        if (storeKey) {
            for (int i = 0; i < this.kerbKeys.length; i++) {
                if (!privCredSet.contains(this.kerbKeys[i])) {
                    privCredSet.add(this.kerbKeys[i]);
                }
                this.encKeys[i].destroy();
                this.encKeys[i] = null;
                if (this.debug) {
                    System.out.println("Added server's key" + this.kerbKeys[i]);
                    System.out.println("\t\t[Krb5LoginModule] " + "added Krb5Principal  "
                            + this.kerbClientPrinc.toString() + " to Subject");
                }
            }
        }
        return this.succeeded;

    }

    @Override
    public void initialize(Subject subject, CallbackHandler callbackHandler, Map<String, ?> sharedState,
            Map<String, ?> options) {

        // initialize any configured options
        this.subject = subject;

        this.debug = "true".equalsIgnoreCase((String) options.get("debug"));
        this.serverPrincName = (String) options.get("serverPrincipal");
        this.refreshKrb5Config = "true".equalsIgnoreCase((String) options.get("refreshKrb5Config"));

        this.callbackHandler = callbackHandler;

        if (this.debug) {
            System.out.print("Debug is  " + this.debug + " server principal is " + this.serverPrincName + " refreshKrb5Config is " + refreshKrb5Config + "\n");
        }
    }

    @Override
    public boolean login() throws LoginException {
        if (refreshKrb5Config) {
            try {
                if (debug) {
                    System.out.println("Refreshing Kerberos configuration");
                }
                sun.security.krb5.Config.refresh();
            } catch (KrbException ke) {
                LoginException le = new LoginException(ke.getMessage());
                le.initCause(ke);
                throw le;
            }
        }

        // prompt for name
        this.promptForName();
        // prompt for password
        this.promptForPass();

        // TODO Auto-generated method stub

        try {
            this.serverPrincipal = new PrincipalName(this.serverPrincName);
            this.principal = new PrincipalName(this.krb5PrincName.toString(), PrincipalName.KRB_NT_PRINCIPAL);
        } catch (KrbException e) {
            LoginException le = new LoginException(e.getMessage());
            le.initCause(e);
            throw le;
        }

        try {

            if (this.debug) {
                System.out.println("Acquire TGT using AS Exchange");
            }
            KrbAsReqBuilder builder = new KrbAsReqBuilder(this.principal, this.password);
    		// this differs from the standard Krb5LoginModule since it requires us to set the server principal
    		builder.setTarget(this.serverPrincipal);
    		this.cred = builder.action().getCreds();   
            this.encKeys = EncryptionKey.acquireSecretKeys(this.password, this.principal.getSalt());

//            encKeys = builder.getKeys();
//
            // Get the TGT using AS Exchange
            if (this.debug) {
                System.out.println("principal is " + this.principal);
                HexDumpEncoder hd = new HexDumpEncoder();
                for (int i = 0; i < this.encKeys.length; i++) {
                    System.out.println("EncryptionKey: keyType=" + this.encKeys[i].getEType() + " keyBytes (hex dump)="
                            + hd.encode(this.encKeys[i].getBytes()));
                }
            }
        } catch (KrbException e) {
            LoginException le = new LoginException(e.getMessage());
            le.initCause(e);
            throw le;
        } catch (IOException ioe) {
            LoginException ie = new LoginException(ioe.getMessage());
            ie.initCause(ioe);
            throw ie;
        }

        // we should hava a non-null cred
        if (this.cred == null) {
            throw new LoginException("TGT Can not be obtained from the KDC ");
        }
        this.succeeded = true;
        return true;
    }

    @Override
    public boolean logout() throws LoginException {
        if (this.debug) {
            System.out.println("\t\t[Krb5LoginModule]: " + "Entering logout");
        }

        if (this.subject.isReadOnly()) {
            this.cleanKerberosCred();
            throw new LoginException("Subject is Readonly");
        }

        this.subject.getPrincipals().remove(this.kerbClientPrinc);
        // Let us remove all Kerberos credentials stored in the Subject
        Iterator<Object> it = this.subject.getPrivateCredentials().iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if (o instanceof KerberosTicket || o instanceof KerberosKey) {
                it.remove();
            }
        }
        // clean the kerberos ticket and keys
        this.cleanKerberosCred();

        this.succeeded = false;
        if (this.debug) {
            System.out.println("\t\t[Krb5LoginModule]: " + "logged out Subject");
        }
        return true;
    }

    /**
     * Clean Kerberos credentials
     */
    private void cleanKerberosCred() throws LoginException {
        // Clean the ticket and server key
        try {
            if (this.kerbTicket != null) {
                this.kerbTicket.destroy();
            }
            if (this.kerbKeys != null) {
                for (int i = 0; i < this.kerbKeys.length; i++) {
                    this.kerbKeys[i].destroy();
                }
            }
        } catch (DestroyFailedException e) {
            throw new LoginException("Destroy Failed on Kerberos Private Credentials");
        }
        this.kerbTicket = null;
        this.kerbKeys = null;
        this.kerbClientPrinc = null;
    }

    private void promptForName() throws LoginException {

        this.krb5PrincName = new StringBuffer("");

        if (this.callbackHandler == null) {
            throw new LoginException("No CallbackHandler " + "available " + "to garner authentication "
                    + "information from the user");
        }
        try {
            String defUsername = System.getProperty("user.name");

            Callback[] callbacks = new Callback[1];
			MessageFormat form = new MessageFormat(
					rb.getString("Kerberos.username.defUsername."));
            Object[] source = { defUsername };
            callbacks[0] = new NameCallback(form.format(source));
            this.callbackHandler.handle(callbacks);
            String username = ((NameCallback) callbacks[0]).getName();
            if (username == null || username.length() == 0) {
                username = defUsername;
            }
            this.krb5PrincName.insert(0, username);

        } catch (java.io.IOException ioe) {
            throw new LoginException(ioe.getMessage());
        } catch (UnsupportedCallbackException uce) {
            throw new LoginException(uce.getMessage() + " not available to garner " + " authentication information "
                    + " from the user");
        }

    }

    private void promptForPass() throws LoginException {

        // FIXME: move this into login, and remove from here and from forName
        if (this.callbackHandler == null) {
            throw new LoginException("No CallbackHandler " + "available " + "to garner authentication "
                    + "information from the user");
        }
        try {
            Callback[] callbacks = new Callback[1];
            String userName = this.krb5PrincName.toString();
			MessageFormat form = new MessageFormat(
					rb.getString("Kerberos.password.for.username."));
            Object[] source = { userName };
            callbacks[0] = new PasswordCallback(form.format(source), false);
            this.callbackHandler.handle(callbacks);
            char[] tmpPassword = ((PasswordCallback) callbacks[0]).getPassword();
            if (tmpPassword == null) {
                // treat a NULL password as an empty password
                tmpPassword = new char[0];
            }
            this.password = new char[tmpPassword.length];
            System.arraycopy(tmpPassword, 0, this.password, 0, tmpPassword.length);
            ((PasswordCallback) callbacks[0]).clearPassword();

            // clear tmpPassword
            for (int i = 0; i < tmpPassword.length; i++) {
                tmpPassword[i] = ' ';
            }
            tmpPassword = null;
            if (this.debug) {
                System.out.println("\t\t[Krb5LoginModule] " + "user entered username: " + this.krb5PrincName);
                System.out.println();
            }
        } catch (java.io.IOException ioe) {
            throw new LoginException(ioe.getMessage());
        } catch (UnsupportedCallbackException uce) {
            throw new LoginException(uce.getMessage() + " not available to garner " + " authentication information "
                    + "from the user");
        }
    }

}
