package Stanford::KDC::RegressionTesting;

use strict;
use warnings;
use autodie;

## no critic (RequireArgUnpacking);
## no critic (Documentation::RequirePodAtEnd);
## no critic (Documentation::RequirePodSections);
## no critic (InputOutput::RequireCheckedSyscalls);
## no critic (CodeLayout::ProhibitParensWithBuiltins);
## no critic (ErrorHandling::RequireCheckingReturnValueOfEval);

use Stanford::Orange::Util qw/ spew trim run_command_improved /;

# Make this a Moose class
use Moose;
use namespace::autoclean;

use Data::Dumper;
use Carp;
use DateTime;
use File::Temp qw/ tempfile /;
use IPC::Run qw/ run timeout /;
use English qw( -no_match_vars );

use Stanford::FlexTest;
use Stanford::FlexTest::Collection;
use Stanford::KDC::HeimdalHistory;

use Readonly;

# $RANDOM_PW_LENGTH:      length of random password.
# $RANDOM_PW_NUM_DIGITS:  number of digits in random password.
# $RANDOM_PW_NUM_SYMBOLS: number of symbols in random password.
Readonly my $RANDOM_PW_LENGTH      => 20;
Readonly my $RANDOM_PW_NUM_DIGITS  => 2;
Readonly my $RANDOM_PW_NUM_SYMBOLS => 2;

# $AD_PASSWORD_SLEEP_TIME_SECONDS: An extra amount of time in seconds
# that we should wait when changing the password for the AD password
# to get changed in the AD.
Readonly my $AD_PASSWORD_SLEEP_TIME_SECONDS => 3;

# The password expiration sync runs every two hours.  We consider that the
# password expiration test passes if the number of seconds since the last
# error is at least this many hours.
Readonly my $PASSWORD_EXPIRATON_HOURS_LIMIT => 2.5;

# This is the default limit on the number of files in the krb5sync spool
# directory before we consider there to be a problem with files not
# getting processed fast enough.
Readonly my $KRB5SYNC_SPOOL_DIR_MAX_LENGTH => 100;

Readonly my $NUM_SECONDS_IN_HOUR => 3600;

has debug => (
    is       => 'rw',
    isa      => 'Bool',
    required => 1,
    default  => 0,
);

# Which kdc we want to connect to
has kdc_servers => (
    is       => 'rw',
    isa      => 'ArrayRef[Str]',
    required => 1,
    default  => sub { [] },
);

has kdc_server_port => (
    is       => 'rw',
    isa      => 'Str',
    required => 1,
    default  => '88',
);

has kdc_master => (
    is       => 'rw',
    isa      => 'Str',
    required => 1,
);

has realm => (
    is       => 'rw',
    isa      => 'Str',
    required => 1,
    default  => 'stanford.edu',
);

has kadmin_flavor => (
    is       => 'rw',
    isa      => 'Str',
    required => 0,
);

has kadmin_server => (
    is       => 'rw',
    isa      => 'Str',
    required => 1,
);

has kadmin_principal => (
    is       => 'rw',
    isa      => 'Str',
    required => 0,
);

has kadmin_principal_keytab => (
    is       => 'rw',
    isa      => 'Str',
    required => 0,
);

has kadmin_password => (
    is       => 'rw',
    isa      => 'Str',
    required => 0,
);

has use_local => (
    is       => 'rw',
    isa      => 'Bool',
    required => 1,
    default  => 0,
);

has win_realm => (
    is       => 'rw',
    isa      => 'Str',
    required => 1,
    default  => 'WIN.STANFORD.EDU',
);

has win_dc => (
    is       => 'rw',
    isa      => 'Str',
    required => 1,
    default  => 'mothra.win.stanford.edu',
);

has win_ldap_base => (
    is       => 'rw',
    isa      => 'Str',
    required => 1,
    default  => 'ou=Accounts,dc=WINUAT,dc=STANFORD,dc=EDU',
);

has in_nagios_mode => (
    is       => 'rw',
    isa      => 'Bool',
    required => 1,
    default  => 0,
);

has principal_prefix => (
    is       => 'rw',
    isa      => 'Str',
    required => 1,
    default  => 'testing__',
);

has krb5_sync_queue_dir => (
    is       => 'rw',
    isa      => 'Str',
    required => 1,
    default  => '/var/spool/krb5-sync',
);

sub BUILD {
    my ($self) = shift;
    return;
}

sub DEMOLISH {
    my ($self) = shift;
    $self->delete_krb5_conf();
    return;
}

=head2 Methods

=over

=item $self->delete_krb5_conf

delete_krb5_conf

=cut

sub delete_krb5_conf {
    my ($self) = shift;

    my $krb5_conf_file = $ENV{'KRB5_CONFIG'};
    if ($krb5_conf_file && (-e $krb5_conf_file)) {
        $self->progress("about to unlink $krb5_conf_file");
        unlink $krb5_conf_file;
    }
    return;
}

=item $self->progress

progress

=cut

sub progress {
    my ($self) = shift;

    my ($msg) = @_;
    if ($self->debug()) {
        print $msg . "\n";
    }
    return;
}

=item $self->to_string

to_string

=cut

sub to_string {
    my ($self) = shift;

    my $kadmin_principal
      = $self->kadmin_principal()
      ? $self->kadmin_principal()
      : '<not defined>';
    my $kadmin_principal_keytab
      = $self->kadmin_principal_keytab()
      ? $self->kadmin_principal_keytab()
      : '<not defined>';

    my @kdc_servers = @{ $self->kdc_servers() };
    my $kdc_servers = join(q{,}, @kdc_servers);

    my $krb5_config_env = $ENV{'KRB5_CONFIG'};

    my $krb5_conf = $self->krb5_conf();

    my $local = $self->use_local();
    my $local_fmt;
    if ($local) {
        $local_fmt = 'TRUE';
    } else {
        $local_fmt = 'FALSE';
    }

    return <<"EOG";
kadmin principal: $kadmin_principal
kadmin keytab:    $kadmin_principal_keytab
kdc servers:      $kdc_servers
use local:        $local_fmt
KRB5_CONFIG:      $krb5_config_env
krb5_conf:
----------
$krb5_conf
----------
EOG
}

=item $self->run_command

run_command

=cut

sub run_command {
    my ($self) = shift;
    my (@cmd)  = @_;

    $self->progress(q{about to run command '} . join(q{ }, @cmd) . q{'});

    return run_command_improved(@cmd);
}

=item $self->krb5_conf

krb5_conf

=cut

sub krb5_conf {
    my ($self) = shift;

    my $kadmin_server = $self->kadmin_server();
    if (!$kadmin_server) {
        croak 'cannot create a krb5.conf file without the kadmin server';
    }

    my $kdc_master = $self->kdc_master();
    if (!$kdc_master) {
        croak 'cannot create a krb5.conf file without the kdc master';
    }

    my @kdc_servers = @{ $self->kdc_servers() };
    if (!@kdc_servers) {
        croak 'cannot create a krb5.conf file without at least one kdc';
    }

    my $realm = $self->realm();
    my $port  = $self->kdc_server_port();

    # Construct the string for the kdc servers.
    my @kdc_configs    = ();
    my $cur_kdc_config = q{};
    foreach my $kdc_server (@kdc_servers) {
        $cur_kdc_config .= "        kdc            = $kdc_server:$port";
        push(@kdc_configs, $cur_kdc_config);
    }
    my $kdc_config = join("\n", @kdc_configs);

    # Windows AD
    my $win_realm     = $self->win_realm();
    my $win_dc        = $self->win_dc();
    my $win_ldap_base = $self->win_ldap_base();

    my $krb5_conf_string = <<"EOK";
[libdefaults]
    default_realm         = $realm
    ticket_lifetime       = 25h
    renew_lifetime        = 7d
    forwardable           = true
    noaddresses           = true
    allow_weak_crypto     = true
    rdns                  = false
[appdefaults]
    krb5-sync = {
        ad_keytab         = /var/lib/heimdal-kdc/krb5-sync.keytab
        ad_principal      = service/krb5-sync\@stanford.edu
        ad_realm          = $win_realm
        ad_admin_server   = $win_dc
        ad_ldap_base      = $win_ldap_base
        ad_instances      = sunet
        queue_dir         = /var/spool/krb5-sync
        ad_queue_only     = true
    }
[realms]
    $realm = {
$kdc_config
        master_kdc     = $kdc_master:$port
        admin_server   = $kadmin_server
        kpasswd_server = $kadmin_server
        default_domain = $realm
        kadmind_port   = 749
    }

    $win_realm = {
        kdc            = $win_dc:88
        kpasswd_server = $win_dc
    }

EOK
    return trim($krb5_conf_string);
}

=item $self->set_krb5_conf

set_krb5_conf

=cut

sub set_krb5_conf {
    my ($self) = shift;

    my ($fh, $filename) = tempfile();

    spew($self->krb5_conf, $filename);

    ## no critic (Variables::RequireLocalizedPunctuationVars);
    $ENV{'KRB5_CONFIG'} = $filename;

    return;
}

=item $self->which_kadmin

Returns the array (type, version_number) where "type" will be either
'mit' or 'heimdal' and version_number will be the version.  If it cannot
figure it out, raises an exception.

=cut

sub which_kadmin {
    my @cmd = qw(kadmin --version);

    my ($stdout, $stderr, $rc);
    eval { ($stdout, $stderr, $rc) = run_command_improved(@cmd); };
    my $aterror = $EVAL_ERROR;
    if ($aterror =~ m{command.*not.*found}ixsm) {
        croak 'could not find the kadmin command; maybe it is not installed?';
    }

    my $type;
    if ($stderr =~ m{heimdal|Kungliga}ixsm) {
        $type = 'heimdal';
    } elsif ($stderr =~ m{invalid.*option}ixsm) {
        $type = 'mit';
    } else {
        croak 'could not determine flavor of kadmin command?!?';
    }

    my $version_number;
    if ($stderr =~ m{\s(\S+)[)]}ixsm) {
        $version_number = $1;
    } else {
        croak 'could not determine version number of kadmin command?!?';
    }

    return ($type, $version_number);
}

=item $self->is_old_heimdal

Returns 1 if the kadmin command installed is heimdal and version 1.x.x.

=cut

sub is_old_heimdal {
    my ($type, $version) = which_kadmin();

    if (($type eq 'heimdal') && ($version =~ m{^1[.]}ixsm)) {
        return 1;
    } else {
        return 0;
    }
}

=item $self->run_kadmin_command

run_kadmin_command

=cut

sub run_kadmin_command {
    my ($self) = shift;

    my ($action, $args_aref) = @_;

    # Make sure we point to the correct krb5.conf:
    $self->set_krb5_conf();

    # Construct the kadmin command.
    my @cmd = ('kadmin');

    my @args;
    if ($args_aref) {
        @args = @{$args_aref};
    } else {
        @args = ();
    }

    if ($self->use_local()) {
        # LOCAL MODE
        push(@cmd, '-l');
    } else {
        # NON-LOCAL MODE
        my $keytab    = $self->kadmin_principal_keytab();
        my $principal = $self->kadmin_principal();

        if (!$principal) {
            croak 'cannot run kadmin command in non-local '
              . 'without the principal to run as';
        }

        if (!$keytab) {
            croak 'cannot run kadmin command in non-local '
              . 'without the keytab file';
        }

        push(@cmd, '-p', $principal);
        push(@cmd, '-K', $keytab);
    }

    #<<<  perltidy: I like it like this!
    push(@cmd,
        '-c', $ENV{'KRB5_CONFIG'},
        $action,
        @args,
        );
    #>>>

    my ($stdout, $stderr, $rc) = $self->run_command(@cmd);

    # Delete temporary krb5.conf file
    $self->delete_krb5_conf();

    return ($stdout, $stderr, $rc);
}

=item $self->create_principal

create_principal

=cut

sub create_principal {
    my ($self) = shift;

    my %args = (
        'principal' => undef,
        'password'  => undef,
        'krb5_sync' => 0,
        @_,
    );

    my $principal = $args{'principal'};
    my $password  = $args{'password'};
    my $krb5_sync = $args{'krb5_sync'};

    if (!$principal) {
        croak 'missing required principal';
    }

    # If no password passed, use a random password
    my $password_option;
    if (!$password) {
        $password_option = '-r';
    } else {
        $password_option = "--password=$password";
    }

    #<<<  perltidy: I like it like this!
    my @kadmin_args = (
        $password_option,
        '--max-ticket-life=unlimited',
        '--max-renewable-life=unlimited',
        '--expiration-time=never',
        '--pw-expiration-time=never',
        '--attributes=disallow-svr',
        );
    #>>>

    # If this is the NEW heimdal, we have to add the policy attribute.
    if (!is_old_heimdal()) {
        push(@kadmin_args, '--policy=default');
    }

    # Add the principal to add.
    push(@kadmin_args, $principal);

    my ($stdout, $stderr, $rc)
      = $self->run_kadmin_command('add', \@kadmin_args);

    $self->progress("[$stdout,$stderr]");

    if ($rc == 0) {
        if (!$krb5_sync) {
            $self->purge_sync_pool($principal);
        }
        return 1;
    } else {
        croak $stderr;
    }
}

=item $self->principal_exists

principal_exists

=cut

sub principal_exists {
    my ($self)      = shift;
    my ($principal) = @_;

    my ($stdout, $stderr, $rc) = $self->get_principal($principal);

    if ($rc == 0) {
        return 1;
    } else {
        return 0;
    }
}

=item $self->get_principal

get_principal

=cut

sub get_principal {
    my ($self)      = shift;
    my ($principal) = @_;

    if (!$principal) {
        croak 'missing required principal';
    }

    return $self->run_kadmin_command('get', [$principal]);
}

=item $self->delete_principal

delete_principal

=cut

sub delete_principal {
    my ($self)      = shift;
    my ($principal) = @_;

    if (!$principal) {
        croak 'missing required principal';
    }

    my ($stdout, $stderr, $rc)
      = $self->run_kadmin_command('delete', [$principal]);

    if ($rc == 0) {
        return 1;
    } else {
        return 0;
    }
}

=item $self->purge_sync_pool

purge_sync_pool

=cut

sub purge_sync_pool {
    my ($self)      = shift;
    my ($principal) = @_;

    my $queue_dir = $self->krb5_sync_queue_dir;

    # Get all the files in the queue directory matching $principal.
    opendir(my $DIR, $queue_dir);

    my @files_to_process = ();
    while (my $file = readdir($DIR)) {
        if ($file =~ m{^$principal-ad-(password|enable)}xsm) {
            # This matches, so delete
            my $abs_path = File::Spec->catfile(($queue_dir), $file);
            unlink $abs_path;
            $self->progress("deleted krb5-sync file '$abs_path'");
        }
    }
    closedir($DIR);
    return;
}

=item $self->change_password

change_password

=cut

sub change_password {
    my ($self) = shift;
    my ($principal, $password, $krb5_sync) = @_;

    if (!$principal) {
        croak 'missing required principal';
    }

    if (!$password) {
        croak 'missing required password';
    }

    if (!$krb5_sync) {
        $krb5_sync = 0;
    }

    my ($stdout, $stderr, $rc)
      = $self->run_kadmin_command('passwd',
        [qq{--password=$password}, $principal]);

    if ($rc == 0) {
        if (!$krb5_sync) {
            $self->purge_sync_pool($principal);
        }
        return 1;
    } else {
        croak $stderr;
    }
}

=item $self->ext_keytab

ext_keytab

=cut

sub ext_keytab {
    my ($self) = shift;
    my ($principal, $keytab_file) = @_;

    if (!$principal) {
        croak 'missing required principal';
    }

    if (!$keytab_file) {
        croak 'missing required keytab_file';
    }

    my ($stdout, $stderr, $rc)
      = $self->run_kadmin_command('ext_keytab',
        ["--keytab=$keytab_file", $principal]);

    if ($rc == 0) {
        return 1;
    } else {
        croak $stderr;
    }
}

=item $self->check

check

=cut

sub check {
    my ($self) = shift;

    my ($stdout, $stderr, $rc) = $self->run_kadmin_command('check');

    if ($rc == 0) {
        return 1;
    } else {
        return 0;
    }
}

=item $self->list

list

=cut

sub list {
    my ($self) = shift;

    my ($search_string) = @_;

    if (!$search_string) {
        $search_string = q{*};
    }

    my ($stdout, $stderr, $rc)
      = $self->run_kadmin_command('list', [qq{$search_string}]);

    if ($rc != 0) {
        croak "failed to list principals matching '$search_string'";
    } else {
        my @principals = split(/\n/xsm, trim($stdout));
        return @principals;
    }

}

=item $self->kinit

kinit

=cut

sub kinit {
    my ($self) = shift;

    my ($principal, $password) = @_;

    if (!$principal) {
        croak 'missing required principal';
    }

    if (!$password) {
        croak 'missing required password';
    }

    my ($fh1, $password_file) = tempfile();
    spew($password, $password_file);

    my ($fh2, $credentials_cache) = tempfile();

    #<<<  perltidy: I like it like this!
    my @cmd = (
        'kinit',
        '-c',  $credentials_cache,
        "--password-file=$password_file",
        $principal,
        );
    #>>>

    my ($stdout, $stderr, $rc) = $self->run_command(@cmd);
    unlink($password_file);
    unlink($credentials_cache);

    if ($rc == 0) {
        return 1;
    } else {
        croak $stderr;
    }
}

# Returns 1 if we have tickets

=item $self->have_tickets

have_tickets

=cut

sub have_tickets {
    my ($self) = shift;

    #<<<  perltidy: I like it like this!
    my @cmd = (
        'klist',
    );
    #>>>

    my ($stdout, $stderr, $rc) = $self->run_command(@cmd);

    # Match any part of string (ignore newlines)
    # E.g., "krbtgt/heimdal.stanford.edu@heimdal.stanford.edu"
    # or "krbtgt/stanford.edu@stanford.edu"
    if ($stdout =~ m{krbtgt/\S+@\S}xsm) {
        return 1;
    } else {
        return 0;
    }
}

# Returns 1 if we have tickets

=item $self->destroy_tickets

destroy_tickets

=cut

sub destroy_tickets {
    my ($self) = shift;

    #<<<  perltidy: I like it like this!
    my @cmd = (
        'kdestroy',
    );
    #>>>

    my ($stdout, $stderr, $rc) = $self->run_command(@cmd);

    if ($rc == 0) {
        return 1;
    } else {
        return 0;
    }
}

=item $self->generate_password

generate_password

=cut

sub generate_password {
    my ($self) = shift;

    my ($length) = @_;

    if (!$length) {
        $length = $RANDOM_PW_LENGTH;
    }

    my @chars = ('A' .. 'Z', 'a' .. 'z', '0' .. '9');

    my $string = q{};

    for (1 .. $length) {
        $string .= $chars[rand @chars];
    }

    return $string;
}

# This genrates a password that AD likes, namely, one that contains
# numbers and symbols

=item $self->generate_ad_password

generate_ad_password

=cut

sub generate_ad_password {
    ## no critic (ProhibitNoisyQuotes);
    my ($self) = shift;

    my ($length) = @_;

    if (!$length) {
        $length = $RANDOM_PW_LENGTH;
    }

    my $string = q{};

    # Generate $RANDOM_PW_NUM_DIGITS random digits
    my @numbers = ('0' .. '9');
    for (1 .. $RANDOM_PW_NUM_DIGITS) {
        $string .= $numbers[rand @numbers];
    }

    # Generate $RANDOM_PW_NUM_SYMBOLS random symbols
    my @symbols = ('!', '#', '$', '%', '&', '*', '-', '+', '=');
    for (1 .. $RANDOM_PW_NUM_SYMBOLS) {
        $string .= $symbols[rand @symbols];
    }

    # Generate enough letters to make up the rest
    my $num_letters
      = $length - ($RANDOM_PW_NUM_DIGITS + $RANDOM_PW_NUM_SYMBOLS);
    my @chars = ('A' .. 'Z', 'a' .. 'z');
    for (1 .. $num_letters) {
        $string .= $chars[rand @chars];
    }

    return $string;
}

=item $self->process_krb5_sync_pool

process_krb5_sync_pool

=cut

sub process_krb5_sync_pool {
    my ($self)      = shift;
    my ($principal) = @_;

    # Make sure that the KRB5_CONFIG is set to empty so that the
    # default system krb5.conf file is used.
    delete $ENV{'KRB5_CONFIG'};

    # We only process the files with the AD test principal
    my $queue_dir = $self->krb5_sync_queue_dir;
    opendir(my $DIR, $queue_dir);

    my @files_to_process = ();
    while (my $file = readdir($DIR)) {
        if ($file =~ m{$principal}xsm) {
            my $abs_path = File::Spec->catfile(($queue_dir), $file);
            push(@files_to_process, $abs_path);
        }
    }
    closedir($DIR);
    my @files_to_process_sorted = sort @files_to_process;
    $self->progress('number of files matching test AD is '
          . scalar(@files_to_process_sorted));

    foreach my $file_to_process (@files_to_process_sorted) {
        my @cmd = ('/usr/sbin/krb5-sync', '-f', $file_to_process);
        my ($stdout, $stderr, $rc) = $self->run_command(@cmd);

        if ($stderr) {
            croak "error running krb5-sync-backend process: $stderr";
        } else {
            $self->progress("stdout was $stdout");
        }
    }
    return;
}

=item $self->pad

pad

=cut

sub pad {
    my ($x) = @_;

    ## no critic (ValuesAndExpressions::ProhibitMagicNumbers);
    if ($x < 10) {
        return '0' . $x;
    } else {
        return q{} . $x;
    }
}

# We generate a random principal name. It will have the form
# "testing__YYYYMMDDHHMMSS__RRRRRR" where "RRRRRR" are six random
# lower-case letters. We put the current date to make it easier for the
# heimdal history cleanup process to avoid deleting any principals we are
# in the middle of testing.
#
# Note that this function generates the NAME of the principal, not the
# principal itself.

=item $self->random_long_principal

random_long_principal

=cut

sub random_long_principal {
    my ($self) = shift;

    my @chars = ('a' .. 'z');

    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst)
      = localtime(time);

    ## no critic (ValuesAndExpressions::ProhibitMagicNumbers);
    $year += 1900;
    $mon  = pad($mon + 1);
    $mday = pad($mday);
    $hour = pad($hour);
    $min  = pad($min);
    $sec  = pad($sec);

    my $prefix = $self->principal_prefix();
    my $string = $prefix . "$year$mon$mday$hour$min$sec" . '__';

    for (1 .. 6) {
        $string .= $chars[rand @chars];
    }

    return $string;
}

#####################################################################

# TESTS

# All tests return an array of the form:
#
# (
#   [sub_test_status1, msg1],
#   [sub_test_status2, msg2],
# )
#
# where "sub_test_statusN" has one of these values:
#  *  1 (sub-test passed)
#  *  0 (sub-test failed)
#  * -1 (sub-test skipped)
# "msg" is the text message for a successful sub-test.
#
# Runs:
#  * kadmin list 'kadmin'
#
# Returns an array of principals that start with 'kadmin'.

=item $self->test_simple

test_simple

=cut

sub test_simple {
    my ($self) = shift;

    my ($result, $fmsg);
    my $flextests = Stanford::FlexTest::Collection->new();
    $flextests->ok_mode(!$self->in_nagios_mode);

    my @principals_found = ();
    my @principals = ('kadmin/admin', 'kadmin/changepw');
    foreach my $principal (@principals) {
        if ($self->principal_exists($principal)) {
            push(@principals_found, $principal);
        } else {
        }
    }

    my $number_principals_found = scalar(@principals_found);
    $self->progress("test_simple: found $number_principals_found principals");

    # Test 1. Did we find the two principals?
    $result = (($number_principals_found >= 2) ? 1 : 0);
    $fmsg = 'found at least two principals';
    $flextests->add($result, $fmsg);

    return $flextests;
}

# Create, verify, and delete a principal. We don't need any slaves
# here.

=item $self->test_basic

test_basic

=cut

sub test_basic {
    my ($self) = shift;

    my ($result, $fmsg);
    my $flextests = Stanford::FlexTest::Collection->new();
    $flextests->ok_mode(!$self->in_nagios_mode);

    my $password = $self->generate_password($RANDOM_PW_LENGTH);

    my $principal     = $self->random_long_principal();
    my $principal_bad = $self->random_long_principal();

    # Test 1. We just generated a random principal. It jolly well better
    # not exist.
    $result = !$self->principal_exists($principal_bad);
    $fmsg   = 'verified that first principal does not already exist';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 2. Verify that the second random principal also does not
    # exist.
    $result = !$self->principal_exists($principal);
    $fmsg   = 'verified that second principal does not already exist';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 3. Create $principal.
    $result = $self->create_principal(
        'principal' => $principal,
        'password'  => $password
    );
    $fmsg = 'created principal successfully';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 4. Verify that $principal exists.
    $result = $self->principal_exists($principal);
    $fmsg   = 'principal now exists';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 5. Delete $principal exists.
    $result = $self->delete_principal($principal);
    $fmsg   = 'deleted principal';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 6. Verify that $principal was properly deleted.
    $result = !$self->principal_exists($principal);
    $fmsg   = 'verified that principal no longer exists';
    $flextests->add($result, $fmsg);

    return $flextests;
}

=item $self->test_dictionary_password

test_dictionary_password

=cut

sub test_dictionary_password {
    my ($self) = shift;

    my ($result, $fmsg);
    my $flextests = Stanford::FlexTest::Collection->new();
    $flextests->ok_mode(!$self->in_nagios_mode);

    if ($self->use_local) {
        print "in local mode so skipping these checks\n";
        return;
    }

    my $principal = $self->random_long_principal();

    # Test 1. We just generated a random principal. It jolly well better
    # not exist.
    $result = !$self->principal_exists($principal);
    $fmsg   = 'principal does not yet exist';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 2. Try to create the principal using a too-simple password.
    my $password = 'password';
    eval {
        $self->create_principal(
            'principal' => $principal,
            'password'  => $password
        );
    };
    my $aterror = $EVAL_ERROR;
    $result = ($aterror =~ m{Password.is.in.the.password.dictionary}ixsm);
    $fmsg   = 'dictionary password successfully caught';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 3. Verify that the previous principal create falied.
    $result = !$self->principal_exists($principal);
    $fmsg   = 'principal create with dictionary password prevented';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    return $flextests;
}

=item $self->test_password_history

test_password_history

=cut

sub test_password_history {
    my ($self) = shift;

    my ($result, $fmsg);
    my $flextests = Stanford::FlexTest::Collection->new();
    $flextests->ok_mode(!$self->in_nagios_mode);

    my $principal = $self->random_long_principal();

    # Test 1. We just generated a random principal. It jolly well better
    # not exist.
    $result = !$self->principal_exists($principal);
    $fmsg   = 'principal does not yet exist';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 2. Create the principal.
    $self->create_principal('principal' => $principal);
    $result = $self->principal_exists($principal);
    $fmsg   = 'created test_password_history principal';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 3. Change the password.
    my $new_password;

    $new_password = $self->generate_password();
    $result       = $self->change_password($principal, $new_password);
    $fmsg         = 'changed password';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 4. Change the password to the SAME password.
    eval { $self->change_password($principal, $new_password); };
    my $aterror = $EVAL_ERROR;
    $result = ($aterror =~ m{Password.is.in.the.password.dictionary}ixsm);
    $fmsg   = 're-used password caught';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Test 5. Delete the principal
    $result = $self->delete_principal($principal);
    $fmsg   = 'deleted test_password_history principal';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    return $flextests;

}

=item $self->test_ad_krb5_sync

test_ad_krb5_sync

=cut

sub test_ad_krb5_sync {
    my ($self)              = shift;
    my ($ad_test_principal) = @_;

    my ($result, $fmsg);
    my $flextests = Stanford::FlexTest::Collection->new();
    $flextests->ok_mode(!$self->in_nagios_mode);

    # Test 1. Make sure the principal $ad_test_principal exists.
    $result = $self->principal_exists($ad_test_principal);
    $fmsg   = 'AD test principal exists';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Step 2. OK, the principal exists, so change its password.
    my $random_password = $self->generate_ad_password();

    # Note that the third argument in the next call is "1" which tells the
    # change_password method to sync the password with the AD.
    $result = $self->change_password($ad_test_principal, $random_password, 1);
    $fmsg = 'AD test principal password changed on kerberos';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Step 3. Process any files in the krb5-sync spool so that the
    # password is pushed to the Active Directory.
    $self->process_krb5_sync_pool($ad_test_principal);

    # Step 4. Do a kinit against the AD domain to verify the password
    # propagated correctly. Wait a few seconds to give the password pushed
    # in Step 3 time to get there.
    sleep($AD_PASSWORD_SLEEP_TIME_SECONDS);
    my $principal_fully_qualified
      = $ad_test_principal . q{@} . $self->win_realm;

    # Do a kinit against the Windows AD realm.
    $result = $self->kinit($principal_fully_qualified, $random_password);
    $fmsg = 'kinit against AD test principal works';
    $flextests->add($result, $fmsg);

    if (!$result && $self->in_nagios_mode) {
        return $flextests;
    }

    # Step 5. Delete the password history for this principal (otherwise it
    # grows too large).
    my $heimdal_history = Stanford::KDC::HeimdalHistory->new();

    ## no critic (ValuesAndExpressions::RequireInterpolationOfMetachars);
    my $principal_fq_local = $ad_test_principal . q{@stanford.edu};
    if ($heimdal_history->exzists($principal_fq_local)) {
        my $msg = "principal $principal_fq_local found "
          . 'in Heimdal password history database';
        $self->progress($msg);
        $heimdal_history->remove($ad_test_principal . q{@stanford.edu});
    } else {
        my $msg = "principal $principal_fq_local NOT found "
          . 'in Heimdal password history database';
        $self->progress($msg);
    }

    return $flextests;
}

=item test_password_sync_log

Look for problems in the password sync log.

=cut

sub test_password_sync_log {
    my ($self)              = shift;
    my ($password_sync_log) = @_;

    my $seconds = $self->pwexp_seconds_since_last_error($password_sync_log);
    if (defined($seconds)) {
        $self->progress("last error is $seconds seconds old");
    } else {
        $self->progress('no last error was found');
    }

    my ($result, $fmsg);
    my $flextests = Stanford::FlexTest::Collection->new();
    $flextests->ok_mode(!$self->in_nagios_mode);

    # The password expiration sync runs every two hours.  We consider that
    # the test passes if the number of seconds since the last error is at
    # least 2.5 hours.
    $result = (!defined($seconds)
          || ($seconds > ($PASSWORD_EXPIRATON_HOURS_LIMIT * 60 * 60)));
    if (!$result) {
        my $hours = sprintf '%.1f', $seconds / $NUM_SECONDS_IN_HOUR;
        $fmsg = "password expiration too recent: $hours hours ago";
    } else {
        $fmsg = 'password expiration most recent error is old enough';
    }
    $self->progress($fmsg);
    $flextests->add($result, $fmsg);

    return $flextests;
}

=item dir_number_files

Returns number of files in a directory. Includes '.' and may include
'..'.

=cut

sub dir_number_files {
    my ($dirname) = @_;

    opendir(my $dh, $dirname);
    my @files = readdir($dh);
    closedir($dh);

    return scalar(@files);
}

=item test_krb5_sync_spool_dir

Looks for problems in the krb5-sync spool directory (only for
masters doing WINDOWS password sync.

=cut

sub test_krb5_sync_spool_dir {
    my ($self) = shift;
    my ($krb_pool_dir, $maxlength) = @_;

    if (!defined($krb_pool_dir)) {
        croak 'missing maxlength argument';
    }

    if (!defined($maxlength)) {
        croak 'missing spool directory argument';
    }

    my $pool_length = dir_number_files($krb_pool_dir);

    if (defined($pool_length)) {
        $self->progress("length of $krb_pool_dir is $pool_length ");
    } else {
        $self->progress("could not determine length of $krb_pool_dir");
    }

    my ($result, $fmsg);
    my $flextests = Stanford::FlexTest::Collection->new();
    $flextests->ok_mode(!$self->in_nagios_mode);

    if (!defined($pool_length)) {
        $fmsg   = "length of $krb_pool_dir could not be determined";
        $result = 0;
    } elsif ($pool_length <= 0) {
        $fmsg   = "length of $krb_pool_dir is not positive";
        $result = 0;
    } elsif ($pool_length > $maxlength) {
        $fmsg
          = "length of $krb_pool_dir is too large ($pool_length > $maxlength)";
        $result = 0;
    } else {
        $fmsg   = "length of $krb_pool_dir is $pool_length";
        $result = 1;
    }

    $self->progress($fmsg);
    $flextests->add($result, $fmsg);

    return $flextests;
}

=item pwexp_seconds_since_last_error

Get the time (in seconds) since there was a problem with the password
expiration sync script.

=cut

sub pwexp_seconds_since_last_error {
    ## no critic (RequireBriefOpen)
    my ($self)              = shift;
    my ($password_sync_log) = @_;

    if (!$password_sync_log) {
        $password_sync_log = '/var/log/password-expiration.log';
    }

    # Get all lines that start with ERROR.
    open(my $FH, '<', $password_sync_log);
    $self->progress("starting scan of $password_sync_log");

    my $time_zone = 'local';

    my $now = DateTime->now('time_zone' => $time_zone);

    my $seconds_since_last_error;
    while (my $line = <$FH>) {
        chomp($line);
        if ($line =~ m{^(\S+)\s+ERROR}xsm) {
            my $timestamp = $1;
            $self->progress("found error line: $line");
            if ($timestamp
                =~ m{^(\d{4})-(\d\d)-(\d\d)T(\d\d):(\d\d):(\d\d)}xsm)
            {
                my $year  = $1;
                my $month = $2;
                my $dom   = $3;
                my $hour  = $4;
                my $min   = $5;
                my $sec   = $6;
                my $dt    = DateTime->new(
                    year      => $year,
                    month     => $month,
                    day       => $dom,
                    hour      => $hour,
                    minute    => $min,
                    second    => $sec,
                    time_zone => $time_zone,
                );

                my $duration = $now->subtract_datetime_absolute($dt);
                $seconds_since_last_error = $duration->seconds();
                $self->progress(
                    "error line is $seconds_since_last_error seconds old");
            }
        }
    }
    close($FH);

    return $seconds_since_last_error;
}

=back

=cut

1;

__END__

=for stopwords mit heimdal kinit krb KDCs OpenLDAP

=head1 NAME

Stanford::KDC::RegressionTesting - Testing framework for Kerberos KDCs

=cut

=head1 Stanford::LDAP::RegressionTesting

Stanford::LDAP::RegressionTesting - Support module for Stanford OpenLDAP regression testing

=cut
