<?php
  /*
    Copyright (C) 2009 - 2012 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

  /**
  * @package phpldap
  * @author Andreas Andersson
  * @version 1.1
  *
  * Impl class
  */
  class LDAPImp {

    var $hConn;
    var $sBase;
    var $sLogFile;

    var $sLastSearch;

    function LDAPImp() {
      return;
    }

    /**
    * Opens connection to server
    * @param string $sHost LDAP Hostname
    * @param int $nPort LDAP Port
    * @return void
    */    
    function open_impl($sHost = "", $nPort = 0) {
      $this->setLog("open, host: ".$sHost.", Port number (0 = default): ".$nPort);

      $this->sHost = $sHost;

      // create ldap connection
      if(!empty($nPort)) {
        // if port defined
        $this->hConn = ldap_connect($this->sHost, $nPort);
      }
      else {
        // this could be a hostname or URL 'ldaps://<server>:<port>'
        $this->hConn = ldap_connect($this->sHost);
      }

      // set default ldap options
      if(defined('LDAP_OPT_PROTOCOL_VERSION')) {
        ldap_set_option($this->hConn, LDAP_OPT_PROTOCOL_VERSION, 3);
      }
      if(defined('LDAP_OPT_REFERRALS')) {
        ldap_set_option($this->hConn, LDAP_OPT_REFERRALS, 0);
      }

      return;
    }

    /**
    * Database connection exists
    * @return boolean
    */
    function connection_exists() {
      if(!$this->hConn) {
        return false;
      }

      return true;
    }

    /**
    * Close connection to server
    * @return boolean
    */
    function close_impl() {
      $this->setLog("close");

      if(!ldap_close($this->hConn)) {
        return false;
      }

      return true;
    }


    /**
    * Bind using an open connection
    * @param string $sUserDn User DN
    * @param string $sPassword
    * @param boolean $bSecure True/False for secure connection using TLS
    * @return boolean
    */    
    function bind_impl($sUserDN = "", $sPassword = "", $bSecure = false) {
      $this->setLog("bind DN: ".$sUserDN.", password: ".$sPassword.", secure: ".$bSecure);

      if($bSecure) {
        if(!$this->startTLS()) {
          return false;
        }
      }

      // simple binding
      return @ldap_bind($this->hConn, $sUserDN, $sPassword);
    }

    /**
    * Activate TLS
    * Requires a working certificate
    * @return boolean
    */
    function startTLS() {
      $this->setLog("start tls");

      if(!ldap_set_option($this->hConn, LDAP_OPT_PROTOCOL_VERSION, 3)) {
        trigger_error("Can't use protocol version 3", E_USER_ERROR);
        setLog("start tls, can't use protocol version 3");
        return false;
      }

      if (!ldap_start_tls($this->hConn)) {
        trigger_error("Unable to start in TLS mode", E_USER_ERROR);
        setLog("start tls, unable to start in tls mode");
        return false;
      }

      return true;
    }

    /**
    * Internal Query function.
    * Call <b>query</b> or <b>query_one_level</b> instead of this one
    * @param string $sQuery Your Query
    * @param string $sType One or two
    * @param array $aAttributes LDAP attributes (optional)
    * @return resource
    */
    function query_handle ($sQuery, $sType = "one", $aAttributes = array()) {
      $this->setLog("query: ".$sQuery.
                    ", type: ".$sType.
                    ", base: ".$this->sBase.
                    ", attr: ".implode(",", $aAttributes));

      if(!$this->hConn) {
        trigger_error("No LDAP connection available", E_USER_WARNING);
        return false;
      }

      // search level one|sub
      if($sType == "one") {
        $hResult = ldap_list($this->hConn, $this->sBase, $sQuery, $aAttributes);
      }
      else if($sType == "sub") {
        $hResult = ldap_search($this->hConn, $this->sBase, $sQuery, $aAttributes);
      }
      else {
        trigger_error("Unknown query option", E_USER_WARNING);
        return false;
      }

      // save search
      $this->sLastSearch = $sQuery;

      if(!$hResult) {
        trigger_error("Invalid search", E_USER_WARNING);
        return false;
      }

      return $hResult;
    }

    /**
    * Get Entry Impl
    * @param string $sDN
    * @param array $aAttributes
    * @param string $sFilter
    * @return resource
    */
    function get_entry_impl($sDN, $aAttributes = array(), $sFilter = "(objectclass=*)") {
      $this->setLog("get entry. dn: ".$sDN.", filter: ".$sFilter." attr: ".implode(",", $aAttributes));
      return ldap_read ($this->hConn, $sDN, $sFilter, $aAttributes);
    }


    /**
    * If you want to set or change the base dn
    * 
    * @param string $sBaseDn LDAP Base DN
    * @return void
    */
    function set_base_dn_impl($sBaseDn) {
      $this->setLog("set base dn: ".$sBaseDn);
      $this->sBase = $sBaseDn;
    }

    /**
    * Get num rows from ldap
    * 
    * @param object $hResult
    * @return integer
    */
    function num_rows_impl($hResult) {
      $this->setLog("rows in result: ".ldap_count_entries ($this->hConn, $hResult));
      return ldap_count_entries ($this->hConn, $hResult);
    }

    /**
    * Returns result array
    * 
    * @param object $hResult
    * @return array Multi dimensional array
    */
    function get_result_impl($hResult) {
      $this->setLog("get result");
      return ldap_get_entries ($this->hConn, $hResult);
    }

    /**
    * Free resultset
    * @param object $hResult Resultset from search
    * @return void
    */
    function free_result_impl($hResult) {
      $this->setLog("free result");
      ldap_free_result($hResult);
    }

    /**
    * Get first entry in resultset
    * @param object $hResult Resultset
    * @return resource
    */
    function get_first_entry_impl($hResult) {
      $this->setLog("get first entry in result");
      return ldap_first_entry ($this->hConn, $hResult);
    }


    /**
    * Get binary attribute value
    * @param object $hEntry LDAP Entry
    * @param string $sAttribute
    * @return binary
    */
    function get_value_binary_impl($hEntry, $sAttribute) {
      $this->setLog("get binary attr: ".$sAttribute);
      return ldap_get_values_len ($this->hConn, $hEntry, $sAttribute);
    }

    /**
    * Get attributenames
    * @param object $hEntry
    * @return array
    */
    function get_attributes_impl($hEntry) {
      $this->setLog("get attributes");
      return ldap_get_attributes ($this->hConn, $hEntry);
    }

    /**
    * Translate characters from UTF
    * @param string Unconverted string
    * @return string
    */
    function convert_characters_impl($sData) {
      return utf8_decode($sData);
    }

    /**
    * Returns last query
    * @return string LDAP Query
    */
    function get_last_search_query() {

      if(empty($this->sLastSearch)) {
        return "";
      }

      return $this->sLastSearch;
    }

    /**
    * Serversided sort of resultset
    * @param object $hResult
    * @param string Sort By Attribute
    * @return void
    */
    function sort_impl($hResult, $sSortAttribute) {
      $this->setLog("server sided sort, attr:".$sSortAttribute);
      ldap_sort ($this->hConn, $hResult, $sSortAttribute);
    }
    
    /**
    * Add To Logfile
    * @param string sLog Info to log
    * @return bool
    */
    function setLog($sLog) {
      if(!empty($this->sLogFile)) {
        $fp = @fopen($this->sLogFile, "a");

        if(!$fp) {
          return false;
        }

        if (@fwrite($fp, date("Y-m-d H:i:s")." - ".$sLog."\n") === FALSE) {
          return false;
        }
        @fclose($fp);
        return true;
      } // log file exist
    }
  }
?>
