<?php
  /*
    Copyright (C) 2009 - 2012 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */


  /**
  * Include implementation file
  */
  require_once dirname(__FILE__)."/ldap_class_impl.php";

  /**
  * @package phpldap
  * @author Andreas Andersson
  * @version 1.1
  * Authenticate example:
  * <code>
  * <?
  * <b>require "ldap/ldap_class.php";</b>
  *
  * $o = new LDAP("ldapserver.yourdomain.com");
  *
  *  // auth test
  *  if($o->authenticate("andreas.andersson", "secret", true)) {
  *    echo "authenticated\n";
  *  }
  *  else {
  *    echo "not authenticated\n";
  *  }
  *
  * $o->close();
  * ?>
  * </code>
  */
  class LDAP extends LDAPImp {

    var $RESULT_IS_NULL = "Resultset is null!";

    /**
    * Constructor
    *   Will open a connection if Host is available
    *   Will perform a bind if user DN is available
    * @param string $sHost LDAP Hostname (optional)
    * @param int $nPort LDAP Port (optional)
    * @param string $sBase LDAP Base E.g: dc=base,dc=com (optional)
    * @param string $sUserDN User DN (optional)
    * @param string $sPassword (optional)
    * @param boolean $bSecure Use Secure Connection (optional)
    * @return boolean
    */
    function LDAP($sHost = "", $nPort = 0, $sUserDN = "", $sPassword = "", $bSecure = false) {

      if(!empty($sHost)) {
        if(!$this->open($sHost, $nPort, $sUserDN, $sPassword, $bSecure)) {
          return false;
        }
      }

      return true;
    }

    /**
    * Opens connection to server
    *   Will perform a bind if user DN is available
    * @param string $sHost LDAP Hostname Default ldap://
    * @param int $nPort LDAP Port Default 389
    * @param string $sBase LDAP Base e.g. dc=domain,dc=com (optional)
    * @param string $sUserDN Full User DN (optional)
    * @param string $sPassword (optional)
    * @param boolean $bSecure Use Secure Connection (optional)
    * @return boolean
    */
    function open($sHost = "", $nPort = 0, $sUserDN = "", $sPassword = "", $bSecure = false) {

      if(empty($sHost)) {
        trigger_error("Host is missing", E_USER_WARNING);
        return false;
      }

      // open connection
      $this->open_impl($sHost, $nPort);

      if(!$this->connection_exists()) {
        trigger_error("Failed connecting to LDAP Server ( Hostname: ".$this->sHost." )", E_USER_WARNING);
        return false;
      }

      // if bind with username & password
      if(!empty($sUserDN)) {
        if(!$this->bind($sUserDN, $sPassword)) {
          trigger_error("Bind with DN $sUserDN failed!", E_USER_WARNING);
          return false;
        }
      }

      return true;
    }

    /**
    * Bind using an open connection
    * <code>
    * $ldap = new LDAP('yourhost.domain.com');
    *   if(<b>$ldap->bind('uid=andreas.andersson, ou=People, dc=domain,dc=com', 'secret', true)</b>) {
    *     echo 'Authenticated!';
    *   }
    * $ldap->close();
    * </code>
    * @param string $sUserDn User DN
    * @param string $sPassword
    * @param boolean $bSecure True/False for secure connection using TLS (optional)
    * @return boolean
    */
    function bind($sUserDn = "", $sPassword = "", $bSecure = false) {
      if(empty($sUserDn)) {
        trigger_error("User DN is missing", E_USER_WARNING);
        return false;
      }

      if(!$this->connection_exists()) {
        trigger_error("Unable to connect to LDAP server", E_USER_WARNING);
        return false;
      }

      return $this->bind_impl($sUserDn, $sPassword, $bSecure);
    }

    /**
    * Closes connection to server
    * @return boolean
    */
    function close() {

      // think its better to return true in this case as
      // someone might think we have an open connection
      if(!$this->connection_exists()) {
        trigger_error("Unable to connect to LDAP server", E_USER_WARNING);
        return true;
      }

      if(!$this->close_impl()) {
        return false;
      }

      return true;
    }

    /**
    * Set LDAP Option
    * @param string $sLDAPOption
    * @param string $sLDAPOptionValue
    */
    function setLDAPOption($sLDAPOption, $sLDAPOptionValue) {
      ldap_set_option($this->hConn, $sLDAPOption, $sLDAPOptionValue);
      return;
    }

    /**
    * Authenticate Method
    *   Performs the following tasks:
    *   * Get the full DN for the user
    *   * Performs a bind
    *   * Returns true on successfull bind
    * @param string $sUsername id
    * @param string $sPassword
    * @param string $sFilter Optional, Default (uid=?)
    * @param string $bSecure Optional, Use Secure Authentication Default false
    * @return boolean
    */
    function authenticate($sUsername = "", $sPassword = "", $sLdapFilter = "(uid=?)", $bSecure = false) {

      // get DN for user
      $sDn = $this->find_user($sUsername, $sLdapFilter);

      // user does not exist
      if(empty($sDn)) {
        return false;
      }

      // simple binding
      if(!$this->bind($sDn, $sPassword, $bSecure)) {
        return false;
      }

      return true;
    }


    /**
    * Find User By Id
    *   Requires an open connection
    * @param string $sUsername Id
    * @param string $sLdapFilter LDAP Filter. Default is (uid=?)
    * @return string User DN, Empty if no user is found
    */
    function find_user($sUsername, $sLdapFilter = "(uid=?)") {
      $sUserDn = "";

      if(empty($sUsername)) {
        trigger_error("Username required!", E_USER_WARNING);
        return false;
      }

      // replace ? with username in filter
      $sLdapFilter = str_replace("?", $sUsername, $sLdapFilter);

      // sub search for user
      $rsQuery = $this->query($sLdapFilter, array("dn")) 
                 or trigger_error("Unable to fetch result from db!", E_USER_ERROR);

      if($this->num_rows($rsQuery) == 0) {
        return "";
      }

      $aResult = $this->get_result($rsQuery);
      $sUserDn = $aResult[0]["dn"];
      $this->free_result($rsQuery);

      return $sUserDn;
   }

    /**
    * Query sub levels. Will search all sub trees.
    * Calls: <b>query_handle</b>
    * @param string $sQuery Your Query
    * @param array $aAttributes LDAP attributes (optional)
    * @return boolean
    */
    function query ($sQuery, $aAttributes = array()) {
      if(empty($sQuery)) {
        trigger_error("Query string is missing", E_USER_WARNING);
        return false;
      }
      return $this->query_handle ($sQuery, "sub", $aAttributes);
    }

    /**
    * Query one level
    * Calls: <b>query_handle</b>
    * @param string $sQuery Your query
    * @param array $aAttributes LDAP attributes (optional)
    * @return boolean
    */
    function query_one_level ($sQuery, $aAttributes = array()) {
      if(empty($sQuery)) {
        trigger_error("Query string is missing", E_USER_WARNING);
        return false;
      }
      return $this->query_handle ($sQuery, "one", $aAttributes);
    }

    /**
    * Get a single entry using full DN
    * Returns resource to one entry
    * @param string $sDN Full DN to existing entry
    * @param array $aAttributes LDAP returning attributes (optional)
    * @param string $sFilter If you want to use a special filter (optional)
    * @return resource to Entry or NULL
    */
    function get_entry($sDN, $aAttributes = array(), $sFilter = "(objectclass=*)") {
      return $this->get_entry_impl($sDN, $aAttributes, $sFilter);
    }

    /**
    * If you want to set or change the base dn
    * <code>setBaseDn('dc=domain,dc=com');</code>
    * @param string $sBaseDn LDAP Base DN
    * @return void
    */    
    function set_base_dn($sBaseDn) {
      $this->set_base_dn_impl($sBaseDn);
    }

    /**
    * Return nr of rows in resultset
    * @param object $hResult Resultset from search
    * @return integer
    */
    function num_rows($hResult) {

      // returns 0 if invalid resultset
      if(!$hResult) {
        return 0;
      }
      return $this->num_rows_impl($hResult);
    }

    /**
    * Returns array of data
    * You can access the data using:
    * <code>
    * $aResult = $ldap->get_result($rsQuery);
    * echo $aResult[0]["cn"][0];
    * </code>
    * @param object $l_hResult Resultset from search
    * @return array
    */
    function get_result($hResult) {
      if(!$hResult) {
        trigger_error($this->RESULT_IS_NULL, E_USER_WARNING);
        return array();
      }
      return $this->get_result_impl($hResult);
    }

    /**
    * Free resultset
    * @param object $hResult Resultset from search
    * @return void
    */
    function free_result($hResult) {
      if(!$hResult) {
        trigger_error($this->RESULT_IS_NULL, E_USER_WARNING);
        return;
      }
      $this->free_result_impl($hResult);
    }

    /**
    * Get first entry in resultset
    * @param object $hResult Resultset
    * @return resource
    */
    function get_first_entry($hResult) {

      if(!$hResult) {
        trigger_error($this->RESULT_IS_NULL, E_USER_WARNING);
        return;
      }
      return $this->get_first_entry_impl($hResult);
    }


    /**
    * Get binary attribute value
    * <code>
    * $hEntry = $ldap->get_first_entry($hResult);
    * $binaryValue = $ldap->get_value_binary($hEntry, "jpegPhoto");
    * </code>
    * @param object $hEntry LDAP Entry
    * @param string $sAttribute
    * @return binary
    */
    function get_value_binary($hEntry, $sAttribute = "") {

      if(empty($sAttribute)) {
        trigger_error("Missing attribute", E_USER_WARNING);
        return;
      }

      if(!$hEntry) {
        trigger_error($this->RESULT_IS_NULL, E_USER_WARNING);
        return;
      }
      return $this->get_value_binary_impl($hEntry, $sAttribute);
    }

    /**
    * Get attributenames
    * Returns array
    * <code>
    * $aAttributes = $this->get_attributes($hEntry);
    * echo $aAttributes[$i];
    * </code>
    * @param object $hEntry Entry handler
    * @return array Returns an empty array on failure
    */
    function get_attributes($hEntry) {

      if(!$hEntry) {
        trigger_error($this->RESULT_IS_NULL, E_USER_WARNING);
        return array();
      }

      return $this->get_attributes_impl($hEntry);
    }

    /**
    * Translate characters from UTF
    * @param string Unconverted string
    * @return string
    */
    function convert_characters($sData) {
      return $this->convert_characters_impl($sData);
    }

    /**
    * Return last used search query
    * @return string
    */
    function get_last_search_query() {
      return $this->get_last_search_query_impl();
    }

    /**
    * Sort resultdata.
    * @param object $hResult Resultset from search
    * @param string $sSortAttribute Attribute to sort on
    * @return void
    */
    function sort($hResult, $sSortAttribute) {
      if(!$hResult) {
        trigger_error($this->RESULT_IS_NULL, E_USER_WARNING);
        return;
      }
      $this->sort_impl($hResult, $sSortAttribute);
    }

    /**
    * Log LDAP Operations
    * @param string $sLogFile
    * @return void
    */
    function setLogFile($sLogFile) {
      $this->sLogFile = $sLogFile;
    }
  } // end class LDAP
?>
