<?php
  /*
    Copyright (C) 2009-2012 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

/**
* @package cn=monitor
* @author Andreas Andersson
* @version 1.0
*
*  090425 - Andreas Andersson
*           Created
*  110527 - Andreas Andersson
*           Updated all logic in this file
*/
class Server {
  var $serverid = 0;
  var $environmentid = 0;
  var $hostname = "";
  var $vendorName = "";
  var $vendor = "";
  var $message = "";
  var $exitcode = 0;
  var $respondPort = false;
  var $respondSecurePort = false;
  var $respondLDAP = false;
  var $updated = 0;
  var $exectime = 0.00;
  var $replicaType = "";
  var $replicationAgreements = 0;
  var $replicationAgreementsOk = 0;
  
  function Server($servername) {
    if(empty($servername)) {
      $this->exitcode = 1;
      return false;
    }
    
    // set config id:s used by json sending
    $configId = getEnvironmentId($servername);
    if(!empty($configId)) {
      $this->environmentid = $configId[0];
      $this->serverid      = $configId[1];
    }
    
    $this->hostname = $servername;
    $this->init();    
  }
  
  // initialize function
  function init() {

    $startTime = microtimeFloat();

    $cache_connection = $this->getCacheConnection();

    // verify that the port responds
    $runVerification = 1;
    if(!empty($cache_connection) && ($this->respondPort == true)) {
      logToFile("ServerObject", "init - port cached", $this->hostname, 3);
      $runVerification = 0;
    }
    if($runVerification) {
      logToFile("ServerObject", "init - port verification", $this->hostname, 3);
      $this->verifyRespondPort();
    }
    if(!$this->respondPort) {
      logToFile("ServerObject", "init - port verification failed", $this->hostname, 3);
      $this->respondLDAP = false;
      $this->setExecutionTime($startTime, microtimeFloat());
      return false;
    }

    // verify ldap connection
    // if connection cache enabled we can skip this step
    if(!empty($cache_connection) && ($this->respondLDAP == true)) {
      logToFile("ServerObject", "init - ldap verification cached", $this->hostname, 3);
      $this->setExecutionTime($startTime, microtimeFloat());
      return;
    }

    // finally check respond to LDAP
    $this->verifyRespondLDAP();
    $this->setExecutionTime($startTime, microtimeFloat());    
    return;
  }

  // get cache connection status  
  function getCacheConnection() {
    $hostname = $this->getHostName();
    // if something is wrong and we don't have a hostname
    if(empty($hostname)) {
      return "";
    }
  
    // if we have a connection cache, skip verification if connected
    $configId = getEnvironmentId($hostname);
    if(empty($configId)) {
      return "";
    }
    $cache_connection = getConfigValue("cache_connection", $configId[0], $configId[1]);

    if(!empty($cache_connection)) {
      logToFile("ServerObject", "init - cache_connection is enabled", $hostname, 3);
    }
    else {
      logToFile("ServerObject", "init - cache_connection is not enabled", $hostname, 3);    
    }
    return $cache_connection;
  }

  function verifyRespondLDAP() {

    // this function requires successfull check on port
    if(!$this->respondPort) {
      return false;
    }
    $this->respondLDAP = false;

    // if we don't know the vendor, use the verifyVendor function instead
    if(($this->vendor == "unknown") || ($this->vendor == "")) {
      return $this->verifyVendor();
    }

    // verify connection
    if(verifyLDAPConnection($this)) {
      $this->respondLDAP = true;
      return true;
    }
    
    return false;
  }
  
  /***
    checks Respond LDAP + Get Vendor Name if missing
    Should only be called internally
  */
  function verifyVendor() {

    // verify LDAP Server and Set Vendor / VendorName
    if(verifyLDAPServer($this)) {
      $this->respondLDAP = true;
      $this->updated = 1;
      return true;
    }

    return false;
  }

  function verifyRespondPort() {
    $this->respondPort = false;

    // get port
    $configuredPort = getPort($this->hostname);

    if(socketOpen($this->hostname, $configuredPort)) {
      $this->respondPort = true;
    }
    return $this->respondPort;
  }

  // returns boolean
  function verifyRespondSecurePort() {

    $cache_connection = $this->getCacheConnection();

    // verify that the secure port responds
    $runVerification = 1;
    if(!empty($cache_connection) && ($this->respondSecurePort == true)) {
      logToFile("ServerObject", "verifyRespondSecurePort - secureport cached", $this->hostname, 3);
      $runVerification = 0;
    }
    if($runVerification) {
      logToFile("ServerObject", "verifyRespondSecurePort - secureport verification", $this->hostname, 3);
 
      // get port
      $configuredSecurePort = getSecurePort($this->hostname);
  
      $this->respondSecurePort = false;

      if(socketOpen($this->hostname, $configuredSecurePort)) {
        $this->respondSecurePort = true;
      }
    }
    if(!$this->respondSecurePort) {
      logToFile("ServerObject", "verifyRespondSecurePort - secureport verification failed", $this->hostname, 3);
    }    
    
    return $this->respondSecurePort;
  }
  
  // call this function if you know the server is unavailable to reset values
  function clearAvailability() {
    $this->respondPort = $this->respondSecurePort = $this->respondLDAP;
  }

  /**
  * Setters...
  */

  function setMessage($sMessage) {
    $this->message = $sMessage;
  }
  
  function setVendor($sVendor) {
    $this->vendor = $sVendor;
  }

  function setVendorName($sVendorName) {
    $this->vendorName = $sVendorName;
  }

  function setHostName($sHostName) {
    $this->hostname = $sHostName;
  }

  function setExitCode($nExitCode) {
    $this->exitcode = $nExitCode;
  }

  // clear update status
  function setClearUpdate() {
    $this->updated = 0;
  }

  function setReplicaType($inReplicaType) {
    $this->replicaType = $inReplicaType;
  }

  function setReplicationAgreements($inReplicationAgreements) {
    $this->replicationAgreements = $inReplicationAgreements;
  }

  function setReplicationAgreementsOk($inReplicationAgreementsOk) {
    $this->replicationAgreementsOk = $inReplicationAgreementsOk;
  }

  function setExecutionTime($startTime, $endTime) {
    $this->exectime = round((($endTime - $startTime) * 1000),2);
  }

  /**
  * Getters...
  */

  // returns string
  function getVendor() {
    return $this->vendor;
  }

  // returns string
  function getVendorName() {
    return $this->vendorName;
  }

  // returns string
  function getMessage() {
    return $this->message;
  }

/*
  To be removed
  // returns string
  function getAvailabilityMessage() {
    return $this->message;
  }
*/
  function getRespondPort() {
    return $this->respondPort;
  }

  function getRespondSecurePort() {
    return $this->respondSecurePort;
  }

  function getRespondLDAP() {
    return $this->respondLDAP;
  }

  // returns int exit code 0 on success
  function getExitCode() {
    return $this->exitcode;
  }

  // returns string
  function getHostName() {
    return $this->hostname;
  }
  
  // returns time to find out if LDAP responds
  function getExecutionTime() {
    return $this->exectime;
  }

  // returns boolean
  function isAvailable() {
    return $this->respondLDAP;
  }
  
  // returns boolean
  function isUpdated() {
    return $this->updated;
  }


}


?>
