<?php
  /*
    Copyright (C) 2009-2012 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

  /**********************************************************************
  * OpenLDAP Functions
  *
  * Specific LDAP functions for Directory Servers
  *
  * Available functionality:
  * (X) setMonitor
  * (X) setSnmp
  * (X) getHostNameOfServer
  * (X) setCache
  * (X) setBackEndDatabases
  * (X) setIndex
  * (X) setIndexesList
  * (X) getReplicationCSN = getMaxCSNValue
  * (X) convertCSNValue
  * (X) setReplicationAgreement
  * (X) setReplicaInfo
  * (X) setVendorVersion
  * (X) setPerformanceCounterList
  *
  **********************************************************************/

  /**
  * Returns a monitor object
  */
  function openldap_setMonitor($ldap, $currentMonitor, $operation = Array()) {

    // start time
    if(empty($operation) || in_array("start_time", $operation)) {
      $ldap->set_base_dn("cn=Time,cn=Monitor");

      $hResult = @$ldap->query("cn=Start", array("monitorTimestamp"));

      if(!$hResult) {
        return $currentMonitor;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentMonitor->setStartTime($aEntries[0]["monitortimestamp"][0]);
      $ldap->free_result($hResult);
    }

    // current time
    if(empty($operation) || in_array("cur_time", $operation)) {
      $ldap->set_base_dn("cn=Time,cn=Monitor");

      $hResult = @$ldap->query("cn=Current", array("monitorTimestamp"));

      if(!$hResult) {
        return $currentMonitor;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentMonitor->setCurrentTime($aEntries[0]["monitortimestamp"][0]);
      $ldap->free_result($hResult);
    }

    // total connections
    if(empty($operation) || in_array("tot_con", $operation)) {
      $ldap->set_base_dn("cn=Connections,cn=Monitor");

      $hResult = @$ldap->query("cn=Total", array("monitorCounter"));

      if(!$hResult) {
        return $currentMonitor;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentMonitor->setTotalConnections($aEntries[0]["monitorcounter"][0]);
      $ldap->free_result($hResult);
    }

    // current connections
    if(empty($operation) || in_array("cur_con", $operation)) {
      $ldap->set_base_dn("cn=Connections,cn=Monitor");

      $hResult = @$ldap->query("cn=Current", array("monitorCounter"));

      if(!$hResult) {
        return $currentMonitor;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentMonitor->setCurrentConnections($aEntries[0]["monitorcounter"][0]);
      $ldap->free_result($hResult);
    }

    // thread
    if(empty($operation) || in_array("thread", $operation)) {
      $ldap->set_base_dn("cn=Threads,cn=Monitor");

      $hResult = @$ldap->query("cn=Max", array("monitoredInfo"));

      if(!$hResult) {
        return $currentMonitor;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentMonitor->setThreads($aEntries[0]["monitoredinfo"][0]);
      $ldap->free_result($hResult);
    }
        
    // entries sent
    if(empty($operation) || in_array("ent_sent", $operation)) {
      $ldap->set_base_dn("cn=Statistics,cn=Monitor");

      $hResult = @$ldap->query("cn=Entries", array("monitorCounter"));

      if(!$hResult) {
        return $currentMonitor;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentMonitor->setEntriesSent($aEntries[0]["monitorcounter"][0]);
      $ldap->free_result($hResult);
    }

    // bytes in various forms sent
    if(empty($operation) || in_array("mbyte_sent", $operation)
                         || in_array("kbyte_sent", $operation)
                         || in_array("byte_sent", $operation)) {
      $ldap->set_base_dn("cn=Statistics,cn=Monitor");

      $hResult = @$ldap->query("cn=Bytes", array("monitorCounter"));

      if(!$hResult) {
        return $currentMonitor;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentMonitor->setBytesSent($aEntries[0]["monitorcounter"][0]);
      $ldap->free_result($hResult);
    }

    return $currentMonitor;
  }

  /**
  * Returns a snmp object
  */
  function openldap_setSnmp($ldap, $currentSnmp, $operation = Array()) {

    $ldap->set_base_dn("cn=Operations,cn=Monitor");

    // Search Operations
    if(empty($operation) || in_array("search", $operation)) {
      $hResult = @$ldap->query("cn=Search", array("monitorOpCompleted"));
          
      if(!$hResult) {
        return $currentSnmp;
      }
        
      // Search Operations
      $aEntries = $ldap->get_result($hResult);
      $currentSnmp->setSearchOps($aEntries[0]["monitoropcompleted"][0]);
      $ldap->free_result($hResult);
    }

    // IN Operations
    if(empty($operation) || in_array("in", $operation)) {
      $hResult = @$ldap->query("cn=Operations", array("monitorOpCompleted"));

      if(!$hResult) {
        return $currentSnmp;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentSnmp->setInOps($aEntries[0]["monitoropcompleted"][0]);
      $ldap->free_result($hResult);
    }
    
    
    // Binds
    // Unfortunately OpenLDAP doesn't keep statistics for SASL / Anonymous and Simple
    // I will save this value as Simple.
    if(empty($operation) || in_array("simple_bind", $operation)) {
      $hResult = @$ldap->query("cn=Bind", array("monitorOpCompleted"));

      if(!$hResult) {
        return $currentSnmp;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentSnmp->setSimpleAuthBinds($aEntries[0]["monitoropcompleted"][0]);
      $ldap->free_result($hResult);
    }

    // Add Operations
    if(empty($operation) || in_array("add", $operation)) {
      $hResult = @$ldap->query("cn=Add", array("monitorOpCompleted"));

      if(!$hResult) {
        return $currentSnmp;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentSnmp->setAddEntryOps($aEntries[0]["monitoropcompleted"][0]);
      $ldap->free_result($hResult);
    }

    // Modify Operations
    if(empty($operation) || in_array("modify", $operation)) {
        $hResult = @$ldap->query("cn=Modify", array("monitorOpCompleted"));

      if(!$hResult) {
        return $currentSnmp;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentSnmp->setModifyEntryOps($aEntries[0]["monitoropcompleted"][0]);
      $ldap->free_result($hResult);
    }

    // ModifyRDN Operations
    if(empty($operation) || in_array("modifyrdn", $operation)) {
      $hResult = @$ldap->query("cn=Modrdn", array("monitorOpCompleted"));

      if(!$hResult) {
        return $currentSnmp;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentSnmp->setModifyRdnOps($aEntries[0]["monitoropcompleted"][0]);
      $ldap->free_result($hResult);
    }

    // Compare Operations
    if(empty($operation) || in_array("compare", $operation)) {
      $hResult = @$ldap->query("cn=Compare", array("monitorOpCompleted"));

      if(!$hResult) {
        return $currentSnmp;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentSnmp->setCompareOps($aEntries[0]["monitoropcompleted"][0]);
      $ldap->free_result($hResult);
    }
    // Remove Operations
    if(empty($operation) || in_array("remove", $operation)) {
      $hResult = @$ldap->query("cn=Delete", array("monitorOpCompleted"));

      if(!$hResult) {
        return $currentSnmp;
      }

      $aEntries = $ldap->get_result($hResult);
      $currentSnmp->setRemoveEntryOps($aEntries[0]["monitoropcompleted"][0]);
      $ldap->free_result($hResult);
    }
      
    //print_r($currentMonitor);

    return $currentSnmp;
  }

  function openldap_getHostNameOfServer($ldap) {
    $sReturn = "";
    $hResult = @$ldap->get_entry("cn=config", array("olcSaslHost"));

    if(!$hResult) {
      return $sReturn;
    }

    $nRows = $ldap->num_rows($hResult);
    if($nRows == 1) {
      $aEntries = $ldap->get_result($hResult);
      $sReturn = $aEntries[0]["olcsaslhost"][0];
    }
    $ldap->free_result($hResult);

    return $sReturn;
  }

  function openldap_setCache($ldap, $oCache) {
    $ldap->set_base_dn("cn=config");
    $hResult = @$ldap->query("objectclass=olcBdbConfig", 
                             array("olcDbConfig", "olcDbCacheSize", "olcSuffix", "olcdbdncachesize", "olcdbidlcachesize")); 
  
    if(!$hResult) {
      $oCache->setMessage("Unable to query server", "general");
      return $oCache;
    }

    $oEntries = $ldap->get_result($hResult);

    for($i = 0; $i < (count($oEntries) -1); $i++) {
      //if empty, no need to get cache sizes
      if(empty($oEntries[$i]["olcsuffix"][0])) {
        continue;
      }
      $backEnd = $oEntries[$i]["olcsuffix"][0];
      $oCache->setBackendDb($backEnd);

      if(!empty($oEntries[$i]["olcdbcachesize"][0])) {
        $oCache->setMaxEntriesCache($oEntries[$i]["olcdbcachesize"][0], $backEnd);                
      }

      // DN Cache
      if(!empty($oEntries[$i]["olcdbdncachesize"][0])) {
        $oCache->setDNMaxCache($oEntries[$i]["olcdbdncachesize"][0], $backEnd);                
      }

      // Index Cache
      if(!empty($oEntries[$i]["olcdbidlcachesize"][0])) {
        $oCache->setIndexMaxCache($oEntries[$i]["olcdbidlcachesize"][0], $backEnd);                
      }

      if(!empty($oEntries[$i]["olcdbconfig"][0])) {
        for($j = 0; $j < count($oEntries[$i]["olcdbconfig"]); $j++) {
          if(empty($oEntries[$i]["olcdbconfig"][$j])) {
            continue;
          }
          if(strpos($oEntries[$i]["olcdbconfig"][$j], "set_cachesize") !== FALSE) {
            $aCacheSize = explode(" ", $oEntries[$i]["olcdbconfig"][$j]);
            $oCache->setMaxCache(($aCacheSize[1] * 1073741824) + $aCacheSize[2], $backEnd);            
          } // found
        } // for each value
      } // if olcdbconfig found
            
    } // for each cache config

    $ldap->free_result($hResult);

    // now we need to check the current cache size for this backend
    $ldap->set_base_dn("cn=Databases,cn=Monitor");
    $hResultCurrent = @$ldap->query("(&(objectclass=monitoredObject)(olmBDBEntryCache=*))", 
                              array("namingContexts", "olmBDBEntryCache", "olmBDBDNCache", "olmBDBIDLCache")); 

    $oEntriesCurrent = $ldap->get_result($hResultCurrent);

    for($i = 0; $i < (count($oEntriesCurrent) -1); $i++) {
      if(empty($oEntriesCurrent[$i]["namingcontexts"][0])) {
        continue;      
      }
      $backEnd = $oEntriesCurrent[$i]["namingcontexts"][0];

      if(!empty($oEntriesCurrent[$i]["olmbdbentrycache"][0])) {
        $oCache->setEntriesInCache($oEntriesCurrent[$i]["olmbdbentrycache"][0], $backEnd);      
      }

      // DN Cache
      if(!empty($oEntriesCurrent[$i]["olmbdbdncache"][0])) {
        $oCache->setDNCurrentCache($oEntriesCurrent[$i]["olmbdbdncache"][0], $backEnd);      
      }

      // Index Cache
      if(!empty($oEntriesCurrent[$i]["olmbdbidlcache"][0])) {
        $oCache->setIndexCurrentCache($oEntriesCurrent[$i]["olmbdbidlcache"][0], $backEnd);      
      }

    } // for each found entry

    $ldap->free_result($hResultCurrent);

    return $oCache;  
  }

  /**
  * OpenLDAP Back end database list
  * Returns a List of available backends
  */
  function openldap_setBackEndDatabases($ldap) {

    // Returns a list in the format:
    // $returnList[0][0] = CN
    // $returnList[0][1] = Back End Name
    $returnList = Array();

    $ldap->set_base_dn("cn=config");

    // @ to hide err msg:es
    // Search for both Bdb and Hdb database types
    $hResult = @$ldap->query("(|(objectclass=olcBdbConfig)(objectclass=olcHdbConfig))", array("olcdatabase", "olcsuffix"));

    if(!$hResult) {
      return $returnList;
    }

    $oEntries = $ldap->get_result($hResult);

    for($i = 0; $i < (count($oEntries) -1); $i++) {
      if(!empty($oEntries[$i]["olcsuffix"][0])) {
        $counter = count($returnList);
        $returnList[$counter][0] = $oEntries[$i]["olcdatabase"][0];
        $returnList[$counter][1] = $oEntries[$i]["olcsuffix"][0];
      }
    }

    $ldap->free_result($hResult);
    return $returnList;  
  }

  /**
  * OpenLDAP Index Object
  * Returns a index object
  */
  function openldap_setIndex($ldap, $servername, $oReturn, $backenddatabase, $queryindex) {
    $ldap->set_base_dn("cn=config");
    $hResult = @$ldap->get_entry("olcDatabase=".$backenddatabase.",cn=config", array("olcDbIndex"));

    if(!$hResult) {
      return $oReturn;
    }

    $oEntries = $ldap->get_result($hResult);

    // loop to find correct index
    for($i = 0; $i < (count($oEntries[0]["olcdbindex"]) -1); $i++) {
      if(!empty($oEntries[0]["olcdbindex"][$i])) {
        $currentList = explode(" ", $oEntries[0]["olcdbindex"][$i]);
        if($currentList[0] == $queryindex) {
          $indexList = explode(",", $currentList[1]);
          for($j = 0; $j < count($indexList); $j++) {
            $oReturn->setIndexType(trim($indexList[$j]));
            $oReturn->setIndexTypeByHost($servername, trim($indexList[$j]));
          } // for each index type
        } // found index
      }
    }

    $ldap->free_result($hResult);
    return $oReturn;  
  }

  /**
  * OpenLDAP Back end index list
  * Returns a List of available indexes for specific backend
  */
  function openldap_setIndexesList($ldap, $backenddatabase) {

    // Returns a list in the format:
    // $returnList[0] = index
    $returnList = Array();

    $ldap->set_base_dn("cn=config");
    $hResult = @$ldap->get_entry("olcDatabase=".$backenddatabase.",cn=config", array("olcDbIndex"));

    if(!$hResult) {
      return $returnList;
    }

    $oEntries = $ldap->get_result($hResult);

    for($i = 0; $i < (count($oEntries[0]["olcdbindex"]) -1); $i++) {
      if(!empty($oEntries[0]["olcdbindex"][$i])) {
        $currentList = explode(" ", $oEntries[0]["olcdbindex"][$i]);
        $returnList[count($returnList)] = $currentList[0];
      }
    }

    $ldap->free_result($hResult);
    return $returnList;  
  }

  function openldap_getMaxCSNValue($csnArray) {

    // array[0] = time, array[1] = raw format
    $csnReturnValue = array("", "");
    $maxCSNValue = 0;
    
    for($i = 0; $i < count($csnArray); $i++) {
      if(!empty($csnArray[$i])) {
        $aDate = explode("#", $csnArray[$i]);
        $currentCSNValue = strtotime(openldap_convertCSNValue($aDate[0]));
        if($currentCSNValue > $maxCSNValue) {
          $maxCSNValue = $currentCSNValue;
          $csnReturnValue[0] = openldap_convertCSNValue($aDate[0]);
          $csnReturnValue[1] = $csnArray[$i];
        }
      }
    }
    return $csnReturnValue;
  }

  /**
  * Returns a Replication agreements Objects
  */
  function openldap_setReplicationAgreement($oServerReplica, $servername, $csnCheck = false) {
    global $CONFIG;
    // a bit different here compared to iplanet.
    // the idea is to get all hosts within the environment, exclude $servername and build up
    // replication agreements based on CSN info

    // we need to get the backend db:s
    // not common that we query from vendor specific functions back to ldap functions
    // in this case its necessary as OpenLDAP needs backend databases
    $rootDN = setRootDN($servername, array("namingcontexts"));
    $replicaRoots = $rootDN->getNamingContexts();
    $oServerReplica->setBackendDb($replicaRoots);

    $configId = getEnvironmentId($servername);
    $serverlist = getServers($configId[0]);

    for($i = 0; $i < count($serverlist); $i++) {
      $currentserver = $serverlist[$i][1];

      // we need to check server availability
      $oServer = setServer($currentserver);
      if(!$oServer->isAvailable()) {
        continue;
      }

      $ldap = createConnection($currentserver);
      if(!$ldap) {
        $oServerReplica->setMessage("Failed retrieving replication agreements");
      }
      
      for($j = 0; $j < count($replicaRoots); $j++) {
        $hAgreement = @$ldap->get_entry($replicaRoots[$j], array("contextCSN"));
        if($hAgreement) {
          $agreementEntry = $ldap->get_result($hAgreement);

          // if this is the master (or the server we view) the contextcsn is set as master
          if($currentserver == $servername) {
            if(!empty($agreementEntry[0]["contextcsn"][0])) {
              $csnValue = openldap_getMaxCSNValue($agreementEntry[0]["contextcsn"]);
              $oServerReplica->setMasterCSN(
                                            $replicaRoots[$j], 
                                            $csnValue[0]);
            } // contextCSN for master
            continue;
          }

          // creating new agreement
          $oAgreement = new ReplicaAgreement();
          $oAgreement->setHostName($currentserver);
          if(!empty($agreementEntry[0]["contextcsn"][0])) {
            $csnValue = openldap_getMaxCSNValue($agreementEntry[0]["contextcsn"]);
            $oAgreement->setCSN($csnValue[0]);
            $oAgreement->setMessage("Based on ContextCSN");
            $csnList = "";
            for($k = 0; $k < count($agreementEntry[0]["contextcsn"]); $k++) {
              if(!empty($agreementEntry[0]["contextcsn"][$k])) {
                $csnList .= $agreementEntry[0]["contextcsn"][$k]."\n";
              }
            }

            $oAgreement->setLastStatusRaw($csnList);
            $oAgreement->setReplicaRoot($replicaRoots[$j]);            
          }

          $ldap->free_result($hAgreement);
          
          $oServerReplica->setAgreement($oAgreement);
        } // agreement exist

      } // for each replica root

      if($currentserver != $servername) {
        $oServerReplica->setServerlist($currentserver);
      }
      closeConnection($ldap);
    } // for each server
    
    // before exiting we should run through all agreements and check if we have any issues
    $agreements = $oServerReplica->getAgreements();
    for($i = 0; $i < count($agreements); $i++) {
      $replicaCSN = $agreements[$i]->getCSN();
      $replicaRoot = $agreements[$i]->getReplicaRoot();
      $masterCSN = $oServerReplica->getMasterCSN($replicaRoot);
      $replicaCSNTime = strtotime($replicaCSN);
      $masterCSNTime = strtotime($masterCSN);
      // max acceptable delay is 60s
      // if beyond this value between master and client error will be triggered
      if(($masterCSNTime - $replicaCSNTime) > 60) {
        // one minute delay
        $oServerReplica->agreements[$i]->setReplicationError(true);
        $oServerReplica->setAgreementsOk(false);
        $oServerReplica->setExitCode(1);
        $oServerReplica->agreements[$i]->setMessage("Based on ContextCSN with max delay of 60s");
      }
      else {
        $oServerReplica->setAgreementsOk(true);
      }
    }
    return $oServerReplica;
  }

  function openldap_convertCSNValue($csn) {
    if(empty($csn)) {
      return "";
    }

    $newDate = substr($csn, 0, 4);
    $newDate .= "-".substr($csn, 4, 2);
    $newDate .= "-".substr($csn, 6, 2);
    $newDate .= " ".substr($csn, 8, 2);
    $newDate .= ":".substr($csn, 10, 2);
    $newDate .= ":".substr($csn, 12, 2);

    return $newDate;
  }

  /**
  * Returns a Replica info
  * Showing nr of changes handled by replica, type of replica
  * Supplier parameter used for possibility to disconnect verification
  */
  function openldap_setReplicaInfo($ldap, $oReplicaInfo) {
    $oReplicaInfo->setIsIplanet(false);
    
    // as we won't know what type of replica we are dealing with. set it to unknown
    $configId = getEnvironmentId($oReplicaInfo->getHostName());
    $replicatype = getConfigValue("replicatype", $configId[0], $configId[1]);
    $oReplicaInfo->setReplicaTypeByName("unknown");
    if(!empty($replicatype)) {
      $oReplicaInfo->setReplicaTypeByName("master");
    }

    $rootDN = setRootDN($oReplicaInfo->getHostName(), array("namingcontexts"));
    $replicaRoots = $rootDN->getNamingContexts();

    if(empty($replicaRoots)) {
      $oReplicaInfo->setMessage("Unable to fetch result");
      return $oReplicaInfo;
    }

    for($i = 0; $i < count($replicaRoots); $i++) {
      $oReplicaInfo->setReplicaRoot($replicaRoots[$i]);

      // for clients that has cn=config configured we can also try to fetch
      // replica database ID, this is optional as we try to read this from cn=config
      $ldap->set_base_dn("cn=config");
      $hResult = @$ldap->query("(&(objectclass=olcDatabaseConfig)(olcSuffix=".$replicaRoots[$i].")(olcSyncrepl=*))", array("olcSyncrepl"));
      if(!empty($hResult)) {
        $nRows = $ldap->num_rows($hResult);
        if($nRows == 1) {
          $oEntries = $ldap->get_result($hResult);
          if(!empty($oEntries[0]["olcsyncrepl"])) {
             $nStartPos = (strpos($oEntries[0]["olcsyncrepl"][0], "rid=") + 4);
             $nEndPos   = strpos($oEntries[0]["olcsyncrepl"][0], " ", $nStartPos);
             $nVal = substr($oEntries[0]["olcsyncrepl"][0], $nStartPos, ($nEndPos - $nStartPos));
             if(!empty($nVal)) {
               $oReplicaInfo->setReplicaId($nVal, $replicaRoots[$i]);
             }
          }
        }
        $ldap->free_result($hResult);
      }
    }    
    
    return $oReplicaInfo;
  }

  /**
  * Open LDAP Returns Vendor and Version
  */
  function openldap_setVendorInfo($ldap, $oVendorInfo) {
    $ldap->set_base_dn("cn=monitor");
    $hResult = @$ldap->get_entry("cn=monitor", array("monitoredInfo", "+"));

    if($hResult) {
      $aEntries = $ldap->get_result($hResult);
      if(isset($aEntries[0]["monitoredinfo"][0])) {
        $oVendorInfo->setVendorVersion($aEntries[0]["monitoredinfo"][0]);
      }
      $ldap->free_result($hResult);
    }  
    return $oVendorInfo;
  }

  /**
  * Open LDAP Returns available operations
  */
  function openldap_setPerformanceCounterList($byte_operation) {
    return array("ent_sent", $byte_operation, "tot_con", "exec_time", "search", "cur_con",
                 "add", "modify", "modifyrdn", "compare", "remove", "in",
                 "simple_bind");
  }
?>