<?php
  /*
    Copyright (C) 2009-2012 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

  /**********************************************************************
  * OpenDS Functions (OpenDS, OpenDJ etc ...)
  *
  * Specific LDAP functions for Directory Servers
  *
  * Available functionality:
  * (X) setMonitor
  * (X) setSnmp
  * (X) getHostNameOfServer
  * (X) setCache (limited, only entries in cache, hits, ratio)
  * (0) setBackEndDatabases
  * (X) setIndex
  * (X) setIndexesList
  * (0) getReplicationCSN
  * (0) convertCSNValue
  * (X) setReplicationAgreement
  * (0) setReplicaInfo
  * (0) setVendorInfo (not needed)
  * (X) setPerformanceCounterList
  *
  **********************************************************************/

  /**
  * Returns a monitor object
  */
  function opends_setMonitor($ldap, $currentMonitor, $operation = Array()) {

      $ldap->set_base_dn("cn=monitor");

      // Step 1 get basic monitoring information form server
      $hResult = @$ldap->get_entry("cn=monitor");

      if(!$hResult) {
        $sMessage = "Unable to query server";
        $currentMonitor->setMessage($sMessage);
        return $currentMonitor;
      }
      $nRows = $ldap->num_rows($hResult);
 
      if($nRows > 0) {
        $aEntries = $ldap->get_result($hResult);
        if(!empty($aEntries[0]["vendorversion"])) {
          $currentMonitor->setVersion($aEntries[0]["vendorversion"][0]);
        }

        // as connection peak doesn't exist in redhat/fedora
        if(!empty($aEntries[0]["maxconnections"])) {
          $currentMonitor->setConnectionPeak($aEntries[0]["maxconnections"][0]);
        }
        if(!empty($aEntries[0]["currentconnections"])) {
          $currentMonitor->setCurrentConnections($aEntries[0]["currentconnections"][0]);
        }
        if(!empty($aEntries[0]["totalconnections"])) {
          $currentMonitor->setTotalConnections($aEntries[0]["totalconnections"][0]);
        }
        if(!empty($aEntries[0]["starttime"])) {
          $currentMonitor->setStartTime($aEntries[0]["starttime"][0]);
        }
        if(!empty($aEntries[0]["currenttime"])) {
          $currentMonitor->setCurrentTime($aEntries[0]["currenttime"][0]);
        }
      }
      else {
        $currentMonitor->setMessage("Unable to fetch result");
      }
    $ldap->free_result($hResult);

    // Step 2: Get operation read entries and bytes statistics from all ports
    $returnAttributes = array("searchresultentries", "byteswritten");
    $hResult = @$ldap->query("(&(searchResultEntries=*)(objectclass=ds-monitor-entry))", $returnAttributes);
    if(!$hResult) {
      $currentSnmp->setMessage("Unable to open ldap connection");
      return $currentSnmp;
    }

    $entriesSent = 0;
    $bytesSent = 0;
    $oEntries = $ldap->get_result($hResult);
    // for each port / instance
    for($i = 0; $i < (count($oEntries) - 1); $i++) {
      if(!empty($oEntries[$i]["searchresultentries"])) {
        $entriesSent += $oEntries[$i]["searchresultentries"][0];
      }
      if(!empty($oEntries[$i]["byteswritten"])) {
        $bytesSent += $oEntries[$i]["byteswritten"][0];
      }
    }    
    $ldap->free_result($hResult);

    $currentMonitor->setEntriesSent($entriesSent);
    $currentMonitor->setBytesSent($bytesSent);

    return $currentMonitor;
  }


  function opends_setSnmp($ldap, $currentSnmp, $operation = Array()) {

    $ldap->set_base_dn("cn=monitor");

    $returnAttributes = array("bindrequests", "searchrequests", "comparerequests",
                              "operationsinitiated", "addrequests", "modifyrequests",
                              "modifydnrequests", "deleterequests");
    // cn cnmp (@ to hide ev. error message)
    $hResult = @$ldap->query("(&(searchResultEntries=*)(objectclass=ds-monitor-entry))", $returnAttributes);
    if(!$hResult) {
      $currentSnmp->setMessage("Unable to open ldap connection");
      return $currentSnmp;
    }

    $oEntries = $ldap->get_result($hResult);
    $bindAnonymous = 0; // do not exist
    $bindUnAuth = 0; // do not exist
    $bindSimple = 0; // include all bind operations
    $bindStrong = 0; // do not exist
    $bindSecurityErrors = 0; // do not exist
    $searchWhole = 0; // do not exist
    $searchRequests = 0;
    $compareOps = 0;
    $errors = 0; // do not exist
    $inops = 0;
    $addops = 0;
    $modifyops = 0;
    $modifyrdnops = 0;
    $removeentryops = 0;
    $securityErrors = 0;

    for($i = 0; $i < (count($oEntries) - 1); $i++) {
      // set any bind as simple bind operation
      if(!empty($oEntries[$i]["bindrequests"])) {
        $bindSimple += $oEntries[$i]["bindrequests"][0];
      }

      // compare operations
      if(!empty($oEntries[$i]["comparerequests"])) {
        $compareOps += $oEntries[$i]["comparerequests"][0];
      }

      // operations initiated
      if(!empty($oEntries[$i]["operationsinitiated"])) {
        $inops += $oEntries[$i]["operationsinitiated"][0];
      }

      // add operations
      if(!empty($oEntries[$i]["addrequests"])) {
        $addops += $oEntries[$i]["addrequests"][0];
      }

      // modify operations
      if(!empty($oEntries[$i]["modifyrequests"])) {
        $modifyops += $oEntries[$i]["modifyrequests"][0];
      }

      // modifyrdn operations
      if(!empty($oEntries[$i]["modifydnrequests"])) {
        $modifyrdnops += $oEntries[$i]["modifydnrequests"][0];
      }

      // remove operations
      if(!empty($oEntries[$i]["deleterequests"])) {
        $removeentryops += $oEntries[$i]["deleterequests"][0];
      }

      // search operations
      if(!empty($oEntries[$i]["searchrequests"])) {
        $searchRequests += $oEntries[$i]["searchrequests"][0];
      }
    }
    $ldap->free_result($hResult);

    $currentSnmp->setAnonymousBinds($bindAnonymous);
    $currentSnmp->setUnAuthenticatedBinds($bindUnAuth);
    $currentSnmp->setSimpleAuthBinds($bindSimple);
    $currentSnmp->setStrongAuthBinds($bindStrong);
    $currentSnmp->setWholeSubtreeSearchOps($searchWhole);
    $currentSnmp->setBindSecurityErrors($bindSecurityErrors);
    $currentSnmp->setCompareOps($compareOps);
    $currentSnmp->setErrors($errors);
    $currentSnmp->setInOps($inops);
    $currentSnmp->setAddEntryOps($addops);
    $currentSnmp->setModifyEntryOps($modifyops);
    $currentSnmp->setModifyRdnOps($modifyrdnops);
    $currentSnmp->setRemoveEntryOps($removeentryops);
    $currentSnmp->setSearchOps($searchRequests);
    $currentSnmp->setSecurityErrors($securityErrors);

    return $currentSnmp;
  }

  function opends_getHostNameOfServer($ldap) {
    $sReturn = "";

    $hResult = @$ldap->get_entry("cn=System Information,cn=monitor", 
                                 array("systemName"));

    if(!$hResult) {
      return $sReturn;
    }
    $nRows = $ldap->num_rows($hResult);

    if($nRows == 1) {
      $aEntries = $ldap->get_result($hResult);
      if(!empty($aEntries[0]["systemname"])) {
        $sReturn = $aEntries[0]["systemname"][0];
      }
    }
    $ldap->free_result($hResult);

    return $sReturn;
  }

  function opends_setCache($ldap, $oCache) {

    $ldap->set_base_dn("cn=monitor");

    // @ to hide err msg:es
    $hResult = @$ldap->get_entry("cn=Entry Caches,cn=monitor");

    if(!$hResult) {
      $oCache->setMessage("Unable to query server", "general");
      return $oCache;
    }

    $oEntries = $ldap->get_result($hResult);
    $nRows = $ldap->num_rows($hResult);
    if($nRows == 1) {
      $backEnd = "General";
      $oCache->setBackendDb($backEnd);    
      if(isset($oEntries[0]["entrycachehits"])) {
        $oCache->setCacheHits($oEntries[0]["entrycachehits"][0], $backEnd);
      }
      if(isset($oEntries[0]["entrycachehitratio"])) {
        $oCache->setHitRatio($oEntries[0]["entrycachehitratio"][0], $backEnd);
      }
      if(isset($oEntries[0]["currententrycachecount"])) {
        $oCache->setEntriesInCache($oEntries[0]["currententrycachecount"][0], $backEnd);
      }
    }
    $ldap->free_result($hResult);
    return $oCache;  
  }

  /**
  * OpenDS Back end database list
  * Returns a List of available backends
  */
  function opends_setBackEndDatabases($ldap) {

    // Returns a list in the format:
    // $returnList[0][0] = ds-cfg-backend-id
    // $returnList[0][1] = Back End Name
    $returnList = Array();

    $ldap->set_base_dn("cn=Backends,cn=config");

    // @ to hide err msg:es
    $hResult = @$ldap->query("(objectclass=ds-cfg-local-db-backend)", array("ds-cfg-backend-id", "ds-cfg-base-dn"));

    if(!$hResult) {
      return $returnList;
    }

    $oEntries = $ldap->get_result($hResult);
    for($i = 0; $i < (count($oEntries) -1); $i++) {
      if(!empty($oEntries[$i]["ds-cfg-backend-id"][0])) {
        $counter = count($returnList);
        $returnList[$counter][0] = $oEntries[$i]["ds-cfg-backend-id"][0];
        $returnList[$counter][1] = $oEntries[$i]["ds-cfg-base-dn"][0];
      }
    }

    $ldap->free_result($hResult);
    return $returnList;  
  }

  /**
  * OpenDS Back end index list
  * Returns a List of available indexes for specific backend
  */
  function opends_setIndexesList($ldap, $backenddatabase) {

    // Returns a list in the format:
    // $returnList[0] = index
    $returnList = Array();

    $ldap->set_base_dn("cn=Index,ds-cfg-backend-id=".$backenddatabase.",cn=Backends,cn=config");

    // @ to hide err msg:es
    $hResult = @$ldap->query("(objectclass=ds-cfg-local-db-index)", array("ds-cfg-attribute"));

    if(!$hResult) {
      return $returnList;
    }

    $oEntries = $ldap->get_result($hResult);

    for($i = 0; $i < (count($oEntries) -1); $i++) {
      if(!empty($oEntries[$i]["ds-cfg-attribute"][0])) {
        $returnList[count($returnList)] = $oEntries[$i]["ds-cfg-attribute"][0];
      }
    }

    $ldap->free_result($hResult);
    return $returnList;  
  }

  /**
  * OpenDS Translate Index to match how we want to store and manage it in the index object
  * Returns shortname of index
  */
  function translateIndex($index) {
    $index = strtolower($index);
    if($index == "equality") {
      return "eq";
    }
    if($index == "substring") {
      return "sub";
    }
    if($index == "presence") {
      return "pres";
    }
    if($index == "ordering") {
      return "ord";
    }
    if($index == "approximate") {
      return "approx";
    }
    return "";
  }

  /**
  * OpenDS Index Object
  * Returns a index object
  */
  function opends_setIndex($ldap, $servername, $oReturn, $backenddatabase, $queryindex) {

    $ldap->set_base_dn("cn=Index,ds-cfg-backend-id=".$backenddatabase.",cn=Backends,cn=config");
    $hResult = @$ldap->query("(&(ds-cfg-attribute=".$queryindex.")(objectclass=ds-cfg-local-db-index))", array("ds-cfg-index-type"));

    if(!$hResult) {
      return $oReturn;
    }

    $oEntries = $ldap->get_result($hResult);
    $nRows = $ldap->num_rows($hResult);
    
    // we only want one result
    if(($nRows == 1) && (!empty($oEntries[0]["ds-cfg-index-type"][0]))) {
      for($i = 0; $i < (count($oEntries[0]["ds-cfg-index-type"]) -1); $i++) {
        if(!empty($oEntries[0]["ds-cfg-index-type"][$i])) {
          $indexType = translateIndex($oEntries[0]["ds-cfg-index-type"][$i]);
          $oReturn->setIndexType($indexType);
          $oReturn->setIndexTypeByHost($servername, $indexType);
        }
      }
    }

    $ldap->free_result($hResult);
    return $oReturn;  
  }

  /**
  * Returns a Replication agreements Objects
  * $csnCheck if you want to disable CSN check
  */
  function opends_setReplicationAgreement($oServerReplica, $servername, $csnCheck = false) {
    global $CONFIG;
    
    $replicationHosts = array();
    $replicationSuffix = array();

    $ldap = createConnection($servername);

    if(!$ldap) {
      $oServerReplica->setMessage("Failed retrieving replication agreements");
      return $oServerReplica;
    }

    $replicationHostsCurrent = opends_getHostNameOfServer($ldap);
    if(empty($replicationHostsCurrent)) {
      closeConnection($ldap);
      $oServerReplica->setMessage("Unable to determine local replication host");
      return $oServerReplica;
    }
    //echo "t:".$replicationHostsCurrent.":t";

    // first get available replication roots
    $ldap->set_base_dn("cn=Multimaster Synchronization,cn=Synchronization Providers,cn=config");
    $hResult = @$ldap->query("objectclass=ds-cfg-replication-domain", 
                             array("ds-cfg-replication-server", "ds-cfg-base-dn")); 

    if(!$hResult) {
      closeConnection($ldap);
      $oReplicaInfo->setMessage("Unable to fetch result");
      return $oReplicaInfo;
    }

    $oEntries = $ldap->get_result($hResult);
    for($i = 0; $i < (count($oEntries) -1); $i++) {
      // replication hosts exists
      if(!empty($oEntries[$i]["ds-cfg-replication-server"])) {
        for($j = 0; $j < (count($oEntries[$i]["ds-cfg-replication-server"]) - 1); $j++) {
          //echo $oEntries[$i]["ds-cfg-replication-server"][$j]."<br>\n";
          // add host if replication server is not in the host array + not the server we are looking at + ":"
          if(!in_array($oEntries[$i]["ds-cfg-replication-server"][$j], $replicationHosts) &&
              (strpos($oEntries[$i]["ds-cfg-replication-server"][$j], $replicationHostsCurrent.":") === false)) {
            $replicationHosts[count($replicationHosts)] = $oEntries[$i]["ds-cfg-replication-server"][$j];
          }
        }
      }
      // replication suffix exists
      if(!empty($oEntries[$i]["ds-cfg-base-dn"])) {
        if(!in_array($oEntries[$i]["ds-cfg-base-dn"][0], $replicationHosts)) {
          $replicationSuffix[count($replicationSuffix)] = $oEntries[$i]["ds-cfg-base-dn"][0];        
        }      
      }
    }

    $ldap->free_result($hResult);

    //print_r($replicationHosts);
    //print_r($replicationSuffix);
    // if we have a host list we should add the backend db.s
    if(!empty($replicationHosts)) {
      $oServerReplica->setBackendDb($replicationSuffix);    
    }

    // by now we have a list of replication host (except current host) + suffixes
    // as for now we only support multimaster replication on all suffixes
    $ldap->set_base_dn("cn=replication, cn=monitor");
    for($i = 0; $i < count($replicationHosts); $i++) {
      for($j = 0; $j < count($replicationSuffix); $j++) {
        // split host and port
        $aHostAndPort = explode(":", $replicationHosts[$i]);
        $oAgreement = new ReplicaAgreement();
        $oAgreement->setHostName($aHostAndPort[0]);
        $hResult = @$ldap->query("(&(Replication-Server=".$replicationHosts[$i].")(domain-name=".$replicationSuffix[$j]."))",
                                 array("received-updates", "server-state")); 
        $nRows = $ldap->num_rows($hResult);
        // one replication agreement match that is we have contact with replication server
        if($nRows == 1) {
          $oEntry = $ldap->get_result($hResult);
          if(!empty($oEntry[0]["received-updates"][0])) {
            $oAgreement->setNrOfChanges($oEntry[0]["received-updates"][0]);
          }
          // if available in cn=monitor it should be up and working?
          $oAgreement->setReplicationError(false);
          $oServerReplica->setAgreementsOk(true);
          
          if(!empty($oEntry[0]["server-state"])) {
            $highestValue = 0;
            for($k = 0; $k < (count($oEntry[0]["server-state"]) - 1); $k++) {
              //echo "P:".$oEntry[0]["server-state"][$k]."<br>\n";
              // get time from string "#### <date> ####"
              $nStart = strpos($oEntry[0]["server-state"][$k], ' ');
              $nEnd   = (strrpos($oEntry[0]["server-state"][$k], ' ') - $nStart);
              $nDate  = strtotime(substr($oEntry[0]["server-state"][$k], $nStart, $nEnd));
              if($nDate > $highestValue) {
                $highestValue = $nDate;
              }
            } // for
            //echo "Set Date:".date("Y-m-d H:i:s", $highestValue)."<br>\n";
            date_default_timezone_set('UTC');
            $oAgreement->setUpdateStartRaw(date("Y-m-d H:i:s", $highestValue));
          }
        }
        else {
          // unable to contact replication server for this host and suffix
          $oAgreement->setLastStatus("CN=Monitor believe this agreements is configured but can't find operational status in cn=monitor suffix");
          $oAgreement->setReplicationError(true);
          $oServerReplica->setAgreementsOk(false);
        }

        $oAgreement->setPort($aHostAndPort[1]);
        $oAgreement->setReplicaRoot($replicationSuffix[$j]);

        // add object to server replica object
        $oServerReplica->setAgreement($oAgreement);
        $oServerReplica->setServerlist($aHostAndPort[0]);


        $ldap->free_result($hResult);
      }
    }


    // print_r($oServerReplica);
    
    closeConnection($ldap);

    return $oServerReplica;
  }

  /**
  * Returns a Replica info
  * Showing nr of changes handled by replica, type of replica
  * Supplier parameter used for possibility to disconnect verification
  */
  function opends_setReplicaInfo($ldap, $oReplicaInfo) {

    $ldap->set_base_dn("cn=Multimaster Synchronization,cn=Synchronization Providers,cn=config");

    // @ to hide err msg:es
    $hResult = @$ldap->query("objectclass=ds-cfg-replication-domain", 
                             array("ds-cfg-base-dn", "ds-cfg-server-id")); 

    if(!$hResult) {
      $oReplicaInfo->setMessage("Unable to fetch result");
      return $oReplicaInfo;
    }
    $oReplicaInfo->setIsIplanet(false);

    $oEntries = $ldap->get_result($hResult);
    for($i = 0; $i < (count($oEntries) -1); $i++) {
      $currentReplicaRoot = $oEntries[$i]["ds-cfg-base-dn"][0];

      $oReplicaInfo->setReplicaRoot($currentReplicaRoot);
      $oReplicaInfo->setReplicaType(3, $currentReplicaRoot);
      $oReplicaInfo->setReplicaId($oEntries[$i]["ds-cfg-server-id"][0], $currentReplicaRoot);

    }
    $ldap->free_result($hResult);

    $ldap->set_base_dn("cn=replication,cn=monitor");
    $hResult = @$ldap->query("(&(objectclass=ds-monitor-entry)(domain-name=*)(status=*))",
                             array("sent-updates", "domain-name")); 
    if(!$hResult) {
      $oReplicaInfo->setMessage("Unable to fetch result");
      return $oReplicaInfo;
    }
    $oEntries = $ldap->get_result($hResult);

    for($i = 0; $i < (count($oEntries) -1); $i++) {
      $oReplicaInfo->setNrOfChanges($oEntries[$i]["sent-updates"][0], $oEntries[$i]["domain-name"][0]);
    }
    $ldap->free_result($hResult);    
    return $oReplicaInfo;
  }

  /**
  * OpenDS Returns available operations
  */
  function opends_setPerformanceCounterList($byte_operation) {    
    return array("ent_sent", $byte_operation, "tot_con", "exec_time", "search",
                   "cur_con", "add", "modify", "modifyrdn", "compare", "remove", 
                   "in", "simple_bind");
  }

?>