<?php
  /*
    Copyright (C) 2009-2011 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

  /**********************************************************************
  * iPlanet Functions (RHDS, CentOS, Fedora, Sun, ODSEE, Netscape ...
  *
  * Specific LDAP functions for Directory Servers
  *
  * Available functionality:
  * (X) setMonitor
  * (X) setSnmp
  * (X) getHostNameOfServer
  * (X) setCache
  * (X) setBackEndDatabases
  * (X) setIndex
  * (X) setIndexesList
  * (X) getReplicationCSN
  * (X) convertCSNValue
  * (X) setReplicationAgreement
  * (X) setReplicaInfo
  * (X) setPerformanceCounterList
  *
  **********************************************************************/

  /**
  * Returns a monitor object
  */
  function iplanet_setMonitor($ldap, $currentMonitor, $operation = Array()) {

      $ldap->set_base_dn("cn=monitor");


      // cn monitor (@ to hide ev. error message)
      $hResult = @$ldap->get_entry("cn=monitor");

      if(!$hResult) {
        $sMessage = "Unable to query server";
        $currentMonitor->setMessage($sMessage);
        return $currentMonitor;
      }
      $nRows = $ldap->num_rows($hResult);
 
      if($nRows > 0) {
        $aEntries = $ldap->get_result($hResult);
        if(!empty($aEntries[0]["version"])) {
          $currentMonitor->setVersion($aEntries[0]["version"][0]);
        }

        // as connection peak doesn't exist in redhat/fedora
        if(!empty($aEntries[0]["connectionpeak"])) {
          $currentMonitor->setConnectionPeak($aEntries[0]["connectionpeak"][0]);
        }
        if(!empty($aEntries[0]["threads"])) {
          $currentMonitor->setThreads($aEntries[0]["threads"][0]);
        }
        if(!empty($aEntries[0]["currentconnections"])) {
          $currentMonitor->setCurrentConnections($aEntries[0]["currentconnections"][0]);
        }
        if(!empty($aEntries[0]["totalconnections"])) {
          $currentMonitor->setTotalConnections($aEntries[0]["totalconnections"][0]);
        }
        if(!empty($aEntries[0]["entriessent"])) {
          $currentMonitor->setEntriesSent($aEntries[0]["entriessent"][0]);
        }
        if(!empty($aEntries[0]["bytessent"])) {
          $currentMonitor->setBytesSent($aEntries[0]["bytessent"][0]);
        }
        if(!empty($aEntries[0]["starttime"])) {
          $currentMonitor->setStartTime($aEntries[0]["starttime"][0]);
        }
        if(!empty($aEntries[0]["currenttime"])) {
          $currentMonitor->setCurrentTime($aEntries[0]["currenttime"][0]);
        }
      }
      else {
        $currentMonitor->setMessage("Unable to fetch result");
      }
    $ldap->free_result($hResult);

    return $currentMonitor;
  }

  function iplanet_setSnmp($ldap, $currentSnmp, $operation = Array()) {
      $ldap->set_base_dn("cn=monitor");

      // cn cnmp (@ to hide ev. error message)
      $hResult = @$ldap->get_entry("cn=snmp,cn=monitor", array()); 
      if(!$hResult) {
        $currentSnmp->setMessage("Unable to open ldap connection");
        return $currentSnmp;
      }
      $nRows = $ldap->num_rows($hResult);
      if($nRows > 0) {
        $aEntries = $ldap->get_result($hResult);
        if(!empty($aEntries[0]["anonymousbinds"])) {
          $currentSnmp->setAnonymousBinds($aEntries[0]["anonymousbinds"][0]);
        }
        if(!empty($aEntries[0]["unauthbinds"])) {
          $currentSnmp->setUnAuthenticatedBinds($aEntries[0]["unauthbinds"][0]);
        }
        if(!empty($aEntries[0]["simpleauthbinds"])) {
          $currentSnmp->setSimpleAuthBinds($aEntries[0]["simpleauthbinds"][0]);
        }
        if(!empty($aEntries[0]["strongauthbinds"])) {
          $currentSnmp->setStrongAuthBinds($aEntries[0]["strongauthbinds"][0]);
        }
        if(!empty($aEntries[0]["wholesubtreesearchops"])) {
          $currentSnmp->setWholeSubtreeSearchOps($aEntries[0]["wholesubtreesearchops"][0]);
        }
        if(!empty($aEntries[0]["bindsecurityerrors"])) {
          $currentSnmp->setBindSecurityErrors($aEntries[0]["bindsecurityerrors"][0]);
        }
        if(!empty($aEntries[0]["compareops"])) {
          $currentSnmp->setCompareOps($aEntries[0]["compareops"][0]);
        }
        if(!empty($aEntries[0]["errors"])) {
          $currentSnmp->setErrors($aEntries[0]["errors"][0]);
        }
        if(!empty($aEntries[0]["inops"])) {
          $currentSnmp->setInOps($aEntries[0]["inops"][0]);
        }
        if(!empty($aEntries[0]["addentryops"])) {
          $currentSnmp->setAddEntryOps($aEntries[0]["addentryops"][0]);
        }
        if(!empty($aEntries[0]["modifyentryops"])) {
          $currentSnmp->setModifyEntryOps($aEntries[0]["modifyentryops"][0]);
        }
        if(!empty($aEntries[0]["modifyrdnops"])) {
          $currentSnmp->setModifyRdnOps($aEntries[0]["modifyrdnops"][0]);
        }
        if(!empty($aEntries[0]["removeentryops"])) {
          $currentSnmp->setRemoveEntryOps($aEntries[0]["removeentryops"][0]);
        }
        if(!empty($aEntries[0]["searchops"])) {
          $currentSnmp->setSearchOps($aEntries[0]["searchops"][0]);
        }
        if(!empty($aEntries[0]["securityerrors"])) {
          $currentSnmp->setSecurityErrors($aEntries[0]["securityerrors"][0]);
        }
      }
 
    $ldap->free_result($hResult);

    return $currentSnmp;
  }

  function iplanet_getHostNameOfServer($ldap) {
    $sReturn = "";

    $hResult = @$ldap->get_entry("cn=config,cn=ldbm database,cn=plugins,cn=config", 
                                 array("nsslapd-directory"));

    if(!$hResult) {
      return $sReturn;
    }
    $nRows = $ldap->num_rows($hResult);

    if($nRows == 1) {
      $aEntries = $ldap->get_result($hResult);
      $sReturn = $aEntries[0]["nsslapd-directory"][0];
      $sReturn = preg_replace("/^.*.slapd-/i", "", $sReturn);
      $sReturn = str_replace("/db", "", $sReturn);
      $sReturn = str_replace("/opt/", "", $sReturn);
    }
    $ldap->free_result($hResult);

    return $sReturn;
  }

  function sun_getHostNameOfServer($ldap) {
    $sReturn = "";

    $hResult = @$ldap->get_entry("cn=disk0,cn=disk,cn=monitor", array("disk-dir"));

    if(!$hResult) {
      return $sReturn;
    }

    $nRows = $ldap->num_rows($hResult);
    if($nRows == 1) {
      $aEntries = $ldap->get_result($hResult);
      $sReturn = $aEntries[0]["disk-dir"][0];
      $sReturn = preg_replace("/^.*.slapd-/i", "", $sReturn);
      $sReturn = str_replace("/db", "", $sReturn);
    }
    $ldap->free_result($hResult);

    return $sReturn;
  }

  function iplanet_setCache($ldap, $oCache) {

    $ldap->set_base_dn("cn=ldbm database,cn=plugins,cn=config");

    // @ to hide err msg:es
    $hResult = @$ldap->query("cn=monitor");

    if(!$hResult) {
      $oCache->setMessage("Unable to query server", "general");
      return $oCache;
    }

    $oEntries = $ldap->get_result($hResult);

    for($i = 0; $i < (count($oEntries) -1); $i++) {

      // get back end name and make sure it's really a backend
      $backEnd = $oEntries[$i]["dn"];
      $backEnd = str_replace("cn=ldbm database", "", $backEnd);
      $backEnd = str_replace("cn=plugins", "", $backEnd);
      $backEnd = str_replace("cn=config", "", $backEnd);
      $backEnd = str_replace("cn=monitor", "", $backEnd);
      $backEnd = trim(str_replace(",", "", $backEnd));

      if(!empty($backEnd)) {

        $oCache->setBackendDb($backEnd);

        // exists on some ex. sunone
        if(isset($oEntries[$i]["ldapentrycount"])) {
          $oCache->setEntriesInDb($oEntries[$i]["ldapentrycount"][0], $backEnd);
        }
        if(isset($oEntries[$i]["currententrycachecount"])) {
          $oCache->setEntriesInCache($oEntries[$i]["currententrycachecount"][0], $backEnd);
        }
        if(isset($oEntries[$i]["entrycachehits"])) {
          $oCache->setCacheHits($oEntries[$i]["entrycachehits"][0], $backEnd);
        }
        if(isset($oEntries[$i]["entrycachehitratio"])) {
          $oCache->setHitRatio($oEntries[$i]["entrycachehitratio"][0], $backEnd);
        }
        if(isset($oEntries[$i]["currententrycachesize"])) {
          $oCache->setCurrentCache($oEntries[$i]["currententrycachesize"][0], $backEnd);
        }
        if(isset($oEntries[$i]["maxentrycachesize"])) {
          $oCache->setMaxCache($oEntries[$i]["maxentrycachesize"][0], $backEnd);
        }
        if(isset($oEntries[$i]["maxentrycachecount"])) {
          $oCache->setMaxEntriesCache($oEntries[$i]["maxentrycachecount"][0], $backEnd);
        }
      }
    }

    $ldap->free_result($hResult);
    return $oCache;  
  }

  /**
  * iPlanet Back end database list
  * Returns a List of available backends
  */
  function iplanet_setBackEndDatabases($ldap) {

    // Returns a list in the format:
    // $returnList[0][0] = CN
    // $returnList[0][1] = Back End Name
    $returnList = Array();

    $ldap->set_base_dn("cn=ldbm database,cn=plugins,cn=config");

    // @ to hide err msg:es
    $hResult = @$ldap->query("(objectclass=nsBackendInstance)", array("cn", "nsslapd-suffix"));

    if(!$hResult) {
      return $returnList;
    }

    $oEntries = $ldap->get_result($hResult);

    for($i = 0; $i < (count($oEntries) -1); $i++) {
      if(!empty($oEntries[$i]["cn"][0])) {
        $counter = count($returnList);
        $returnList[$counter][0] = $oEntries[$i]["cn"][0];
        $returnList[$counter][1] = $oEntries[$i]["nsslapd-suffix"][0];
      }
    }

    $ldap->free_result($hResult);
    return $returnList;  
  }

  /**
  * iPlanet Back end index list
  * Returns a List of available indexes for specific backend
  */
  function iplanet_setIndexesList($ldap, $backenddatabase) {

    // Returns a list in the format:
    // $returnList[0] = index
    $returnList = Array();

    $ldap->set_base_dn("cn=".$backenddatabase.",cn=ldbm database,cn=plugins,cn=config");

    // @ to hide err msg:es
    $hResult = @$ldap->query("(objectclass=nsIndex)", array("cn"));

    if(!$hResult) {
      return $returnList;
    }

    $oEntries = $ldap->get_result($hResult);

    for($i = 0; $i < (count($oEntries) -1); $i++) {
      if(!empty($oEntries[$i]["cn"][0])) {
        $returnList[count($returnList)] = $oEntries[$i]["cn"][0];
      }
    }

    $ldap->free_result($hResult);
    return $returnList;  
  }

  /**
  * iPlanet Index Object
  * Returns a index object
  */
  function iplanet_setIndex($ldap, $servername, $oReturn, $backenddatabase, $queryindex) {

    $ldap->set_base_dn("cn=".$backenddatabase.",cn=ldbm database,cn=plugins,cn=config");
    $hResult = @$ldap->query("(&(cn=".$queryindex.")(objectclass=nsIndex))", array("nsindextype"));

    if(!$hResult) {
      return $oReturn;
    }

    $oEntries = $ldap->get_result($hResult);
    $nRows = $ldap->num_rows($hResult);
    
    // we only want one result
    if(($nRows == 1) && (!empty($oEntries[0]["nsindextype"][0]))) {
      for($i = 0; $i < (count($oEntries[0]["nsindextype"]) -1); $i++) {
        if(!empty($oEntries[0]["nsindextype"][$i])) {
          $oReturn->setIndexType($oEntries[0]["nsindextype"][$i]);
          $oReturn->setIndexTypeByHost($servername, $oEntries[0]["nsindextype"][$i]);
        }
      }
    }

    $ldap->free_result($hResult);
    return $oReturn;  
  }

  /**
  * Replication Latency requires directory manager password
  * Input server, replicating server, database to query
  * Returns CSN if available
  */
  function iplanet_getReplicationCSN($servername, $masterserver, $database) {
    $sReturn = "";

    // if we should ignore replica status
    $configId = getEnvironmentId($masterserver);
    $ignoreReplicaStatus = getConfigValue("replication:ignoreconsumer", $configId[0], $configId[1]);
    if(!empty($ignoreReplicaStatus)) {
      return $sReturn;
    }

    // Verify if host exists under a different name
    $servername = verifyReturnedHost($servername);

    // verify if port is responding
    $oServer = setServer($servername);
    if(!$oServer->isAvailable()) {
      return $sReturn;
    }

    // should be secure
    $ldap = createConnection($servername);

    if(!$ldap) {
      return $sReturn;
    }
    
    $ldap->set_base_dn($database);

    // @ to hide err msg:es
    $hResult = @$ldap->query_one_level('(&(nsUniqueId=ffffffff-ffffffff-ffffffff-ffffffff)(objectClass=nsTombstone))', array("nsds50ruv", "nsruvReplicaLastModified")); 

    if(!$hResult) {
      closeConnection($ldap);
      return $sReturn;
    }

    $nRows = @$ldap->num_rows($hResult);
    if($nRows == 1) {
      $oEntries = $ldap->get_result($hResult);

      // get CSN value (unformated)
      $sReturn = iplanet_getCSNValueFromRUV($oEntries[0]["nsds50ruv"], $masterserver);
    } // if one row found

    $ldap->free_result($hResult);
    closeConnection($ldap);
    return $sReturn;  
  }
  
  // servername is the server we want to look for
  function iplanet_getCSNValueFromRUV($oNsds50ruv, $servername) {
    $sReturn = "";
    if(empty($oNsds50ruv) || empty($servername)) {
      return $sReturn;
    }
    $configId = getEnvironmentId($servername);
    $serverAlias = getConfigValue("alias", $configId[0], $configId[1]);

    // The ruv may refer to the master name or alias
    // to support both we create an array and add both if present
    $serverNameArray = array($servername);
    if(!empty($serverAlias)) {
      $serverNameArray[1] = $serverAlias;
    }

    // find correct RUV
    for($i = 0; $i < (count($oNsds50ruv) - 1); $i++) {
      foreach ($serverNameArray as $hostname) {
        if(strpos($oNsds50ruv[$i], $hostname) !== FALSE) {
          $csnUnformated = trim(preg_replace("/\{.*\}/", "", $oNsds50ruv[$i]));
          if($csnUnformated) {
            $oCSN = explode(" ", $csnUnformated);
            // I will only save the last CSN
            $sReturn = trim($oCSN[1]);
          }  // if csn value exists
        }  // if master was matched
      }// foreach master name
    }  // for loop
    return $sReturn;
  }

  function iplanet_convertCSNValue($csn) {
    if(empty($csn)) {
      return "";
    }
    $aCSN = unpack("a8tm/a4seq/a4masterid/a4subseq", $csn);  
    $tm = hexdec($aCSN["tm"]); 
    $seq = hexdec($aCSN["seq"]);
    $masterid = hexdec($aCSN["masterid"]);
    $subseq = hexdec($aCSN["subseq"]);
    // by setting UTC timezone the value generated on CN=Monitor server will be in UTC
    date_default_timezone_set('UTC');
    return date("Y-m-d H:i:s", $tm);
  }

  /**
  * Returns a Replication agreements Objects
  * $csnCheck if you want to disable CSN check
  */
  function iplanet_setReplicationAgreement($oServerReplica, $servername, $csnCheck = false) {
    global $CONFIG;
    $sMessage = "";
    $sAgreement = "";
    $sPassword = "";

    $ldap = createConnection($servername);

    if(!$ldap) {
      $oServerReplica->setMessage("Failed retrieving replication agreements");
      return $oServerReplica;
    }
    $ldap->set_base_dn("cn=mapping tree,cn=config");
    
    // @ to hide err msg:es
    $hBackends = @$ldap->query_one_level("cn=*"); 

    // if no backends are available
    if(!$hBackends) {
      $oServerReplica->setMessage("Unable to fetch agreement result");
      closeConnection($ldap);
      return $oServerReplica;
    }

    // get available backend entries
    $oBackendEntries = $ldap->get_result($hBackends);
    // get all backends
    $availableBackends = Array();
    for($i = 0; $i < (count($oBackendEntries) -1); $i++) {
      // if dn contains '"', replace with nothing as we will add them later
      $sBackend = str_replace("\"", "", $oBackendEntries[$i]["cn"][0]);
      
      $availableBackends[count($availableBackends)] = $sBackend;
    } // for each entry
    $ldap->free_result($hBackends);
    
    $aAllReplicationBackends = Array();
    // for each backend
    for($i = 0; $i < count($availableBackends); $i++) {

      $backendDN = "cn=replica,cn=\"";
      $backendDN .= $availableBackends[$i];
      $backendDN .= "\",cn=mapping tree,cn=config";

      // query backend agreements
      $ldap->set_base_dn($backendDN);
      $hAgreement = @$ldap->query_one_level("cn=*");
  
  
      // if agreement exists
      if($hAgreement) {
        $availableAgreements = $ldap->get_result($hAgreement);

        // for each agreement
        for($j = 0; $j < (count($availableAgreements) -1); $j++) {
          // ignore default replica agreement
          if($availableAgreements[$j]["cn"][0] != "replica") {
        
            // if not in replica database list
            if(!in_array($availableBackends[$i], $aAllReplicationBackends)) {
              $aAllReplicationBackends[count($aAllReplicationBackends)] = $availableBackends[$i];
            }
        
            $currenthostname = "";
            $currentdatabase = "";
            $oAgreement = new ReplicaAgreement();
          
            // name of agreement
            if(!empty($availableAgreements[$j]["nsds5replicahost"][0])) {
              // set to lower case to avoid mixed case
              $currenthostname = strtolower($availableAgreements[$j]["nsds5replicahost"][0]);
            }
            $oAgreement->setHostName($currenthostname);
            
            /* CN = Name of agreement, usefull?
            if(!empty($availableAgreements[$j]["cn"][0])) {
              $oAgreement->setHostName($availableAgreements[$j]["cn"][0]);
              $currenthostname = $availableAgreements[$j]["cn"][0];
            }
            */

            // sunone agreement disabled
            if(!empty($availableAgreements[$j]["ds5agreementenable"][0])) {
              $replicationDisabled = strtolower($availableAgreements[$j]["ds5agreementenable"][0]);
              if($replicationDisabled == "off") {
                $oAgreement->setDisabledReplication(true);
              }
            }
          
            $oAgreement->setPort($availableAgreements[$j]["nsds5replicaport"][0]);
            $oAgreement->setBindMethod($availableAgreements[$j]["nsds5replicabindmethod"][0]);
            $oAgreement->setBindDn($availableAgreements[$j]["nsds5replicabinddn"][0]);
            $oAgreement->setReplicaRoot($availableAgreements[$j]["nsds5replicaroot"][0]);
            $currentdatabase = $availableAgreements[$j]["nsds5replicaroot"][0];

            if(!empty($availableAgreements[$j]["description"])) {
              $oAgreement->setDescription($availableAgreements[$j]["description"][0]);
            }

            if(!empty($availableAgreements[$j]["nsds5replicaupdateschedule"][0])) {
              $oAgreement->setUpdateSchedule($availableAgreements[$j]["nsds5replicaupdateschedule"][0]);
            }

            $oAgreement->setUpdateStart($availableAgreements[$j]["nsds5replicalastupdatestart"][0]);
            $oAgreement->setUpdateEnd($availableAgreements[$j]["nsds5replicalastupdateend"][0]);
            $oAgreement->setNrOfChanges($availableAgreements[$j]["nsds5replicachangessentsincestartup"][0]);

            // replication status
            if(!empty($availableAgreements[$j]["nsds5replicalastupdatestatus"][0])) {
              $oAgreement->setLastStatus($availableAgreements[$j]["nsds5replicalastupdatestatus"][0]);
              $replicationStatusArray = explode(" ", $availableAgreements[$j]["nsds5replicalastupdatestatus"][0]);
              // also verified if known error exists
              if(($replicationStatusArray[0] != "0") && 
                 ($replicationStatusArray[0] != "1")) {
                // 0 = ok, 1 = busy (which in most cases is temporary)
                $oAgreement->setReplicationError(true);
                $oServerReplica->setAgreementsOk(false);
                // replication error indication
                $oServerReplica->setExitCode(1);
              }
              else {
                // only registered as ok if everything looks ok
                $oServerReplica->setAgreementsOk(true);
              }
            }
            

            $oAgreement->setInProgress($availableAgreements[$j]["nsds5replicaupdateinprogress"][0]);

            // get CSN value unless we know we have replication issues
            if(!$oAgreement->getReplicationError() && $csnCheck) {
              // set master CSN value
              $oServerReplica->setMasterCSN($currentdatabase,
                                            iplanet_convertCSNValue(
                                            iplanet_getCSNValueFromRUV($availableAgreements[$j]["nsds50ruv"], $servername)));
              // set remote replica CSN value
              $oAgreement->setCSN(
                                 iplanet_convertCSNValue(
                                 iplanet_getReplicationCSN($currenthostname, $servername, $currentdatabase)));
            }
            
            // add object to server replica object
            $oServerReplica->setAgreement($oAgreement);
            $oServerReplica->setServerlist($currenthostname);
          } // if a usefull agreement

        } // for each agreement on backend
        $ldap->free_result($hAgreement);
      } // if agreement
    }
    $oServerReplica->setBackendDb($aAllReplicationBackends);
    closeConnection($ldap);

    return $oServerReplica;
  }

  /**
  * Returns a Replica info
  * Showing nr of changes handled by replica, type of replica
  * Supplier parameter used for possibility to disconnect verification
  */
  function iplanet_setReplicaInfo($ldap, $oReplicaInfo) {

    $ldap->set_base_dn("cn=mapping tree,cn=config");

    // @ to hide err msg:es
    $hResult = @$ldap->query("objectclass=nsDS5Replica"); 

    if(!$hResult) {
      $oReplicaInfo->setMessage("Unable to fetch result");
      return $oReplicaInfo;
    }

    $oReplicaInfo->setIsIplanet(true);
//    $nRows = $ldap->num_rows($hResult);

    $oEntries = $ldap->get_result($hResult);
//    print_r($oEntries);
    for($i = 0; $i < (count($oEntries) -1); $i++) {
      $currentReplicaRoot = $oEntries[$i]["nsds5replicaroot"][0];

      $oReplicaInfo->setReplicaRoot($currentReplicaRoot);
      $oReplicaInfo->setReplicaForwardFlag($oEntries[$i]["nsds5flags"][0], $currentReplicaRoot);
      $oReplicaInfo->setReplicaType($oEntries[$i]["nsds5replicatype"][0], $currentReplicaRoot);
      $oReplicaInfo->setReplicaId($oEntries[$i]["nsds5replicaid"][0], $currentReplicaRoot);
      if(!empty($oEntries[$i]["nsds5replicachangecount"][0])) {
        $oReplicaInfo->setNrOfChanges($oEntries[$i]["nsds5replicachangecount"][0], $currentReplicaRoot);
      }
      else {
        $oReplicaInfo->setNrOfChanges(0, $currentReplicaRoot);
      }
    }

    $ldap->free_result($hResult);

    return $oReplicaInfo;
  }

  /**
  * iPlanet Returns available operations
  */
  function iplanet_setPerformanceCounterList($byte_operation) {    
    return array("ent_sent", $byte_operation, "tot_con", "exec_time", "search", "cur_con",
                 "add", "modify", "modifyrdn", "compare", "remove", "in",
                 "anon_bind", "unauth_bind", "simple_bind", "strong_bind", "error");
  }

?>