<?php
  /*
    Copyright (C) 2009-2012 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

/*
  Configurable Parameters
  $CONFIG["general"]["db"]["username"];
  $CONFIG["general"]["db"]["password"];
  $CONFIG["general"]["db"]["host"];
  $CONFIG["general"]["db"]["database"];
  $CONFIG["general"]["db"]["type"];

  OPTIONAL
  $CONFIG["general"]["db"]["port"];
*/

/************************************************
* dbConfigured
* Verify if database is configured
* This function must be called in order to activate
* SQL vendor functions.
* @return true/false if database connection exists
************************************************/
function dbConfigured() {
  global $CONFIG;

  if(empty($CONFIG["general"]["db"]["type"]) ||
     empty($CONFIG["general"]["db"]["host"]) ||
     empty($CONFIG["general"]["db"]["database"]) || 
     empty($CONFIG["general"]["db"]["username"])) {
    return false;
  }

  $vendorFunctionFile = "/sql/".$CONFIG["general"]["db"]["type"]."_functions.php";

  if(file_exists(dirname(__FILE__).$vendorFunctionFile)) {
    logToFile("setServer", "loading sql functions for '".$CONFIG["general"]["db"]["type"] ."'", "", 3);
    require_once dirname(__FILE__).$vendorFunctionFile;    
  }

  // try to open the database connection
  $dbh = dbConnect();

  if($dbh) {
    dbClose($dbh);
    return true;
  }
  logToFile("dbConfigured", "Unable to connect to SQL Database. Type: ".$CONFIG["general"]["db"]["type"].", Check your SQL configuration!", null, 1);  
  return false;
}

/************************************************
* setSQLConnect
* Connect to database
* @return database reference
************************************************/
function dbConnect() {
  global $CONFIG;
  $type = $CONFIG["general"]["db"]["type"];
  $port = "";
  if(!empty($CONFIG["general"]["db"]["port"])) {
    $port = $CONFIG["general"]["db"]["port"];
  }

  $vendorFunctionName = $type."_dbConnect";
  if(function_exists($vendorFunctionName)) {
    $dbh = call_user_func($vendorFunctionName,
                          $CONFIG["general"]["db"]["host"],
                          $CONFIG["general"]["db"]["database"],
                          $port,
                          $CONFIG["general"]["db"]["username"],
                          $CONFIG["general"]["db"]["password"]);
  }
  if(empty($dbh)) {
    return "";
  }
  return $dbh;
}

/************************************************
* dbClose
* Close database connection
* @param database reference
* @return void
************************************************/
function dbClose($dbh) {
  global $CONFIG;
  
  if(empty($dbh)) {
    return;
  }

  $type = $CONFIG["general"]["db"]["type"];
  $vendorFunctionName = $type."_dbClose";
  if(function_exists($vendorFunctionName)) {
    call_user_func($vendorFunctionName, $dbh);
  }
}

/************************************************
* dbQuery
* Query Database
* @param sqlquery
* @param database reference
* @return result reference
************************************************/
function dbQuery($sql, $dbh) {
  global $CONFIG;
  $type = $CONFIG["general"]["db"]["type"];

  $vendorFunctionName = $type."_dbQuery";
  if(function_exists($vendorFunctionName)) {
    return call_user_func($vendorFunctionName, $dbh, $sql);
  }
}

/************************************************
* dbGetNumRows
* Returns number of rows in result
* @param result reference
* @return number of rows
************************************************/
function dbGetNumRows($result) {
  global $CONFIG;
  $type = $CONFIG["general"]["db"]["type"];

  $vendorFunctionName = $type."_dbGetNumRows";
  if(function_exists($vendorFunctionName)) {
    return call_user_func($vendorFunctionName, $result);
  }
}

/************************************************
* dbGetRow
* Returns Row handler
* @param result reference
* @return database fetch row function
************************************************/
function dbGetRow($result) {
  global $CONFIG;
  $type = $CONFIG["general"]["db"]["type"];

  $vendorFunctionName = $type."_dbGetRow";
  if(function_exists($vendorFunctionName)) {
    return call_user_func($vendorFunctionName, $result);
  }
}

/************************************************
* dbFreeResult
* Free Query Result
* @param result reference
* @return void
************************************************/
function dbFreeResult($result) {
  global $CONFIG;
  $type = $CONFIG["general"]["db"]["type"];
  $vendorFunctionName = $type."_dbFreeResult";
  if(function_exists($vendorFunctionName)) {
    return call_user_func($vendorFunctionName, $result);
  }
}

// set hour
function getSelectHour($intype, $invalue = "00", $enabled) {
  echo "\n<select id='".$intype."hour' name='".$intype."hour'";
  if(!$enabled) {
    echo " disabled='disabled'";
  }
  echo ">\n";
    setTimeValue($invalue, 24);
  echo "</select>\n";
}

// set minute
function getSelectMinute($intype, $invalue = "00", $enabled) {
  echo "\n<select id='".$intype."minute' name='".$intype."minute'";
  if(!$enabled) {
    echo " disabled='disabled'";
  }
  echo ">\n";
    setTimeValue($invalue, 60);
  echo "</select>\n";
}

// set second
function getSelectSecond($intype, $invalue = "00", $enabled) {
  echo "\n<select id='".$intype."second' name='".$intype."second'";
  if(!$enabled) {
    echo " disabled='disabled'";
  }
  echo ">\n";
    setTimeValue($invalue, 60);
  echo "</select>\n";
}

// generate time values and set selected
function setTimeValue($invalue, $maxvalue) {
  for($i = 0; $i < $maxvalue; $i++) {
    $value = "";
  
    if($i < 10) {
      $value .= "0";
    }
    $value .= "".$i;

    echo "<option value='";
    echo $value;
    echo "'";
    if($invalue == $value) {
      echo " selected";
    }
    echo ">";
    echo $value;
    echo "</option>\n";
  }
}

/************************************************
* setMonitorHistory
* Store Monitor Object to database
* @param monitor Monitor Object
* @param currentTime Timestamp
* @return void
************************************************/
function setMonitorHistory($monitor, $currentTime, $executionTime) {

  $currentHostName = $monitor->getHostName();
  if(empty($currentTime) || empty($currentHostName)) {
    return;
  }

  $sql  = "INSERT INTO monitor (";
  $sql .= "col_mon_date";
  $sql .= ",col_mon_host";
  $sql .= ",col_mon_con_peak";
  $sql .= ",col_mon_thread";
  $sql .= ",col_mon_cur_con";
  $sql .= ",col_mon_ent_sent";
  $sql .= ",col_mon_tot_con";
  $sql .= ",col_mon_byte_sent";
  $sql .= ",col_mon_exec_time";
  $sql .= ",col_mon_msg)";
  $sql .= " VALUES(";
  $sql .= "'".$currentTime."'";
  $sql .= ",'".$currentHostName."'";
  $sql .= ",'".$monitor->getConnectionPeak()."'";
  $sql .= ",'".$monitor->getThreads()."'";
  $sql .= ",'".$monitor->getCurrentConnections()."'";
  $sql .= ",'".$monitor->getEntriesSent()."'";
  $sql .= ",'".$monitor->getTotalConnections()."'";
  $sql .= ",'".$monitor->getBytesSent()."'";
  $sql .= ",'".$executionTime."'";
  $sql .= ",'".$monitor->getMessage()."'";
  $sql .= ")";
  $dbh = dbConnect();
  dbQuery($sql, $dbh);
  dbClose($dbh);
}

/************************************************
* setSnmpHistory
* Store SNMP Object to database
* @param snmp SNMP Object
* @param currentTime Timestamp
* @return void
************************************************/
function setSnmpHistory($snmp, $currentTime) {
  $currentHostName = $snmp->getHostName();
  if(empty($currentTime) || empty($currentHostName)) {
    return;
  }

  $sql  = "INSERT INTO operation (";
  $sql .= "col_mon_date";
  $sql .= ",col_mon_host";
  $sql .= ",col_mon_anon_bind";
  $sql .= ",col_mon_unauth_bind";
  $sql .= ",col_mon_simple_bind";
  $sql .= ",col_mon_strong_bind";
  $sql .= ",col_mon_error_bind";
  $sql .= ",col_mon_whole_search";
  $sql .= ",col_mon_search";
  $sql .= ",col_mon_in";
  $sql .= ",col_mon_compare";
  $sql .= ",col_mon_add";
  $sql .= ",col_mon_modify";
  $sql .= ",col_mon_modifyrdn";
  $sql .= ",col_mon_remove";
  $sql .= ",col_mon_error";
  $sql .= ",col_mon_sec_error";
  $sql .= ",col_mon_msg)";
  $sql .= " VALUES(";
  $sql .= "'".$currentTime."'";
  $sql .= ",'".$currentHostName."'";
  $sql .= ",'".$snmp->getAnonymousBinds()."'";
  $sql .= ",'".$snmp->getUnAuthenticatedBinds()."'";
  $sql .= ",'".$snmp->getSimpleAuthBinds()."'";
  $sql .= ",'".$snmp->getStrongAuthBinds()."'";
  $sql .= ",'".$snmp->getBindSecurityErrors()."'";
  $sql .= ",'".$snmp->getWholeSubtreeSearchOps()."'";
  $sql .= ",'".$snmp->getSearchOps()."'";
  $sql .= ",'".$snmp->getInOps()."'";
  $sql .= ",'".$snmp->getCompareOps()."'";
  $sql .= ",'".$snmp->getAddEntryOps()."'";
  $sql .= ",'".$snmp->getModifyEntryOps()."'";
  $sql .= ",'".$snmp->getModifyRdnOps()."'";
  $sql .= ",'".$snmp->getRemoveEntryOps()."'";
  $sql .= ",'".$snmp->getErrors()."'";
  $sql .= ",'".$snmp->getSecurityErrors()."'";
  $sql .= ",'".$snmp->getMessage()."'";
  $sql .= ")";

  $dbh = dbConnect();
  dbQuery($sql, $dbh);
  dbClose($dbh);

}

/************************************************
* setSummary
* Store Summary value to database
* @return void
************************************************/
function setSummary($servername, $startdate, $operation, $summaryvalue) {
  $debug = 0;

  if(empty($servername) || empty($startdate)) {
    return;
  }

  $dbh = dbConnect();

  $sql = "SELECT col_sum_value FROM summary WHERE col_sum_date = '".$startdate."'";
  $sql .= " AND col_sum_host = '".$servername."' AND col_sum_key = '".$operation."'";
  if($debug) { echo $sql."\n"; }
  
  $result = dbQuery($sql, $dbh);
  $rowExists = 0;
  while($row = dbGetRow($result)) {
    $rowExists = 1;
  }
  
  if($rowExists) {
    $sql = "UPDATE summary SET col_sum_value = '".$summaryvalue."', col_sum_updated = '".date("Y-m-d H:i:s")."'";
    $sql .= " WHERE col_sum_date = '".$startdate."' AND col_sum_host = '".$servername."' AND col_sum_key = '".$operation."'";
  }
  else {
    $sql  = "INSERT INTO summary (";
    $sql .= "col_sum_date";
    $sql .= ",col_sum_host";
    $sql .= ",col_sum_updated";
    $sql .= ",col_sum_key";
    $sql .= ",col_sum_value)";
    $sql .= " VALUES(";
    $sql .= "'".$startdate."'";
    $sql .= ",'".$servername."'";
    $sql .= ",'".date("Y-m-d H:i:s")."'";
    $sql .= ",'".$operation."'";
    $sql .= ",'".$summaryvalue."'";
    $sql .= ")";
  }
  
  if($debug) { echo $sql."\n"; }

  dbQuery($sql, $dbh);
  dbClose($dbh);
}

/************************************************
* removeOldCollectedData
* Delete old values from server message, monitor and operations
* @return boolean success
************************************************/
function removeOldCollectedData() {
  $debug = false;

  $days = getConfigValue("cleandb"); 
  // either use default days value (730)
  if($days == 0) {
    $days = 730;  
  }
  else {
    $days = intVal($days);
  }
  // a value below number of days in a month
  // will affect the summary collection and should be avoided
  if($days < 40) {
    echo "Error: Configured clean db option set to below 40. Not supported.\n";
    return false;
  }
  $dateTimestamp = strtotime("-".$days." day");
  $dateString = date("Y-m-d", $dateTimestamp)." 00:00:00";

  // remove old values
  $sqlMessages = "DELETE FROM servermessage WHERE col_srvmsg_date < '".$dateString."';";
  $sqlMonitor = "DELETE FROM monitor WHERE col_mon_date < '".$dateString."';";
  $sqlOperation = "DELETE FROM operation WHERE col_mon_date < '".$dateString."';";
  if($debug) { echo "SQL Delete Servermessage: ".$sqlMessages."\n"; }
  if($debug) { echo "SQL Delete Monitor: ".$sqlMonitor."\n"; }
  if($debug) { echo "SQL Delete Operation: ".$sqlOperation."\n"; }
  $dbh = dbConnect();
  dbQuery($sqlMessages, $dbh);
  dbQuery($sqlMonitor, $dbh);
  dbQuery($sqlOperation, $dbh);
  dbClose($dbh);
  return true;
}

/************************************************
* setServerMessage
* Check and Store New Status Message value to database
* @return boolean first time message collected
************************************************/
function setServerMessage($servername, $currenttime, $messagetype, $messagevalue = "") {
  $debug = 0;
  $statusChanged = true;

  if(empty($servername) || empty($currenttime) || empty($messagetype)) {
    return;
  }

  $dbh = dbConnect();

  // if the last status is as same as we got, don't do anything
  $sql = "SELECT col_srvmsg_key, col_srvmsg_value FROM servermessage WHERE col_srvmsg_host = '".$servername."' ";
  $sql .= "ORDER BY col_srvmsg_date DESC LIMIT 1";
  if($debug) { echo $sql."\n"; }
  
  $result = dbQuery($sql, $dbh);
  $row = dbGetRow($result);
  // if first row in result matches this status
  if(!empty($row)) {
    if($row[0] == $messagetype) {
      $statusChanged = false;
    }
  }
  
  if($statusChanged) {
    // for each new message
    $sql  = "INSERT INTO servermessage (";
    $sql .= "col_srvmsg_date";
    $sql .= ",col_srvmsg_host";
    $sql .= ",col_srvmsg_key";
    $sql .= ",col_srvmsg_value)";
    $sql .= " VALUES(";
    $sql .= "'".$currenttime."'";
    $sql .= ",'".$servername."'";
    $sql .= ",'".$messagetype."'";
    $sql .= ",'".$messagevalue."'";
    $sql .= ")";  
    if($debug) { echo $sql."\n"; }
    dbQuery($sql, $dbh);
  }
  dbClose($dbh);
    
  return $statusChanged;
}

/*********************************************************************
* setServerRestarted
* Check and Store Restarted value to database
* This method exist as we can't include it in with setServerMessage
* @return boolean first time message collected
*********************************************************************/
function setServerRestarted($servername, $currenttime, $messagetype, $messagevalue = "") {
  $serverRestarted = false;
  $debug = 0;

  if(empty($servername) || empty($currenttime) || empty($messagetype)) {
    return;
  }

  $dbh = dbConnect();

  // only applicable for ldap_available status
  if($messagetype == "ldap_available") {
    $serverRestarted = false;

    // check if time has been registered before
    $sql = "SELECT col_srvmsg_value FROM servermessage WHERE col_srvmsg_key = '".$messagetype."'";
    $sql .= " AND col_srvmsg_host = '".$servername."' AND col_srvmsg_value = '".$messagevalue."'";
    if($debug) { echo $sql."\n"; }
    $resultTime = dbQuery($sql, $dbh);
    $rowTime = dbGetRow($resultTime);
    if(empty($rowTime)) {
      if($debug) { echo "Server restarted\n"; }
      $serverRestarted = true;
    }
  }

  if($serverRestarted) {
    // for each new message
    $sql  = "INSERT INTO servermessage (";
    $sql .= "col_srvmsg_date";
    $sql .= ",col_srvmsg_host";
    $sql .= ",col_srvmsg_key";
    $sql .= ",col_srvmsg_value)";
    $sql .= " VALUES(";
    $sql .= "'".$currenttime."'";
    $sql .= ",'".$servername."'";
    $sql .= ",'".$messagetype."'";
    $sql .= ",'".$messagevalue."'";
    $sql .= ")";  
    if($debug) { echo $sql."\n"; }
    dbQuery($sql, $dbh);
  }

  dbClose($dbh);

  return $serverRestarted;
}

?>