<?php
  /*
    Copyright (C) 2009-2012 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

/**********************************************************************
 LDAP Functions

 Stores General LDAP Functions, calls vendor specific methods
 Avoid having vendor specific information here. The exception is functionality
 related to decide what type of replica vendor it is.
 Vendor specific functions are kept in:
 functions/vendor/<vendortype>_functions.php. Necessary files will be included.
**********************************************************************/

/**********************************************************************
  createConnection

  Sets up a LDAP Connection.
  This function is always called to create LDAP connections used for set object
  functions.

  @param string $servername Hostname
  @param int $port Port to open
  @return LDAP object
**********************************************************************/
  function createConnection($servername, $port = 0) {
    global $CONFIG;

    // get environment and serverid
    $configId = getEnvironmentId($servername);

    // if config ID is empty. Can happen if we try to connect to
    // a not configured servername such as missing alias
    if(empty($configId)) {
      return "";
    }

    // port
    if(empty($port)) {
      $port = getPort($servername);
    }

    // for rare occations where you want to use the non standard ldaps scheme
    $scheme = strtolower(getConfigValue("scheme", $configId[0], $configId[1]));
    if(empty($scheme)) {
      $scheme = "ldap";
    }
    
    // suggest that we set scheme ldaps if using port 636
    if($port == 636) {
      $scheme = "ldaps";
    }
    
    // and the other way around. If we are using scheme ldaps. Set secure port
    if($scheme == "ldaps") {
      $port = getSecurePort($servername);
    }

    // get dn to connect
    $dn = getConfigValue("dn", $configId[0], $configId[1]);
    if(empty($dn)) {
      $dn = "";
    }

    // get opt flags
    $ldap_opt_timelimit = getConfigValue("ldap_opt_timelimit", $configId[0], $configId[1]);
    if(empty($ldap_opt_timelimit)) {
      $ldap_opt_timelimit = 20;
    }
    $ldap_opt_network_timeout = getConfigValue("ldap_opt_network_timeout", $configId[0], $configId[1]);
    if(empty($ldap_opt_network_timeout)) {
      $ldap_opt_network_timeout = 20;
    }
    
    // get logfile options
    $logfile = getConfigValue("logldap", $configId[0], $configId[1]);

    // if in rare occations you need to override default secure recommendations
    $secure = getConfigValue("tls", $configId[0], $configId[1]);
    if(empty($secure)) {
      $secure = false;
    }

    $password = getConfigValue("password", $configId[0], $configId[1]);
    if(empty($password)) {
      $password = "";
    }

    $ldap = new LDAP();
    // use logfile
    if(!empty($logfile)) {
      $ldap->setLogFile($logfile);
    }
    
    // think ldaps only will work with openldap URL
    if($scheme == "ldaps") {
      $ldap->open($scheme."://".$servername.":".$port);
    }
    else {
      $ldap->open($servername, $port);
    }

    // Set low limit options
    // If constants exist
    if(defined('LDAP_OPT_TIMELIMIT')) {
      $ldap->setLDAPOption(LDAP_OPT_TIMELIMIT, $ldap_opt_timelimit);
    }
    if(defined('LDAP_OPT_NETWORK_TIMEOUT')) {
      $ldap->setLDAPOption(LDAP_OPT_NETWORK_TIMEOUT, $ldap_opt_network_timeout);
    }
    
    // if dn and password exist
    if(!empty($dn) && !empty($password)) {
      if(!@$ldap->bind($dn, $password, $secure)) {
        logToFile("createConnection", ldap_err2str(ldap_errno($ldap->hConn)), $servername, 1);
        // as bind failed. We close this connection
        $ldap->close();
        return "";
      }
    }
    logToFile("createConnection", var_export($ldap, true), $servername, 3);
    
    if(!$ldap->hConn) {
      logToFile("createConnection", ldap_err2str(ldap_errno($ldap->hConn)), $servername, 1);
      $ldap->close();
      return "";
    }
    return $ldap;
  }

/**********************************************************************
  closeConnection

  Closes LDAP Connection.
  This function should always be called from set functions to be sure that
  LDAP connections are closed correct.

  @param $ldap LDAP Object Reference
  @return void
**********************************************************************/
  function closeConnection($ldap) {
    if($ldap) {
      $ldap->close();
    }
    return;
  }

/**********************************************************************
  authenticate

  Verify if we can authenticate against the server
  
  @param string $servername Server Name
  @param int $port Port
  @return boolean true/false
**********************************************************************/
  function authenticate($servername, $dn, $password, $port = 0) {
    $returnValue = false;

    // create LDAP Connection
    if(!($ldap = createConnection($servername, $port))) {
      closeConnection($ldap);
      return $returnValue;
    }

    if(@$ldap->bind($dn, $password)) {
      $returnValue = true;
      logToFile("authenticate", "bind success", $servername, 2);  
    }
    else {
      logToFile("authenticate", "bind failure", $servername, 2);  
    }
    closeConnection($ldap);

    return $returnValue;
  }

/**********************************************************************
  getDNByID

  Get Distinguished Name by ID
  ID is by default CN or UID
  By setting the configuration property auth

  @param string $servername Server Name
  @param string $userid User ID value to get DN
  @return DN Distinguished Name
**********************************************************************/
  function getDNByID($servername, $userid) {
    $returnDN = "";

    // get environment and serverid
    $configId = getEnvironmentId($servername);

    $dnbyfilter = getConfigValue("dnbyfilter", $configId[0], $configId[1]);
    if(empty($dnbyfilter)) {
      $dnbyfilter = "(|(uid=?)(cn=?))";
    }

    $rootDN = setRootDN($servername);

    // create LDAP Connection
    if(!($ldap = createConnection($servername))) {
      closeConnection($ldap);
      return "";
    }

    // for each naming context. Try to find the DN
    $namingContexts = $rootDN->getNamingContexts();

    for($i = 0; $i < count($namingContexts); $i++) {
      $ldap->set_base_dn($namingContexts[$i]);
      // only first entry of matching query will be used
      $returnDN = @$ldap->find_user($userid, $dnbyfilter);
      if(!empty($returnDN)) {
        break;
      }
    }
    
    closeConnection($ldap);

    logToFile("getDNByID", "return dn:".$returnDN, $servername, 6);  
    return $returnDN;
  }

/**********************************************************************
  setServer

  Sets serverobject and verifies server availability

  @param string $servername Hostname
  @return Server Object
**********************************************************************/
  function setServer($servername) {
    global $SERVER;

    // server is empty    
    if(empty($servername)) {
      return new Server("");
    }

    // quick return if we already have an object ready
    if(!empty($SERVER[$servername])) {
      logToFile("setServer", "read page cache", $servername, 5);  
      return $SERVER[$servername];
    }

    $FILE = new File();
    $cacheFile = getTempFileName("server_".$servername);
    
    // create server object if not created before
    // do we have a cache file on disk?
    if(file_exists($cacheFile)) {
      $unserializedServer = $FILE->read($cacheFile);
      $oReturn = unserialize($unserializedServer);
      // new init for cached object
      $oReturn->init();
      logToFile("setServer", "read file cache: ".var_export($oReturn, true), $servername, 10);  
    }
    else {
      // new cache file, init auto performed
      $oReturn = new Server($servername);
    }
    $SERVER[$servername] = $oReturn;

    if($SERVER[$servername]->isUpdated()) {
      // clear updated flag in server object
      $SERVER[$servername]->setClearUpdate();
      $serializedServer = serialize($SERVER[$servername]);
      $FILE->write($cacheFile, $serializedServer);
      logToFile("setServer", "write cache: ".var_export($SERVER[$servername], true), $servername, 10);  
    }
    
    $vendor = $SERVER[$servername]->getVendor();

    // load ldap vendor function plugin file
    $vendorFunctionFile = "/vendor/".vendorMapToFunction($vendor)."_functions.php";
    if(file_exists(dirname(__FILE__).$vendorFunctionFile)) {
      logToFile("setServer", "loading functions for '".$vendor ."'", $servername, 3);
      require_once dirname(__FILE__).$vendorFunctionFile;    
    }

    return $SERVER[$servername];
  }
  
/**********************************************************************
  setMonitor

  Returns a Monitor object.
  Performance counters can exist in either the monitor or the SNMP object.
  This is an inheritance from iPlanet functionality that also affects database tables.

  You can improve performance by specifying single operations
  <code>
    $oMonitor = setMonitor("myldap.server.com", array("entsent", "totconn"));
  </code>
  @param string $servername Hostname
  @param string $operation Optional array operation parameter
  @return Monitor object
**********************************************************************/
  function setMonitor($servername, $operation = Array()) {
    global $CONFIG;

     // get environment and serverid
    $oServer = setServer($servername);
    $vendor = $oServer->getVendor();
    
    $starttime = microtimeFloat();

    $currentMonitor = new Monitor();
    $currentMonitor->setHostName($servername);

    // create LDAP Connection
    if(!($ldap = createConnection($servername))) {
      $endtime = microtimeFloat();
      $currentMonitor->setExecutionTime($endtime - $starttime);
      closeConnection($ldap);
      return $currentMonitor;
    }

    $vendorFunctionName = vendorMapToFunction($vendor)."_setMonitor";
    if(function_exists($vendorFunctionName)) {
      $currentMonitor = call_user_func($vendorFunctionName, $ldap, $currentMonitor, $operation);
    }

    closeConnection($ldap);
    $endtime = microtimeFloat();
    $currentMonitor->setExecutionTime($endtime - $starttime);
    $currentMonitor->setUptime();

    logToFile("setMonitor", var_export($currentMonitor, true), $servername, 10);  
    return $currentMonitor;
  }

/**********************************************************************
  setSnmp

  Returns a Snmp object

  You can improve performance by specifying a single operation
  <code>
    $oSnmp = setSnmp("myldap.server.com", array("entsent"));
  </code>
  @param string $servername Hostname
  @param string $operation Optional operation array parameter
  @return Snmp object
**********************************************************************/
  function setSnmp($servername, $operation = Array()) {
    global $CONFIG;

    $oServer = setServer($servername);
    $vendor = $oServer->getVendor();

    $currentSnmp = new Snmp();
    $currentSnmp->setHostName($servername);

    // create LDAP Connection
    if(!($ldap = createConnection($servername))) {
      $currentSnmp->setMessage("Unable to contact server");
      closeConnection($ldap);
      return $currentSnmp;
    }
    
    $vendorFunctionName = vendorMapToFunction($vendor)."_setSnmp";
    if(function_exists($vendorFunctionName)) {
      $currentMonitor = call_user_func($vendorFunctionName, $ldap, $currentSnmp, $operation);
    }

    closeConnection($ldap);

    logToFile("setSnmp", var_export($currentSnmp, true), $servername, 10);  
    return $currentSnmp;
  }

/**********************************************************************
  setCache

  Returns Replica Cache object

  @param string $servername server name
  @return Cache object
**********************************************************************/
  function setCache($servername) {
    global $CONFIG;

    $oServer = setServer($servername);
    $vendor = $oServer->getVendor();

    $oReturn = new Cache();
    $oReturn->setHostName($servername);

    $vendorFunctionName = vendorMapToFunction($vendor)."_setCache";
    // if vendor not supported, exit without establishing ldap connection
    if(!function_exists($vendorFunctionName)) {
      $oReturn->setMessage("Servertype '".$vendor."' not supported!", "general");
      $oReturn->setExitCodeMain(1);
      return $oReturn;
    }

    if(!$ldap = createConnection($servername)) {
      $oReturn->setMessage("Unable to connect to server '".$servername."'!", "general");
      return $oReturn;
    }

    // call vendor specific function
    $oReturn = call_user_func($vendorFunctionName, $ldap, $oReturn);

    closeConnection($ldap);

    logToFile("setCache", var_export($oReturn, true), $servername, 10);  
    return $oReturn;
  }

/**********************************************************************
  setBackEndDatabases

  Returns A List of backend databases in the format
  [0][0] = userRoot
  [0][1] = Name of database

  @param string $servername server name
  @return BackEndDatabases Array
**********************************************************************/
  function setBackEndDatabases($servername) {
    $returnList = Array();
    $oServer = setServer($servername);
    $vendor = $oServer->getVendor();
    $vendorFunctionName = vendorMapToFunction($vendor)."_setBackEndDatabases";

    // if not supported, exit before establishing ldap connection
    if(!function_exists($vendorFunctionName)) {
      return array();    
    }

    if(!$ldap = createConnection($servername)) {
      return array();
    }

    // call vendor specific function
    $returnList = call_user_func($vendorFunctionName, $ldap);

    closeConnection($ldap);

    logToFile("setBackEndDatabases", var_export($returnList, true), $servername, 10);  
    return $returnList;
  }

/**********************************************************************
  setIndexList

  Returns A List of indexes available for a specific database
  [0] = Index 1
  [1] = Index 2

  @param string $servername server name
  @return BackEndDatabases Array
**********************************************************************/
  function setIndexesList($servername, $backenddatabase) {
    if(empty($servername) || empty($backenddatabase)) {
      return array();
    }

    $returnList = Array();
    $oServer = setServer($servername);
    $vendor = $oServer->getVendor();
    $vendorFunctionName = vendorMapToFunction($vendor)."_setIndexesList";

    // if not supported, exit before establishing ldap connection
    if(!function_exists($vendorFunctionName)) {
      return array();    
    }

    if(!$ldap = createConnection($servername)) {
      return array();
    }

    // call vendor specific function
    $returnList = call_user_func($vendorFunctionName, $ldap, $backenddatabase);

    closeConnection($ldap);

    logToFile("setIndexesList", var_export($returnList, true), $servername, 10);  
    return $returnList;
  }

/**********************************************************************
  setIndex

  Returns Index Object
  @param array $serverlist server names to query
  @param string $backenddatabase backend database
  @param string $queryindex Index to query
  @return Index Object
**********************************************************************/
  function setIndex($serverlist, $backenddatabase, $queryindex) {
    $oReturn = new Index();
    if(empty($serverlist) || empty($backenddatabase)) {
      return $oReturn;
    }

    $oReturn->setIndex($queryindex);

    for($i = 0; $i < count($serverlist); $i++) {
      $oServer = setServer($serverlist[$i]);
      if($oServer->isAvailable()) {
        $vendor = $oServer->getVendor();
        $vendorFunctionName = vendorMapToFunction($vendor)."_setIndex";
        $oReturn->setHostNames($serverlist[$i]);
          
        if(!$ldap = createConnection($serverlist[$i])) {
          return array();
        }

        // call vendor specific function
        if(function_exists($vendorFunctionName)) {
          $oReturn = call_user_func($vendorFunctionName, $ldap, $serverlist[$i], $oReturn, $backenddatabase, $queryindex);
        }

        closeConnection($ldap);
        logToFile("setIndex", "set index list", $serverlist[$i], 10);  
      }
    }

    return $oReturn;
  }
  
/**********************************************************************
  setVendorInfo

  Set Vendor Name such as Sun or Red Hat directory
  The following function will get vendor version information

  @param string $servername server name
  @return VendorInfo object
**********************************************************************/
  function setVendorInfo($servername) {
    global $CONFIG;

    $FILE = new File();
    $vendorInfoFile = getTempFileName("vendorinfo_".$servername);

    if(file_exists($vendorInfoFile)) {
      $unserializedVendorInfo = $FILE->read($vendorInfoFile);
      $oReturn = unserialize($unserializedVendorInfo);
      // ok returning vendor info
      logToFile("setVendorInfo", "read cache: ".var_export($oReturn, true), $servername, 10);  
      return $oReturn;
    }
    $oReturn = new VendorInfo($servername);

    // should be secure
    $ldap = createConnection($servername);
    
    if(!$ldap) {
      return $oReturn;
    }
    
    $oServer = setServer($servername);
    $vendor = $oServer->getVendor();
    $vendorName = $oServer->getVendorName();
    $vendorFunctionName = vendorMapToFunction($vendor)."_setVendorInfo";

    // for vendor specific functions
    if(function_exists($vendorFunctionName)) {
      $oReturn = call_user_func($vendorFunctionName, $ldap, $oReturn);
    }
    else {
      // general function
      $ldap->set_base_dn("");
      $hResult = @$ldap->get_entry("", array("vendorName", "vendorVersion"));

      if($hResult) {
        $nRows = $ldap->num_rows($hResult);

        if($nRows == 1) {
          $aEntries = $ldap->get_result($hResult);
          // if version available
          if(isset($aEntries[0]["version"][0])) {
            $oReturn->setVendorVersion($aEntries[0]["version"][0]);
          }
          if(isset($aEntries[0]["vendorname"][0])) {
            $oReturn->setVendor($aEntries[0]["vendorname"][0]);
          }
          if(isset($aEntries[0]["vendorversion"][0])) {
            $oReturn->setVersion($aEntries[0]["vendorversion"][0]);
          }
        }
        else {
          $oReturn->setMessage("Invalid result. Unable to fetch vendor information");
        }
        $ldap->free_result($hResult);
      }
    }
    closeConnection($ldap);

    // if we couldn't get proper vendor info. Set from Server
    $returnVendorName = $oReturn->getVendor();
    if(empty($returnVendorName)) {
      $oReturn->setVendor($vendorName);
    }

    // finally create cache
    $serializedVendorInfo = serialize($oReturn);
    $FILE->write($vendorInfoFile, $serializedVendorInfo);
    logToFile("setVendorInfo", "write cache: ".var_export($oReturn, true), $servername, 10);  

    return $oReturn;
  }

/**********************************************************************
  setSchema

  Get Schema from LDAP Server

  @param string $servername Hostname
  @param int $enforcereload Enforces a reload of the schema cache
  @return Schema object
**********************************************************************/
  function setSchema($servername, $enforcereload = false) {
    global $CONFIG;
    $oReturn = new Schema();

    if(empty($servername)) {
      return $oReturn;
    }

    $FILE = new File();
    $cacheFile = getTempFileName("schema_".$servername);

    if(file_exists($cacheFile) && $enforcereload == false) {
      $unserializedSchema = $FILE->read($cacheFile);
      $oReturn = unserialize($unserializedSchema);
      logToFile("setSchema", "read cache: ".var_export($oReturn, true), $servername, 10);  
      // ok returning schema
      return $oReturn;
    }
    logToFile("setSchema", "load schema from server", $servername, 5);

    // create LDAP Connection
    if(!($ldap = createConnection($servername))) {
      closeConnection($ldap);
      $oReturn->setExitCode(1);
      $oReturn->setMessage("Unable to open ldap connection");
      return $oReturn;
    }

    // get root dn in order to get schema DN
    $returnAttributes = array("subschemaSubentry");
    $hResult = @$ldap->get_entry("", $returnAttributes);    

    if(!$hResult) {
      $oReturn->setExitCode(2);
      $oReturn->setMessage("Unable to fetch rootdn");
      closeConnection($ldap);
      return $oReturn;
    }

    $nRows = $ldap->num_rows($hResult);
    $oEntries = $ldap->get_result($hResult);
    $ldap->free_result($hResult);

    $schemadn = "";
    if($nRows == 1) {
      if(!empty($oEntries[0]["subschemasubentry"][0])) {
        $schemadn = $oEntries[0]["subschemasubentry"][0];      
      }
    }

    // if schema dn not found, exit...
    if(empty($schemadn)) {
      $oReturn->setExitCode(2);
      $oReturn->setMessage("Unable to fetch rootdn");
      closeConnection($ldap);
      return $oReturn;
    }
    $ldap->set_base_dn($schemadn);
    $hResult = @$ldap->get_entry($schemadn, array("attributetypes", "objectclasses", "+"));

    if(!$hResult) {
      $oReturn->setExitCode(2);
      $oReturn->setMessage("Unable to fetch schema");
      closeConnection($ldap);
      return $oReturn;
    }

    $nRows = $ldap->num_rows($hResult);
    $oEntries = $ldap->get_result($hResult);
    for($i = 0; $i < (count($oEntries) -1); $i++) {
      $oReturn->setAttributeType($oEntries[$i]["attributetypes"]);
      $oReturn->setObjectClass($oEntries[$i]["objectclasses"]);
      if(!empty($oEntries[$i]["matchingrules"])) {
        $oReturn->setMatchingRule($oEntries[$i]["matchingrules"]);
      }
    }

    $ldap->free_result($hResult);
    closeConnection($ldap);

    $serializedSchema = serialize($oReturn);
    $FILE->write($cacheFile, $serializedSchema);

    logToFile("setSchema", "write cache: ".var_export($oReturn, true), $servername, 10);  
    return $oReturn;
  }

/**********************************************************************
  setReplicationAgreement

  Get replication agreements, replication status

  @param string $servername Hostname
  @return ServerReplica object
**********************************************************************/
  function setReplicationAgreement($servername, $csnCheck = false) {
    global $CONFIG;

    $oServerReplica = new ServerReplica();

    if(empty($servername)) {
      $oServerReplica->setMessage(getLang('page.general.servernotavailable'));
      return $oServerReplica;
    }

    $oServer = setServer($servername);
    $vendor = $oServer->getVendor();
    $vendorFunctionName = vendorMapToFunction($vendor)."_setReplicationAgreement";

    // if not supported, exit before establishing ldap connection
    if(!function_exists($vendorFunctionName)) {
      $oServerReplica->setMessage("Vendor '".$vendor."' not supported", "general");
      return $oServerReplica;
    }

    $oServerReplica->setHostName($servername);
    // call vendor specific function
    $oServerReplica = call_user_func($vendorFunctionName, $oServerReplica, $servername, $csnCheck);

    logToFile("setReplicationAgreement", var_export($oServerReplica, true), $servername, 10);  
    return $oServerReplica;
  }

/**********************************************************************
  setReplicaInfo

  Set basic info about replica for replication purposes

  @param string $servername Hostname to check
  @param string $masterserver Hostname from check
  @param bool $useCache If we want to retrieve information from cache
  @return ReplicaInfo object
**********************************************************************/
  function setReplicaInfo($servername, $masterserver, $useCache = false) {
    global $CONFIG;

    $oReplicaInfo = new ReplicaInfo();

    $configId = getEnvironmentId($servername);
    $oServer = setServer($servername);
    $vendor = $oServer->getVendor();
    $vendorFunctionName = vendorMapToFunction($vendor)."_setReplicaInfo";

    // exit if vendor not supported before establishing ldap connection
    if(!function_exists($vendorFunctionName)) {
       $oReplicaInfo->setMessage("Not supported vendor: '".$vendor."'");
       logToFile("setReplicaInfo", var_export($oReplicaInfo, true), $servername, 10);  
       return $oReplicaInfo;
    }

    // if supplier not equal to server name we should check if we should
    // ignore replica look up
    if(strcasecmp($servername, $masterserver) != 0) {
      $configId = getEnvironmentId($masterserver);
      $ignoreReplicaStatus = getConfigValue("replication:ignoreconsumer", $configId[0], $configId[1]);
      if(!empty($ignoreReplicaStatus)) {
        return $oReplicaInfo;
      }
    }

    // replica info cache
    $FILE = new File();
    $cacheFile = getTempFileName("replicainfo_".$servername);

    if(file_exists($cacheFile) && $useCache) {
      $unserializedReplicaInfo = $FILE->read($cacheFile);
      $oReplicaInfo = unserialize($unserializedReplicaInfo);
      logToFile("setReplicaInfo", "read cache: ".var_export($oReplicaInfo, true), $servername, 10);
      // ok returning replica info
      return $oReplicaInfo;
    }
    $oReplicaInfo->setHostName($servername);

    // create LDAP Connection
    if(!($ldap = createConnection($servername))) {
      closeConnection($ldap);
      return $oReplicaInfo;
    }
    $oReplicaInfo = call_user_func($vendorFunctionName, $ldap, $oReplicaInfo);

    closeConnection($ldap);

    logToFile("setReplicaInfo", "write cache: ".var_export($oReplicaInfo, true), $servername, 10);
    $serializedReplicaInfo = serialize($oReplicaInfo);
    $FILE->write($cacheFile, $serializedReplicaInfo);

    return $oReplicaInfo;
  }

/**********************************************************************
  RootDN

  Get Root DN Object

  @param string $servername Hostname
  @return RootDN object
**********************************************************************/
  function setRootDN($servername, $returnAttributes = array()) {
    $currentRoot = new RootDN();

    $ldap = createConnection($servername);
    
    if(!$ldap) {
      return $currentRoot;
    }
    
    if(empty($returnAttributes)) {
      $returnAttributes = array("+",  
                                "namingcontexts",
                                "supportedextension",
                                "supportedcontrol",
                                "supportedfeatures",
                                "supportedsaslmechanisms",
                                "supportedldapversion",
                                "vendorname",
                                "netscapemdsuffix",
                                "vendorversion");
    }
    
    $currentRoot->setHostName($servername);

    // cn monitor (@ to hide ev. error message)
    $hResult = @$ldap->get_entry("", $returnAttributes);
    if($hResult) {
      $nRows = $ldap->num_rows($hResult);

      if($nRows > 0) {
        $aEntries = $ldap->get_result($hResult);
        if(!empty($aEntries[0]["namingcontexts"])) {
          $currentRoot->setNamingContexts($aEntries[0]["namingcontexts"]);
        }
        if(!empty($aEntries[0]["supportedextension"])) {
          $currentRoot->setSupportedExtension($aEntries[0]["supportedextension"]);
        }
        if(!empty($aEntries[0]["supportedcontrol"])) {
          $currentRoot->setSupportedControl($aEntries[0]["supportedcontrol"]);
        }
        if(!empty($aEntries[0]["supportedfeatures"])) {
          $currentRoot->setSupportedFeatures($aEntries[0]["supportedfeatures"]);
        }
        if(!empty($aEntries[0]["supportedsaslmechanisms"])) {
          $currentRoot->setSupportedSASLMechanisms($aEntries[0]["supportedsaslmechanisms"]);
        }
        if(!empty($aEntries[0]["supportedldapversion"])) {
          $currentRoot->setSupportedLDAPVersion($aEntries[0]["supportedldapversion"]);
        }
        if(!empty($aEntries[0]["vendorname"])) {
          $currentRoot->setVendorName($aEntries[0]["vendorname"][0]);
        }
        if(!empty($aEntries[0]["netscapemdsuffix"])) {
          $currentRoot->setNetscapemdsuffix($aEntries[0]["netscapemdsuffix"][0]);
        }
        if(!empty($aEntries[0]["vendorversion"])) {
          $currentRoot->setVendorVersion($aEntries[0]["vendorversion"][0]);
        }
      }
      else {
        $currentRoot->setMessage("Unable to fetch result");
      }
 
      $ldap->free_result($hResult);
    }

    closeConnection($ldap);

    $currentRoot->setExitCode(0);
    logToFile("setRootDN", var_export($currentRoot, true), $servername, 10);
    return $currentRoot;
  }
  
/**********************************************************************
  queryServer

  Query LDAP server

  @param string $servername server name
  @param string $basesuffix base suffix
  @param string $filter filter to use
  @param string $returnattributes array attributes to be returned
  @param int $QUERY_LIMITED_TO limit query to number of objects
  @return entries to be returned as array
**********************************************************************/
  function queryServer($servername, $basesuffix, $filter, $returnattributes, $QUERY_LIMITED_TO = 200) {
    global $CONFIG;

    // list of query objects
    $oReturn = Array();

    // if ping status == 0 just return default values
    if(empty($servername)) {
      return $oReturn;
    }

    $ldap = createConnection($servername);    
    $ldap->set_base_dn($basesuffix);
    // limit query result
    $ldap->setLDAPOption(LDAP_OPT_SIZELIMIT, $QUERY_LIMITED_TO);

    if(!$ldap->hConn) {
      $oQuery = new Query($servername);
      $oQuery->setMessage("Unable to open ldap connection");
      $oReturn[0] = $oQuery;
      closeConnection($ldap);
      return $oReturn;
    }

    // add operational attributes to query
    $returnattributes[count($returnattributes)] = "modifiersName";
    $returnattributes[count($returnattributes)] = "modifyTimestamp";
    $returnattributes[count($returnattributes)] = "creatorsName";
    $returnattributes[count($returnattributes)] = "createTimestamp";

    // @ to hide err msg:es
    $hResult = @$ldap->query($filter, $returnattributes);

    if(!$hResult) {
      $oQuery = new Query($servername);
      $oQuery->setMessage("Unable to fetch result");
      $oReturn[0] = $oQuery;
      closeConnection($ldap);
      return $oReturn;
    }

    $nRows = $ldap->num_rows($hResult);

    if($nRows > 0) {
      $aEntries = $ldap->get_result($hResult);
      
     // print_r($aEntries);
      
      for($i = 0; $i < $nRows; $i++) {
        $oQuery = new Query($servername);

        // get available attributes
        $aAttributes = $returnattributes;
        $oQuery->setDN($aEntries[$i]['dn']);
      
        for($j = 0; $j < count($aAttributes); $j++) {
          // we take for granted that all attributes are in lower case on entry
          $currentAttribute = strtolower($aAttributes[$j]);
          if(!empty($currentAttribute)) {
            // if requested attribute exists
            if(!empty($aEntries[$i][$currentAttribute])) {
              // to support multivalue            
              for($k = 0; $k < (count($aEntries[$i][$currentAttribute]) - 1); $k++) {
                if(strcasecmp($currentAttribute,"modifiersName") == 0) {
                  $oQuery->setModifiersName($aEntries[$i][$currentAttribute][$k]);
                }
                else if(strcasecmp($currentAttribute,"modifyTimestamp") == 0) {
                  $oQuery->setModifyTimestamp($aEntries[$i][$currentAttribute][$k]);
                }
                else if(strcasecmp($currentAttribute,"creatorsName") == 0) {
                  $oQuery->setCreatorsName($aEntries[$i][$currentAttribute][$k]);
                }
                else if(strcasecmp($currentAttribute,"createTimestamp") == 0) {
                  $oQuery->setCreateTimeStamp($aEntries[$i][$currentAttribute][$k]);
                }
                else {
                  $oQuery->setValue($currentAttribute, $aEntries[$i][$currentAttribute][$k]);
                } // value exists and can be added
              } // value exists
            }
          }
        } // each attribute
        $oReturn[count($oReturn)] = $oQuery;
      } // each entry
    }
    else {
      $oQuery = new Query($servername);
      $oQuery->setMessage("Empty result");
      $oReturn[0] = $oQuery;
    }

    $ldap->free_result($hResult);
    closeConnection($ldap);

    logToFile("queryServer", "returning result", $servername, 10);
    return $oReturn;
  }

/**********************************************************************
  getEntry

  get single LDAP Entry based on DN

  @param string $servername Servername
  @param string $dn Distinguished Name
  @return Query object
**********************************************************************/
  function getEntry($servername, $dn) {
    $returnattributes = Array();
    $oQuery = new Query($servername);

    // create LDAP Connection
    if(!($ldap = createConnection($servername))) {
      closeConnection($ldap);
      return $oQuery;
    }
    // Get Entry

    $hResult = @$ldap->get_entry("$dn");

    if($hResult) {
      $nRows = $ldap->num_rows($hResult);
      if($nRows > 0) {
        $aEntries = $ldap->get_result($hResult);
        // get available attributes
        $aAttributes = $ldap->get_attributes($ldap->get_first_entry($hResult));

        $oQuery->setDN($aEntries[0]['dn']);
      
        for($i = 0; $i < count($aAttributes); $i++) {
          // we take for granted that all attributes are in lower case on entry
          if(!empty($aAttributes[$i])) {
            $oQuery->setAttributes($aAttributes[$i]);
            $currentAttribute = strtolower($aAttributes[$i]);
            // to support multivalue
            for($j = 0; $j < (count($aEntries[0][$currentAttribute]) - 1); $j++) {
              $oQuery->setValue($currentAttribute, $aEntries[0][$currentAttribute][$j]);
            }
          }      
        }
        $ldap->free_result($hResult);
      
        // get additional operational attributes
        $returnattributes[count($returnattributes)] = "modifiersName";
        $returnattributes[count($returnattributes)] = "modifyTimestamp";
        $returnattributes[count($returnattributes)] = "creatorsName";
        $returnattributes[count($returnattributes)] = "createTimestamp";

        $hResult = @$ldap->get_entry("$dn", $returnattributes);

        if($hResult) {
          $aEntries = $ldap->get_result($hResult);
          if(!empty($aEntries[0]["modifiersname"][0])) {
            $oQuery->setModifiersName($aEntries[0]["modifiersname"][0]);
          }
          if(!empty($aEntries[0]["modifytimestamp"][0])) {
            $oQuery->setModifyTimestamp($aEntries[0]["modifytimestamp"][0]);
          }
          if(!empty($aEntries[0]["creatorsname"][0])) {
            $oQuery->setCreatorsName($aEntries[0]["creatorsname"][0]);
          }
          if(!empty($aEntries[0]["createtimestamp"][0])) {
            $oQuery->setCreateTimeStamp($aEntries[0]["createtimestamp"][0]);
          }
        }
        $ldap->free_result($hResult);
      }
      else {
        $oQuery->setMessage("Unable to get entry");
        $ldap->free_result($hResult);
      }
    }
    else {
      $oQuery->setMessage("Unable to get entry");
    }
    closeConnection($ldap);  
    logToFile("getEntry", "returning entry", $servername, 7);
    return $oQuery;
  }


/**********************************************************************
  verifyLDAPConnection

  verifies LDAP Connection

  @param object &$Server Reference to Server Object
  @return bool can connect to LDAP
**********************************************************************/
  function verifyLDAPConnection(&$oServer) {
    $servername = $oServer->getHostName();
    
    if(empty($servername)) {
      return false;
    }

    // create LDAP Connection
    if(!($ldap = createConnection($servername))) {
      closeConnection($ldap);
      return false;
    }

    // Get ROOT DSE
    $hResult = @$ldap->get_entry("", array("dn"));
    if(!$hResult) {
      logToFile("verifyLDAPConnection", ldap_err2str(ldap_errno($ldap->hConn)), $servername, 1);
      closeConnection($ldap);
      return false;
    }

    $nRows = $ldap->num_rows($hResult);
    $ldap->free_result($hResult);
    closeConnection($ldap);

    if($nRows == 1) {
      logToFile("verifyLDAPConnection", "Respond to LDAP", $servername, 5);
      return true;
    }
    logToFile("verifyLDAPConnection", "Do not Respond to LDAP", $servername, 5);
    return false;
  }

/**********************************************************************
  verifyLDAPServer

  Verifies What type of LDAP server we are running against
  Sets vendor information

  @param object &$Server Reference Server Object
  @return bool can connect to LDAP
**********************************************************************/
  function verifyLDAPServer(&$oServer) {
    $servername = $oServer->getHostName();
    if(empty($servername)) {
      return false;
    }
    $configId = getEnvironmentId($servername);

    // If vendorname and vendor manually configured
    $vendor = getConfigValue("vendor", $configId[0], $configId[1]);
    $vendorname = getConfigValue("vendorname", $configId[0], $configId[1]);
    if(!empty($vendor) && !empty($vendorname)) {
      $oServer->setVendor($vendor);
      $oServer->setVendorName($vendorname);
      return true;
    }
    
    $vendor = "";
    $vendorname = "";

    /** Legacy Support **/
    // microsoft ad
    $supported[0]["vendor"] = "microsoft";
    $supported[0]["vendorname"] = "Microsoft AD";
    $supported[0]["dn"] = "";
    $supported[0]["searchattr"] = "dsservicename";
    $supported[0]["searchvalue"] = "NTDS";

    // Oracle Internet Directory
    $supported[1]["vendor"] = "oid";
    $supported[1]["vendorname"] = "Oracle OID";
    $supported[1]["dn"] = "";
    $supported[1]["searchattr"] = "orcldirectoryversion";
    $supported[1]["searchvalue"] = "OID";

    // create LDAP Connection
    if(!($ldap = createConnection($servername))) {
      closeConnection($ldap);
      return false;
    }

    $testReturn = verifyLDAPServer_rootdn($ldap);
    
    if(!empty($testReturn[0])) {
      $vendor = $testReturn[0];
      $vendorname = $testReturn[1];
    }
    else { // legacy support
      // list thru possible vendors. Break on success
      for($i = 0; $i < count($supported); $i++) {
        $searchattribute = $supported[$i]["searchattr"];
        // @ to hide err msg:es
        $hResult = @$ldap->get_entry($supported[$i]["dn"],
                               array($searchattribute, "+"));
        if($hResult) {
          $nRows = @$ldap->num_rows($hResult);
          if($nRows == 1) {
            $oEntries = @$ldap->get_result($hResult);
            if(!empty($oEntries[0][$searchattribute][0])) {
              if(stripos($oEntries[0][$searchattribute][0],
                        $supported[$i]["searchvalue"]) !== FALSE) {
                $vendor = $supported[$i]["vendor"];
                $vendorname = $supported[$i]["vendorname"];
              } // if match
            } // if value not empty
          } // if rows 1
        } // if result
        @$ldap->free_result($hResult);
        if(!empty($vendor)) {
          break;
        } // if vendor
      }  //for      
    } // legacy support
    closeConnection($ldap);
    $oServer->setVendor($vendor);
    $oServer->setVendorName($vendorname);
    
    logToFile("verifyLDAPServer", "Respond to LDAP, vendor name retrieved", $servername, 5);
    return true;
  }

  /**
  * VerifyLDAPServer based on RootDN
  */
  function verifyLDAPServer_rootdn($ldap) {
    $return = Array();
    $hResult = @$ldap->get_entry("", array("version", "vendorname", "vendorversion", "objectclass"));
    if($hResult) {
      $nRows = @$ldap->num_rows($hResult);
      if($nRows == 1) {
        $oEntries = @$ldap->get_result($hResult);
        if(!empty($oEntries[0]["vendorname"][0])) {
          // check 1 RHDS
          if(stripos($oEntries[0]["vendorname"][0], "Fedora") !== FALSE) {
            $return = Array('fedora', 'Fedora DS');
          }
          else if(stripos($oEntries[0]["vendorname"][0], "389 Project") !== FALSE) {
            $return = Array('fedora', '389 DS');
          }
          else if(stripos($oEntries[0]["vendorname"][0], "Siemens") !== FALSE) {
            $return = Array('dirx', 'Siemens');
          }
          else if(stripos($oEntries[0]["vendorname"][0], "Red Hat") !== FALSE) {
            $return = Array('redhat', 'Red Hat DS');
          }
          else if(stripos($oEntries[0]["vendorname"][0], "CentOS") !== FALSE) {
            $return = Array('redhat', 'CentOS DS');
          }
          else if(stripos($oEntries[0]["vendorname"][0], "Netscape") !== FALSE) {
            $return = Array('netscape', 'Netscape');
          }
          else if(stripos($oEntries[0]["vendorname"][0], "Novell") !== FALSE) {
            $return = Array('novell', 'Novell');
          }
          else if(stripos($oEntries[0]["vendorname"][0], "Sun") !== FALSE) {
            // old sun versions, before Oracle 11.1.1.5.0
            $return = Array('sun', 'Sun Java System');
          }
          else if(stripos($oEntries[0]["vendorname"][0], "ForgeRock") !== FALSE) {
            // old sun versions, before Oracle 11.1.1.5.0
            $return = Array('opends', 'ForgeRock AS');
          }
          else if(stripos($oEntries[0]["vendorname"][0], "Oracle") !== FALSE) {
            // a bit tricky here as Oracle now owns both OID, ODSEE and OUD
            if(!empty($oEntries[0]["vendorversion"][0])) {
              // ODSEE still recognized as vendorversion Sun
              if(stripos($oEntries[0]["vendorversion"][0], "Sun") !== FALSE) {
                // Oracle Enterprise Directory
                $return = Array('sun', 'Oracle ODSEE');
              }
              // Oracle Unified Directory
              if(stripos($oEntries[0]["vendorversion"][0], "Oracle Unified Directory") !== FALSE) {
                // Oracle Enterprise Directory
                $return = Array('opends', 'Oracle UD');
              }
            }
          }
          else if(stripos($oEntries[0]["vendorname"][0], "IBM") !== FALSE) {
            $return = Array('ibm', 'IBM');
          }
          else if(stripos($oEntries[0]["vendorname"][0], "Apache Software Foundation") !== FALSE) {
            $return = Array('apacheds', 'ApacheDS');
          }
          
          // General OpenDS
          if(!empty($oEntries[0]["vendorversion"][0])) {
            if(stripos($oEntries[0]["vendorversion"][0], "OpenDS") !== FALSE) {
              $return = Array('opends', 'Sun OpenDS');
            }
          }
          // general Dirx
          if(!empty($oEntries[0]["vendorversion"][0])) {
            if(stripos($oEntries[0]["vendorversion"][0], "DirX") !== FALSE) {
              $return = Array('dirx', 'Siemens');
            }
          }
        }
        else if(!empty($oEntries[0]["objectclass"][0])) {
          // check objectclass
          if(in_array("OpenLDAProotDSE", $oEntries[0]["objectclass"])) {
            $return = Array('openldap', 'OpenLDAP');
          }
        }
          // check 1 RHDS
      } // if rows 1
    } // if result
      
    @$ldap->free_result($hResult);
    return $return;
  }
  
/**********************************************************************
  getExecutionTime

  Get Execution Time
  @param string $servername Hostname
  @return int number of milliseconds connecting to a server
**********************************************************************/
  function getExecutionTime($servername) {
    $starttime = microtimeFloat();
    $oServer = setServer($servername);
    verifyLDAPConnection($oServer);    
    $endtime = microtimeFloat();
    return round((($endtime - $starttime) * 1000), 1);
  }

/**********************************************************************
  setLoadBalancer

  Returns Load Balancer Object
  
  @param int $environmentid environment ID
  @param string $loadbalancer load balancer name
  @param string $ports ports to query in format 389:636
  @param int $nrOfQueries number of queries. Default 25
  @param string $lb_type load balancer type. By default host
  @return object LoadBalancer
**********************************************************************/
  function setLoadBalancer($environmentid, $loadbalancer, $ports, $nrOfQueries = 25, $lb_type = "host") {
    global $CONFIG;

    $oReturn = new LoadBalancer();
    $testPorts = explode(";", $ports);

    // variable containing working ports
    $aPorts = Array();

    // verify if the port is working or not
    for($i = 0; $i < count($testPorts); $i++) {
      if(socketOpen($loadbalancer, $testPorts[$i])) {
        $aPorts[count($aPorts)] = $testPorts[$i];
      }
      else {
        // we should repoort on each port. Both ports are verified further down
        $oReturn->setMessage("Unable to connect to server on selected port: " . $testPorts[$i]);
      }
    }
    
    if(empty($aPorts)) {
      $oReturn->setMessage("Unable to connect to server!\n" .
                           "Server: ".$loadbalancer.", port: " . implode(",", $testPorts));
      return $oReturn;
    }
    
    // verify DNS
    if($lb_type == "dns") {
      $oReturn->setStartTime(round(microtimeFloat() * 1000));
      for($i = 0; $i < $nrOfQueries; $i++) {        
        $ip = getServerIP($loadbalancer);
        $currentHost = getServerDNS($ip);
        $oReturn->setPort($currentHost);
        $oReturn->addHostMatch($currentHost);
        $responsetime = getExecutionTime($currentHost);
        $endtime = microtimeFloat();
        $oReturn->setResponseTime($responsetime, round(($endtime * 1000)), $currentHost, getPort($currentHost));
      }
      $oReturn->setEndTime(round(microtimeFloat() * 1000));
      return $oReturn;
    }

    $vendor = getConfigValue("loadbalancervendor", $environmentid);
    if(empty($vendor)) {
      $oServer = setServer($loadbalancer);
      $vendor = $oServer->getVendor();  
    }
    // here we do a match on either specific vendor type family functions or specific directory server functions
    $vendorFunctionNameSpecific = $vendor."_getHostNameOfServer";
    $vendorFunctionName = vendorMapToFunction($vendor)."_getHostNameOfServer";

    if(!function_exists($vendorFunctionNameSpecific) && !function_exists($vendorFunctionName)) {
      $oReturn->setMessage("Functionality not supported for Vendor ID: ".$vendor);
      return $oReturn;    
    }

    $currentPort = $aPorts[0];
    $oReturn->setStartTime(round(microtimeFloat() * 1000));
    for($i = 0; $i < $nrOfQueries; $i++) {
      $sCurrentHost = "";

      // set protocol and port
      if(count($aPorts) == 2) {
        if($currentPort == $aPorts[0]) {
          $currentPort = $aPorts[1];
        }
        else {
          $currentPort = $aPorts[0];
        }
      }

      // start responsetime
      $starttime = microtimeFloat();

      // open connection against loadbalancer, secure mode
      $ldap = createConnection($loadbalancer, $currentPort);

      if(function_exists($vendorFunctionNameSpecific)) {
        $sCurrentHost = call_user_func($vendorFunctionNameSpecific, $ldap);      
      }
      else if(function_exists($vendorFunctionName)) {
        $sCurrentHost = call_user_func($vendorFunctionName, $ldap);      
      }
     
      // close LDAP Connection
      closeConnection($ldap);

      // end response time
      $endtime = microtimeFloat();
      $responsetime = round((($endtime - $starttime) * 1000), 1);

      // add matching host
      if(!empty($sCurrentHost)) {
        $oReturn->addHostMatch($sCurrentHost);
        
        // add responsetime
        $oReturn->setResponseTime($responsetime, round(($endtime * 1000)), $sCurrentHost, $currentPort);

        // add host to port
        if($currentPort == $aPorts[0]) {
          $oReturn->setPort($sCurrentHost);
        }
        else {
          $oReturn->setSecondaryPort($sCurrentHost);
        }
      }
      else {
        // host not found, set warning message
        $oReturn->setMessage("Unable to retrieve hostname");
        // no need to continue as were not going anyware
        logToFile("setLoadBalancer", var_export($oReturn, true), $loadbalancer, 10);
        return $oReturn;
      }
    }
    $oReturn->setEndTime(round(microtimeFloat() * 1000));

    logToFile("setLoadBalancer", var_export($oReturn, true), $loadbalancer, 10);
    return $oReturn;
  }

/**********************************************************************
  setPerformanceCounterList

  Returns a list of available performance counters for selected vendor
  @param string $vendor Vendor type according to CN=Monitor
  @param string $byte_operation Byte operation to be displayed in kbyte_sent or mbyte_sent
  @return performance counter list [,]
**********************************************************************/
  function setPerformanceCounterList($vendor, $byte_operation = "kbyte_sent") {

    // here we do a match on either specific vendor type family functions or specific directory server functions
    $vendorFunctionName = vendorMapToFunction($vendor)."_setPerformanceCounterList";

    if(!function_exists($vendorFunctionName)) {
      // All servers supports exec_time
      return array('exec_time');
    }
    return call_user_func($vendorFunctionName, $byte_operation);      
  }

/**********************************************************************
  setCertificate

  Returns a Certificate object

  @param string $servername server name
  @param bool $useCache use cache if available
  @return object Certificate
**********************************************************************/
  function setCertificate($servername, $useCache = true) {
    $port = getSecurePort($servername);
    if(socketOpen($servername, $port)) {
      return new Certificate($servername, $port, $useCache);
    }
    else {
      // we should still return a empty object
      return new Certificate("", $port);
    }
  }

  // get php5 behaviour in php4
  // microtime returns 0.1231313 12123 in php4
  function microtimeFloat() {
    list($usec, $sec) = explode(" ", microtime());
    return ((float)$usec + (float)$sec);
  }

  // returns 0 if value lower than offset
  function returnValueOnOffset($nValue, $nOffset) {
    if($nOffset > 0) {
      if($nValue <= $nOffset) {
        return 0;
      }
      else {
        return ($nValue - $nOffset);
      }
    }
    return $nValue;
  }

/**********************************************************************
  getPort

  Get configured LDAP port for this server. Default is 389

  @param string $servername server name
  @return int port
**********************************************************************/
  function getPort($servername) {

    // if a coding mistake is done
    if(empty($servername)) {
      return 389;
    }
  
    $envId = getEnvironmentId($servername);

    // can happen if alias is not set, return standard port
    if(empty($envId)) {
      return 389;
    }
    $port = getConfigValue("port", $envId[0], $envId[1]);

    if(empty($port)) {
      $port = 389;
    }
    return $port;
  }

/**********************************************************************
  verifyReturnedHost

  Verify returned host from directory. The host might be an alias.
  I.e. servername DNS A-Record: ldaphost.mydomain.com and replication over CNAME ldaphost.mydomainx.com

  @param string $servername server name
  @return string server name
**********************************************************************/
  function verifyReturnedHost($servername) {
    $configId = getEnvironmentId($servername);
    // if configId the server can have returned an alias
    if(empty($configId)) {
      $returnedServer = queryConfigByWildcard($servername, array("alias"));

      // if we've found one server matching the alias and only one server
      if(count($returnedServer) == 1) {
        $servername = $returnedServer[0];
      }
    }

    return $servername;
  }

/**********************************************************************
  getSecurePort

  Get Secure port for server. If not configured use port 636

  @param string $servername server name
  @return int port
**********************************************************************/
  function getSecurePort($servername) {

    // if a coding mistake is done
    if(empty($servername)) {
      return 636;
    }

    $envId = getEnvironmentId($servername);
    $port = getConfigValue("secureport", $envId[0], $envId[1]);

    if(empty($port)) {
      $port = 636;
    }
    return $port;
  }
  
/**********************************************************************
  convertLDAPDate

  Convert LDAP Date to Unix timestamp

  @param string $inValue LDAP Date
  @return int number of seconds since UNIX epoche time
**********************************************************************/
  function convertLDAPDate($inValue) {

    $year = substr($inValue,0,2);
    $month = substr($inValue,2,2);
    $day = substr($inValue,4,2);
    $hour = substr($inValue,6,2);
    $minute = substr($inValue,8,2);
    $second = substr($inValue,10,2);
    
    return mktime($hour, $minute, $second, $month, $day, $year);
  }

/**********************************************************************
  getFirstRespondingServerInEnvironment

  Get first responding server in environment

  @param int $environmentid Environment ID. Default 0
  @param int $serverid Server ID. Default 0
  @return string server name
**********************************************************************/
  function getFirstRespondingServerInEnvironment($environmentid = null, $serverid = null) {

    if(!isset($environmentid)) {
      return "";
    }

    // if serverid is available. Only check this server
    if(!empty($serverid)) {
      $oServer = setServer(getServerName($serverid, $environmentid));
      if($oServer->isAvailable()) {
        return $oServer->getHostName();
      }
      else {
        return "";
      }
    }
    
    $env_servers = getServers($environmentid);
    for($i = 0; $i < count($env_servers); $i++) {
      // get server object
      $oServer = setServer($env_servers[$i][1]);
      if($oServer->isAvailable()) {
        if($oServer->getVendor() != "unknown") {
          return $oServer->getHostName();
        } // only accept known vendors
      } // if ldap is available
    }
    return "";
  }

/**********************************************************************
  getTextByServerMessageKey

  Get first responding server in environment

  @param string $messagekey Server Message Key i.e. start_time
  @return string Message Text
**********************************************************************/
  function getTextByServerMessageKey($messageKey) {
    if(trim($messageKey) == "start_time") {
      return getLang('page.general.serverstarted');
    }
    else if(trim($messageKey) == "port_unavailable") {
      return getLang('page.general.notansweringonport');
    }
    else if(trim($messageKey) == "ldap_unavailable") {
      return getLang('page.general.notansweringonldap');
    }
    else if(trim($messageKey) == "ldap_available") {
      return getLang('page.general.ldaponline');
    }
    else if(trim($messageKey) == "ldap_replication") {
      return getLang('page.general.replicationfailure');
    }
    else if(trim($messageKey) == "ldap_restart") {
      return getLang('page.general.serverstarted');
    }
    else if(trim($messageKey) == "collectservermessage") {
      return getLang('page.general.collectservermessage');
    }
    else if(trim($messageKey) == "collectdb") {
      return getLang('page.general.collectdb');
    }
    else if(trim($messageKey) == "collectsummary") {
      return getLang('page.general.collectsummary');
    }
    else if(trim($messageKey) == "collectmailreport") {
      return getLang('page.general.collectmailreport');
    }
    return "";
  }
  
/**********************************************************************
  vendorMapToFunction

  Maps vendor type to function vendor type (ex fedora part of the iplanet functions)
  @param string $vendor Vendor type according to CN=Monitor
  @return vendor function ID
**********************************************************************/
  function vendorMapToFunction($vendor) {
    if(($vendor == "fedora") ||
       ($vendor == "redhat") ||
       ($vendor == "netscape") ||
       ($vendor == "sun")) {
       return "iplanet";
    }
    return $vendor;
  }
  
?>