<?php
  /*
    Copyright (C) 2009-2012 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

  /*****************************************************************
    Config Functions
    080811 Andreas Andersson Created
  *****************************************************************/
  date_default_timezone_set('UTC');
   /***********************
     Get a config or session value for hosts, environments and general
     - Host level $CONFIG["env"][<id>][<hostid>][<keyname>]
     - Environment level $CONFIG["env"][<id>][<keyname>]
     - General level $CONFIG["general"]["env"][<keyname>]
     @param string key name
     @param string environment id
     @param string host id
   ***/
   function getConfigValue($key, $env_id = -1, $host_id = -1) {
     global $CONFIG;
     global $_SESSION;

     // session prioritized over config
     if(isset($_SESSION["env"][$env_id][$host_id][$key])) {
       return $_SESSION["env"][$env_id][$host_id][$key];
     }
     else if(isset($CONFIG["env"][$env_id][$host_id][$key])) {
       return $CONFIG["env"][$env_id][$host_id][$key];
     }
     else if(isset($_SESSION["env"][$env_id][0][$key])) {
       return $_SESSION["env"][$env_id][0][$key];
     }
     else if(isset($CONFIG["env"][$env_id][0][$key])) {
       return $CONFIG["env"][$env_id][0][$key];
     }
     else if(isset($_SESSION["general"]["env"][$key])) {
       return $_SESSION["general"]["env"][$key];
     }
     else if(isset($CONFIG["general"]["env"][$key])) {
       return $CONFIG["general"]["env"][$key];
     }
     else {
       return "";
     }
   }

   /***********************
     Set runtime value for hosts, environments and general
     - Host level $_SESSION["env"][<id>][<hostid>][<keyname>]
     - Environment level $_SESSION["env"][<id>][<keyname>]
     - General level $_SESSION["general"]["env"][<keyname>]
     @param string key name
     @param string environment id
     @param string host id
   ***/
   function setConfigValue($key = "", $value = "", $env_id = -1, $host_id = -1) {
     global $CONFIG;
     global $_SESSION;
     
     if(empty($key)) {
       return false;
     }
     
     // if empty value. Remove key
     if(empty($value)) {
       delConfigValue($key, $env_id, $host_id);
     }
     
     // if host level
     if(($env_id >= 0) && ($host_id >= 0)) {
       $_SESSION["env"][$env_id][$host_id][$key] = $value;
     }

     // if env level
     if(($env_id >= 0) && ($host_id < 0)) {
       $_SESSION["env"][$env_id][0][$key] = $value;
     }

     // if general level
     if(($env_id < 0) && ($host_id < 0)) {
       $_SESSION["general"]["env"][$key] = $value;
     }

     return true;
   }

   /***********************
     Remove runtime value for hosts, environments and general
     - Host level $_SESSION["env"][<id>][<hostid>][<keyname>]
     - Environment level $_SESSION["env"][<id>][<keyname>]
     - General level $_SESSION["general"]["env"][<keyname>]
     @param string key name
     @param string environment id
     @param string host id
   ***/
   function delConfigValue($key = "", $env_id = -1, $host_id = -1) {

     if(empty($key)) {
       return false;
     }

     // if host level
     if(($env_id >= 0) && ($host_id >= 0)) {
       if(isset($_SESSION["env"][$env_id][$host_id][$key])) {
         unset($_SESSION["env"][$env_id][$host_id][$key]);
       }
     }

     // if env level
     if(($env_id >= 0) && ($host_id < 0)) {
       if(isset($_SESSION["env"][$env_id][0][$key])) {
         unset($_SESSION["env"][$env_id][0][$key]);
       }
     }

     // if general level
     if(($env_id < 0) && ($host_id < 0)) {
       if(isset($_SESSION["general"]["env"][$key])) {
         unset($_SESSION["general"]["env"][$key]);
       }
     }  
   }

   /***********************
     Get total configuration including session based environments
     Only use this assembeling function to get config list with names ONLY
     @return array environment and server list $CONFIG["env"][0][0]["name"]
  ***/
  function getTotalConfigEnvironments() {
    global $CONFIG;
    global $_SESSION;
    $aReturn = Array();
    
    if(empty($CONFIG["env"])) {
      return $aReturn;
    }
    
    for($i = 0; $i < count($CONFIG["env"]); $i++) {
      for($j = 0; $j < count($CONFIG["env"][$i]); $j++) {
        $aReturn["env"][$i][$j]["name"] = $CONFIG["env"][$i][$j]["name"];
        if(!empty($CONFIG["env"][$i][$j]["loadbalancer"])) {
          $aReturn["env"][$i][$j]["loadbalancer"] = $CONFIG["env"][$i][$j]["loadbalancer"];
        }
      } // for each server
    }

    if(!empty($_SESSION["envadd"])) {
      for($i = 0; $i < count($_SESSION["envadd"]); $i++) {
        $nMaxEnvironments = count($aReturn["env"]);
        for($j = 0; $j < count($_SESSION["envadd"][$i]); $j++) {
          $aReturn["env"][$nMaxEnvironments][$j]["name"] = $_SESSION["envadd"][$i][$j]["name"];
        }
      }
    }
    
    return $aReturn;
    
  }
  
   /***********************
     Query configuration by wildcard string
     @param string $str string to query
     @param string array $array options to query
     @return integer
  ***/
  function queryConfigByWildcard($querytext, $options = Array('name', 'alias', 'url', 'description', 'loadbalancer')) {
    $CONFIG = getTotalConfigEnvironments();
    
    $aReturn = array();
    
    // for each environment
    for($i = 0; $i < count($CONFIG["env"]); $i++) {
      // for each server in environment
      for($j = 0; $j < count($CONFIG["env"][$i]); $j++) {

        // for each of the options
        for($k = 0; $k < count($options); $k++) {
          $currentOptionValue = getConfigValue($options[$k], $i, $j);
          $currentHost = $CONFIG["env"][$i][$j]["name"];

          // if we have a match based on optionvalue and query text
          if(strripos($currentOptionValue, $querytext) !== FALSE) {
            // not a duplicate add to list
            if(!in_array($currentOptionValue, $aReturn)) {
              $aReturn[count($aReturn)] = $currentHost;
            } // duplicate check
          } // matching name
          
        } // for each option
        
      } // each server
    } // each environment
    return $aReturn;
  }
  
   /***********************
     Get the total number of static config environments
     @return integer
  ***/
  function getStaticEnvironmentConfigID() {
    global $CONFIG;
    return count($CONFIG["env"]);
  }
  
   /***********************
     Get environment and server id by server name or loadbalancer
     returns Array(environment_id, server_id)

     @param string server name
  ***/
  function getEnvironmentId($server) {

    // rewrite config to include sessions
    $CONFIG = getTotalConfigEnvironments();
    
    $aReturn = Array();

    if(!empty($aReturn[0])) {
      return $aReturn;
    }

    for($i = 0; $i < count($CONFIG["env"]); $i++) {
      for($j = 0; $j < count($CONFIG["env"][$i]); $j++) {
        if($server == $CONFIG["env"][$i][$j]["name"]) {
          $aReturn[0] = $i;
          $aReturn[1] = $j;
          break;
        }
        if(!empty($CONFIG["env"][$i][$j]["loadbalancer"])) {
          if($server == $CONFIG["env"][$i][$j]["loadbalancer"]) {
            $aReturn[0] = $i;
            $aReturn[1] = $j;
            break;
          }
        }
      } //for
      if(!empty($aReturn[0])) {
        break;
      }
    } // for
        
    return $aReturn;
  }

  
   /***********************
     Get available environments
  ***/
  function getEnvironments($option = "", $optionExists = true) {

    // rewrite config to include sessions
    $CONFIG = getTotalConfigEnvironments();
    $env = array();

    if(empty($CONFIG["env"])) {
      return $env;
    }

    $currentId = -1;
    for($i = 0; $i < count($CONFIG["env"]); $i++) {
      $addValue = true;
    
      // if still in same environment
      if($currentId == $i) {
        continue;
      }
      
      // handle optional option
      if(!empty($option)) {
        $configValue = getConfigValue($option, $i, 0);      
        if(!empty($configValue) == $optionExists) {
          $addValue = true;
        }
        else {
          $addValue = false;
        }
      }

      if($addValue) {
        $arrayId = count($env);
        $env[$arrayId] = array();
        $env[$arrayId][0] = $i;
        $env[$arrayId][1] = $CONFIG["env"][$i][0]["name"];
        $currentId = $i;
      }
    }
    return $env;
  }

   /***********************
     Get available environments
     @param $environmentid optional
     @return array serverid, servername, environmentid
     
  ***/
  function getServers($environmentId = -1) {

    // rewrite config to include sessions
    $CONFIG = getTotalConfigEnvironments();

    if($environmentId != -1) {
      $envStartId = $environmentId;
      $envEndId = $environmentId + 1;
    }
    else {
      // all environments
      $envStartId = 0;
      $envEndId = count($CONFIG["env"]);
    }
    
    $servers = array();
    for($j = $envStartId; $j < $envEndId; $j++) {
      for($i = 1; $i < count($CONFIG["env"][$j]); $i++) {
        $currentId = count($servers);
        $servers[$currentId] = array();
        $servers[$currentId][0] = $i;
        $servers[$currentId][1] = $CONFIG["env"][$j][$i]["name"];
        $servers[$currentId][2] = $j;
      }
    }
    return $servers;
  }

  /***********************
     Get Servername based on ID
  ***/
  function getServerName($serverId = "", $environmentId = "") {

    // rewrite config to include sessions
    $CONFIG = getTotalConfigEnvironments();

    $returnValue = "";
    
    if(empty($serverId) && empty($environmentId)) {
      $serverId = getFormValue("serverid");
      $environmentId = getFormValue("environmentid");
    }

    if(isset($serverId) && isset($environmentId)) {
      if(empty($CONFIG["env"][$environmentId][$serverId]["name"])) {
        return "";
      }
      return $CONFIG["env"][$environmentId][$serverId]["name"];
    }
    return "";
  }
  
  /***********************
     Get Environment based on ID
  ***/
  function getEnvironmentName($environmentId = "") {

    // rewrite config to include sessions
    $CONFIG = getTotalConfigEnvironments();

    $returnValue = "";

    if(empty($environmentId)) {
      $environmentId = getFormValue("environmentid");
    }

    if(isset($environmentId)) {
      return $CONFIG["env"][$environmentId][0]["name"];
    }
    return "";
  }

   /***********************
     Get available servers unique
     Note. This function will only return the actual servernames
  ***/
  function getServersUnique($environmentid = -1) {
    $env_servers = getServers($environmentid);
    $servers = Array();
    for($i = 0; $i < count($env_servers); $i++) {
      if(!in_array($env_servers[$i][1], $servers)) {
        $servers[count($servers)] = $env_servers[$i][1];
      }
    }
    return $servers;
  }

  /***********************
     Get status if function has been disabled
  ***/
  function functionIsEnabled($sFunction) {
    global $CONFIG;
    if(!empty($CONFIG["general"]["security"]["disable"])) {
      if(in_array($sFunction, explode(",", $CONFIG["general"]["security"]["disable"]))) {
        return false;
      }
    }
    return true;
  }
    
  /***********************
     Get GET or POST variables
  ***/
  function getFormValue($param) {
    global $_GET;
    global $_POST;

    // post always prioritized
    if(isset($_POST[$param])) {
      return $_POST[$param];
    }
    
    if(isset($_GET[$param])) {
      return $_GET[$param];
    }
    
    return null;
  }
    
  /***********************
    Get text based on language
  ***/
  function getLang($key) {
    global $LANGUAGE;

    if(isset($LANGUAGE[$key])) {
      return $LANGUAGE[$key];
    }
    
    return "&lt;Unknown Text&gt;";
  }

  /***********************
    Get Language Code
  ***/
  function getLanguage() {
    global $CONFIG;

    if(!empty($_SESSION["general"]["language"])) {
      return $_SESSION["general"]["language"];
    }
    else if(empty($CONFIG["general"]["language"])) {
      return "en";
    }
    return $CONFIG["general"]["language"];
  }

  /***********************
    Set Language Code
  ***/
  function setLanguage($language) {
    global $CONFIG;

    $_SESSION["general"]["language"] = $language;
  }

  /***********************
    Get TimeZone offset compared to UTC
  ***/
  function get_timezone_offset($inTimeZone = "") {
    global $CONFIG;

    // timezone not available. Check configuration
    if(empty($inTimeZone)) {
      $timeZone = getConfigValue("timezone");
      $timeShift = (getConfigValue("timeshift") * 3600);
      if(empty($timeShift)) {
        $timeShift = 0;
      }
      if(empty($timeZone)) {
        $timeZone = "UTC";
      }
    }
    else {
      // use input timezone
      $timeZone = $inTimeZone;
      $timeShift = 0;
    }
    
    // timezone as UTC, no offset is needed
    if(strtoupper($timeZone) == "UTC") {
      return $timeShift;
    }

    if(phpversion() < 5.2) {
      // for php 5.1 > 5.2 not supporting the DateTimeZone object
      // a bit dirty solution to get the offset between two time zones
      // get date + time in UTC
      date_default_timezone_set('UTC');
      $sDateUTC = date("Y-m-d H:i:s");
      // get date + time in configured time zone
      date_default_timezone_set($timeZone);
      $sDateConfiguredTZ = date("Y-m-d H:i:s");
      // parse the result, set date time as UTC
      $nConfiguredSeconds = (strtotime($sDateConfiguredTZ." UTC") - strtotime($sDateUTC." UTC"));    
      date_default_timezone_set('UTC');
      return ($nConfiguredSeconds + $timeShift);
    }
    else {
      // php version at least v. 5.2
      try {
        $oConfiguredDateTZ = new DateTimeZone($timeZone);
        $oConfiguredTime = new DateTime("now", $oConfiguredDateTZ);
        // get offset based on UTC to configured timezone in seconds
        $nConfiguredSeconds = $oConfiguredDateTZ->getOffset($oConfiguredTime);
        return ($nConfiguredSeconds + $timeShift);
      }
      catch (Exception $e) {
        logToFile("get_timezone_offset", "Invalid TimeZone: ".$timeZone, "", 0);
        return $timeShift;
      }
    }
    // fallback to return timeshift    
    return $timeShift;
  } 
  
  /***********************
    Returns available language prefixes
  ***/
  function getLanguageList() {
    $returnList = Array();
    foreach (glob("language/*.php") as $filename) {
      $filename = str_replace(array("language/", ".php"), "", $filename);
      $returnList[count($returnList)] = $filename;
    } // foreach file
    return $returnList;
  }

  /***********************
    Get and create Temp directory
  ***/
  function getTempDirectory() {

    // get temp directory ending with cnmonitor
    if (function_exists('sys_get_temp_dir')) {
      $tempDirectory = realpath(sys_get_temp_dir())."/cnmonitor";
    }
    else {
      if (!empty($_ENV['TMP'])) { 
        $tempDirectory = realpath($_ENV['TMP'])."/cnmonitor"; 
      }
      else if (!empty($_ENV['TEMP'])) { 
        $tempDirectory = realpath($_ENV['TEMP'])."/cnmonitor"; 
      }
      else if (!empty($_ENV['TMPDIR'])) { 
        $tempDirectory = realpath($_ENV['TMPDIR'])."/cnmonitor"; 
      }
      else {
        $tempDirectory = "/tmp/cnmonitor";
      }
    }
    
    $sessionid = session_id();
    if(empty($sessionid)) {
      $tempDirectory .= "cli";
    }

    if(!empty($CONFIG["general"]["tempdir"])) {
      $tempDirectory = $CONFIG["general"]["tempdir"];
    }
    
    if(!file_exists($tempDirectory)) {
      // we can't continue
      if(!mkdir($tempDirectory, 0700, true)) {
        trigger_error("FATAL ERROR: Unable to create temp directory '".$tempDirectory."'!", E_USER_ERROR);
      }
    }
    return $tempDirectory;
  }

  /***********************
    Get temporary filename to use
    filename should be in the following format:
    <temp directory>/<sessionid>_<added info.tmp>
  ***/
  function getTempFileName($partoffile) {
    $sessionid = session_id();
    if(empty($sessionid)) {
      $sessionid = "cli";
    }
    return getTempDirectory()."/".$sessionid."_".$partoffile.".tmp";
  }

  function existTempFileName($partoffile) {
    $sessionid = session_id();
    if(empty($sessionid)) {
      $sessionid = "cli";
    }
    if(file_exists(getTempDirectory()."/".$sessionid."_".$partoffile.".tmp")) {
      return true;
    }
    return false;
  }

  /***********************
    Remove file in the following format:
    <temp directory>/<sessionid>_<added info.tmp>
    empty file means removing all files in directory for this session
  ***/
  function removeTempFileName($partoffile = "") {
    $sessionid = session_id();

    if(empty($sessionid)) {
      $sessionid = "cli";
    }

    if(!empty($partoffile)) {
      $filename = getTempDirectory()."/".$sessionid."_".$partoffile.".tmp";
      if(file_exists($filename)) {
        unlink($filename);
      }
    }
    else {
      // removes all files in directory
      foreach (glob(getTempDirectory()."/".$sessionid."_*.tmp") as $filename) {
        unlink($filename);
      } // foreach file
    }
  }

  /***********************
    Remove old temporary files
    Older than three weeks (21 days)
  ***/
  function removeOldTempFiles() {
    $dateTimestamp = strtotime("-21 day");
    // to be sure that we get <tmp>/cnmonitor and not <tmp>/cnmonitorcli
    $tmpDirectory = str_replace("cli", "", getTempDirectory());
    if(!is_dir($tmpDirectory)) {
      return;
    }
    foreach (glob($tmpDirectory."/*.tmp") as $filename) {
      if(file_exists($filename)) {
        $fileTime = filectime($filename);
        if($dateTimestamp > $fileTime) {
          unlink($filename);
        }
      } // file exists
    } // foreach file  
  }

  /***********************
    Get Running Mobile Status
  ***/
  function isRunningMobile() {
    $mobileStatus = getConfigValue("mobile");
    if(empty($mobileStatus)) {
      $userAgent = $_SERVER['HTTP_USER_AGENT'];
      if(stripos($userAgent, "iphone") !== FALSE) {
        $mobileStatus = "on";
      }
      else if(stripos($userAgent, "ipod") !== FALSE) {
        $mobileStatus = "on";
      }
      else if(stripos($userAgent, "android") !== FALSE) {
        $mobileStatus = "on";
      }
    }
    else if($mobileStatus != "on") {
      $mobileStatus = "off";
    }
    return $mobileStatus;
  }
  
  /***********************
    Get page based on input variables
  ***/
  function getPage() {
    $environmentId = getFormValue("environmentid");
    $serverId = getFormValue("serverid");
    $pageName = getFormValue("page");

    // for security reasons
    $pageName = str_replace(array("..", "/"), "", $pageName);

    // the three general pages  
    // isset for environment id as environmentid can be 0
    if(!isset($environmentId) &&
       empty($serverId)) {
       if($pageName == "queryconfig.php") {
         return "queryconfig.php";
       }
       return "environments.php";
    }

    // isset for environment id as environmentid can be 0
    if(isset($environmentId) &&
       empty($serverId)) {
       if($pageName == "servergraphenv.php") {
         return "servergraph.php";
       }
       else if($pageName == "loadbalancerenv.php") {
         return "loadbalancer.php";
       }
       else if($pageName == "queryenv.php") {
         return "query.php";
       }
       else if($pageName == "cacheenv.php") {
         return "cache.php";
       }
       else if($pageName == "indexesenv.php") {
         return "indexes.php";
       }
       else if($pageName == "certificateenv.php") {
         return "certificate.php";
       }
       else if($pageName == "schemaenv.php") {
         return "schema.php";
       }
       else if($pageName == "replicationenv.php") {
         return "replication.php";
       }
       return "environment.php";
    }

    if(isset($environmentId) &&
       !empty($serverId)) {
       if(!empty($pageName)) {
         return $pageName;
       }
       return "server.php";
    }
    return "default.php";
  }

  /***********************
    Get page buttons based on page
  ***/
  function getButtons($page) {

    // for security reasons
    $page = str_replace(array("..", "/"), "", $page);

    if($page == "environments.php") {
      return "default.php";
    }
    else if($page == "environment.php" || 
            $page == "servergraphenv.php" ||
            $page == "queryenv.php" ||
            $page == "loadbalancerenv.php" ||
            $page == "cacheenv.php" ||
            $page == "indexesenv.php" ||
            $page == "certificateenv.php" ||
            $page == "schemaenv.php" ||
            $page == "replicationenv.php") {
      return "environment.php";
    }
    else if($page == "server.php" 
            || $page == "query.php" 
            || $page == "servergraph.php"
            || $page == "replication.php"
            || $page == "loadbalancer.php"
            || $page == "cache.php"
            || $page == "schema.php"
            || $page == "certificate.php"
            || $page == "indexes.php") {
      return "server.php";
    }
    
    return "default.php";
  }

  /***********************
    Get page tabs based on page
  ***/
  function getTabs($page) {

    // for security reasons
    $page = str_replace(array("..", "/"), "", $page);

    if($page == "environments.php") {
      return "environments.php";      
    }
    else if($page == "environment.php") {
      return "environment.php";      
    }
    else if($page == "server.php") {
      return "server.php";
    }
    else if($page == "query.php") {
      return "query.php";
    }
    else if($page == "certificate.php") {
      return "certificate.php";
    }
    else if($page == "schema.php") {
      return "schema.php";
    }
    else if($page == "replication.php") {
      return "replication.php";
    }
    else if($page == "cache.php") {
      return "cache.php";
    }
    else if($page == "indexes.php") {
      return "indexes.php";
    }
    else if($page == "servergraph.php") {
      return "servergraph.php";
    }
    else if($page == "loadbalancer.php") {
      return "loadbalancer.php";
    }
    else if($page == "servergraphenv.php") {
      return "servergraph.php";      
    }
    else if($page == "queryenv.php") {
      return "query.php";
    }
    else if($page == "cacheenv.php") {
      return "cache.php";
    }
    else if($page == "indexesenv.php") {
      return "indexes.php";
    }
    else if($page == "certificateenv.php") {
      return "certificate.php";
    }
    else if($page == "schemaenv.php") {
      return "schema.php";
    }
    else if($page == "replicationenv.php") {
      return "replication.php";
    }
    else if($page == "loadbalancerenv.php") {
      return "loadbalancer.php";
    }
    
    return "default.php";
  }

  /***********************
    Build Configuration based on XML file
  ***/
  function setXMLConfiguration($xmlFile) {
    global $CONFIG;

    // on the safe side:
    if(!function_exists("simplexml_load_string")) {
      echo "<h3>PHP Function missing: simplexml_load_string</h3>\n";
      echo "Unable to use XML based configuration file as required PHP function is missing.<br>\n";
      echo "Add PHP XML support or use a configuration file based on PHP arrays.<br>\n";
      exit(0);
    }

    if(!file_exists($xmlFile)) {
      echo "<h3>Configuration file does not exist</h3>\n";
      echo "Can't open file '$xmlFile'.<br>\n";
      exit(0);
    }
    
    $sXML = file_get_contents($xmlFile);
    if(($oXML=@simplexml_load_string($sXML)) === FALSE) {
      echo "<h3>Error loading configuration file</h3>\n";
      echo "Unable to load XML based configuration file.<br>\n";
      echo "Check if you are using a malformed XML file!<br>\n";
      exit(0);
    }
  
    $environmentid = 0;

    // start from the top
    foreach ($oXML->children() as $oChildren) {
      // general settings
      if($oChildren->getName() == "general") {
        setXMLConfigChildren("general", "", $oChildren);
      }
      if($oChildren->getName() == "environment") {
        setXMLConfigChildren("environment", "", $oChildren, true, $environmentid++);
      }
    }
    
    // finally a dirty hack as database config entry is renamed to "db"
    if(!empty($CONFIG["general"]["db"]["db"])) {
      $CONFIG["general"]["db"]["database"] = $CONFIG["general"]["db"]["db"];
      unset($CONFIG["general"]["db"]["db"]);
    }
  }

  /***********************
    For backward compatibility. XML -> PHP vars
  ***/
  function xmlCompat($value) {
    if($value == "environment") {
      $value = "env";
    }
    if($value == "database") {
      $value = "db";
    }

    // replace _ to :
    //$value = str_replace("_", ":", $value);
    
    return $value;
  }

  /***********************
    Build Configuration based on XML file
  ***/
  function setXMLConfigChildren($configbase, $configsubbase, $oChildren, $isenvironment = false, $environmentid = 0, $serverid = 0) {
    global $CONFIG;

    foreach($oChildren->children() as $oChild) {
      $option = trim("".$oChild->getName());
      $value = trim("".$oChild);

      // if encrypted password
      if($option == "encpassword") {
        $option = "password";
        $value = decryptPassword($value);
      }
      
      // empty value could mean 1) subset 2) actual empty value
      if(empty($value)) {
        // could be a sub set
        if(count($oChild->children())) {
          if($isenvironment) {
            $serverid++;
            setXMLConfigChildren($configbase, "", $oChild, $isenvironment, $environmentid, $serverid);
          }
          else {
            setXMLConfigChildren($configbase, $option, $oChild, $isenvironment, $environmentid, $serverid);
          }
        }
        else {
          // or empty value
          if(empty($configsubbase)) {
            if($isenvironment) {
              // or empty value
              $CONFIG[xmlCompat($configbase)][$environmentid][$serverid] = "";
            }
            else {
              $CONFIG[xmlCompat($configbase)][xmlCompat($option)] = "";        
            }
          }
          else {
            $CONFIG[xmlCompat($configbase)][xmlCompat($configsubbase)][xmlCompat($option)] = "";
          }
        }
        continue;
      }
    
      if(empty($configsubbase)) {
        if($isenvironment) {
          $CONFIG[xmlCompat($configbase)][$environmentid][$serverid][xmlCompat($option)] = $value;
        }
        else {
          $CONFIG[xmlCompat($configbase)][xmlCompat($option)] = $value;
        }
      }
      else {
        $CONFIG[xmlCompat($configbase)][xmlCompat($configsubbase)][xmlCompat($option)] = $value;  
      }
    }
  }

  /***********************
    Encrypt password
  ***/
  function encryptPassword($password) {
    if(!function_exists("openssl_encrypt")) {
      echo "Unable to encrypt password\n";
      echo "PHP function openssl_encrypt missing!\n";
      exit(0);
    }
    // trim the password here to get response fast for decryption verification if
    // the password contains empty characters
    return trim(@openssl_encrypt ($password, getConfiguredEncryptionMethod(), "4a921hIuAjHk"));
  } 

  /***********************
    Decrypt password
  ***/
  function decryptPassword($password) {
    return @openssl_decrypt ($password, getConfiguredEncryptionMethod(), "4a921hIuAjHk");
  }
  
  /***********************
    Available encryption methods
  ***/
  function getAvailableEncryptionMethods() {
    if(!function_exists("openssl_get_cipher_methods")) {
      echo "Unable to encrypt password\n";
      echo "PHP function openssl_get_cipher_methods missing.\n";
      exit(0);
    }
    return openssl_get_cipher_methods();
  }

  /***********************
    logToFile
  ***/
  function logToFile($logFunction, $logtext, $servername, $loglevel = 0) {
    if(!empty($servername)) {
      $configId = getEnvironmentId($servername);
      if(empty($configId)) {
        return;
      }
      $configuredLevel = intval(getConfigValue("loglevel", $configId[0], $configId[1]));
    }
    else {
      $configuredLevel = intval(getConfigValue("loglevel"));
      $servername = "<general>";
    }

    if($loglevel <= $configuredLevel) {
      $FILE = new File();
      $logFile = getTempFileName("log");
      $logString = date("Y-m-d H:i:s")." - ".$servername." - ".$logFunction." - ".$logtext."\n";
      $FILE->append($logFile, $logString);
    }
  }
  
  /***********************
    Get current encryption method
  ***/
  function getConfiguredEncryptionMethod() {
    global $CONFIG;
    $method = getConfigValue("encryption");
    if(empty($method)) {
      $method = "AES-128-CBC";
    }
    return $method;
  }

  function loadConfiguration($CONFDIR) {
    global $CONFIG;
    
    $cache = getFormValue("cache");
    if($cache == "clear") {
      unset($_SESSION['env']);
      unset($_SESSION['envadd']);
      unset($_SESSION['general']);
      removeTempFileName();
    }
    else {
      // try to read config from cache
      if(!empty($_SESSION["CONFIGCACHE"])) {
        $CONFIG = $_SESSION["CONFIGCACHE"];
        
        // if we want to disable the cache we should add all cache disable options
        if($cache == "disable") {
          delConfigValue("cache_connection");
        }
        else if($cache == "enable") {
          setConfigValue("cache_connection", "1");
        }
        return;
      }
    }
    // if XML Config file exists
    if(!empty($CONFIG["general"]["xml"])) {
      if(empty($CONFIG["general"]["xmlfile"])) {
        $CONFIG["general"]["xmlfile"] = $CONFDIR."/config.xml";
      }
      setXMLConfiguration($CONFIG["general"]["xmlfile"]);
    }

    $_SESSION["CONFIGCACHE"] = $CONFIG;
  }

  loadConfiguration($CONFDIR);
  // for setServer
  $SERVER = array();
?>
