<?php
  /*
    Copyright (C) 2009-2012 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

// add logstring to logger and print on screen
function addToLogger($str, $oFile, $logFileName) {
  echo $str;
  $oFile->append($logFileName, $str);
}

// try to connect to ldap three times to avoid temporary network glitches
function getServerWithRetries($servername, $oFile, $logFileName) {  
  for($i = 0; $i < 5; $i++) {
    // before we verify anything, make sure that we don't have a cache file
    removeTempFileName("server_".$servername);
    $oServer = setServer($servername);
    
    // if it doesn't respond to port, try again
    if(!$oServer->getRespondPort()) {
      addToLogger("    Failed port connection attempt: ".($i + 1)."\n", $oFile, $logFileName);
      sleep(2);
      continue;
    }

    // if LDAP successful, exit
    if($oServer->getRespondLDAP()) {
      return $oServer;
    }

    // LDAP not successful, unset this object and try again    
    addToLogger("    Failed LDAP connection attempt: ".($i + 1)."\n", $oFile, $logFileName);
    unset($oServer);
    sleep(3);
  }
  // still no connection, exit with one last try
  return $oServer = setServer($servername);
}

/****************************************
* verifyDatabaseConnection Require and Validates Database Connection
* @param object $oFile File object
* @param string Log File location
* return void
****************************************/
function verifyDatabaseConnection($oFile, $logFileName) {
  if(dbConfigured()) {
    addToLogger("Database - Is configured and responding\n\n", $oFile, $logFileName);
  }
  else {
    addToLogger("Database - Is NOT configured! Read Documentation!\n\n", $oFile, $logFileName);
    exit(1);
  }
}

/****************************************
* verifyPIDFileExist Verify if PID file exists and optionally remove it
* @param string PID File location
* @param object $oFile File object
* @param string Log File location
* @param string Script Name
* return void
****************************************/
function verifyPIDFileExist($pidFileName, $oFile, $logFileName, $scriptName) {
  // if pid file exists and won't be removed
  if(pidFileExists($pidFileName, $oFile, $logFileName, $scriptName)) {
    addToLogger("\nPid file exist indicating that another job is running!\n", $oFile, $logFileName);
    addToLogger("If this isn't the case. Remove it manually\n", $oFile, $logFileName);
    addToLogger($pidFileName."\n\n", $oFile, $logFileName);

    if(setServerMessage("CN=Monitor", date("Y-m-d H:i:s"), $scriptName, "")) {
      sendNotification("CN=Monitor - ".$scriptName." script. PID File Exist and has to be removed manually", $oFile, $logFileName, $scriptName);
    }
    exit(1);
  }
}

/****************************************
* Check Configuration function
* @param string servername
* @param object $oFile File object
* return string array with errors $errorArray[0][0] = "ldap_replication"
*                                 $errorArray[0][1] = "Replication verification error 1/3"
****************************************/
function checkConfiguration($servername, $oServer, $oFile) {

  $errorArray = array();
  
  if(empty($servername)) {
    return $errorArray;
  }
  $configId = getEnvironmentId($servername);
  
  $fileVerificationCounter = "collectservermessage_cfgc_".$servername;
  $logVerificationCounter = getTempFileName($fileVerificationCounter);
  $configCheck = intval(getConfigValue("collectconfig", $configId[0], $configId[1]));
  // disable config check
  if($configCheck == -1) {
    return $errorArray;
  }
  if(empty($configCheck)) {
    $configCheck = 6;
  }
  if(file_exists($logVerificationCounter)) {
    $nCounter = intval($oFile->read($logVerificationCounter));
    if($nCounter) {
      // if configuration verification should be run
      if($nCounter >= $configCheck) {
        // configuration verification
         $replicaInfo = setReplicaInfo($servername, $servername, true);
         $replicaInfoType = $replicaInfo->getReplicaType();
         $oServer->setReplicaType($replicaInfoType);
         if(in_array($replicaInfoType, array("master", "hub"))) {
           $replicationInfo = setReplicationAgreement($servername);
           if($replicationInfo->getNrOfAgreements() != $replicationInfo->getAgreementsOk()) {
             $errorArrayIndex = count($errorArray);
             $errorArray[$errorArrayIndex][0] = "ldap_replication";
             $errorArray[$errorArrayIndex][1] = "Replication verification error ".
                                                  $replicationInfo->getAgreementsOk()." / ".
                                                  $replicationInfo->getNrOfAgreements();
           }
        }


        $oFile->write($logVerificationCounter, "1");
      }
      else {
        $oFile->write($logVerificationCounter, "".($nCounter + 1));
      }
    } // if counter
  } // if file exist
  else {
    $oFile->write($logVerificationCounter, "1");
  } // if file does not exist
  return $errorArray;
}

/****************************************
* sendNotifictionBasedOnRetrySetting Check if we should wait with sending alerts
* If Retry = 1 we will ignore current message and create a temporary text file.
* Next time we will notice the text file, remove it and send the alert
* @param string $currentServer Servername
* @param string $collectretry Retry enabled flag
* @param object $oFile File object
* @param string $logFileName Log file name
* @return boolean Ignore status notification
****************************************/
function ignoreNotificationByRetrySetting($currentServer, $collectretry, $oFile, $logFileName) {
  if(empty($collectretry)) {
    return false;
  }
  $logFileRetry = getTempFileName($currentServer."_retry");
  if(file_exists($logFileRetry)) {
    addToLogger("    Removing Retry file. Notification will be sent.\n", $oFile, $logFileName);
    removeTempFileName($currentServer."_retry");
    return false;
  }
  
  addToLogger("    Creating Retry file. Notification will not be sent.\n", $oFile, $logFileName);
  $oFile->append($logFileRetry, date("Y-m-d H:i:s"));
  
  return true;
}

/****************************************
* getControlFlags Return control flag array
* @param string $inScript Script Name
* @return int array Control Flags
****************************************/
function getControlFlags($inScript) {
  if(empty($inScript)) {
    echo "\n\nERROR - Unable to open control file flag, script name missing\n\n";
    exit(1);
  }

  $oFile = new File();
  $logFileControl = getTempFileName($inScript."_control");
  if(file_exists($logFileControl)) {
    $returnArray = explode(":", $oFile->read($logFileControl));
    return $returnArray;
  }
  return array(0);
}

/****************************************
* setControlFlags Set control flag array
* @param string $inScript Script Name
* @param int array Control Flags
* @return void
****************************************/
function setControlFlags($inScript, $controlFlags) {
  if(empty($inScript)) {
    echo "\n\nERROR - Unable to open control file flag, script name missing\n\n";
    exit(1);
  }
  $oFile = new File();
  $logFileControl = getTempFileName($inScript."_control");
  $fileContent = implode(":", $controlFlags);
  $oFile->write($logFileControl, $fileContent);
}


?>