<?php
  /*
    Copyright (C) 2009-2012 Andreas Andersson

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

 /*************************************************************
 * Collect Server Messages
 * In order to use this functionality:
 * - Make sure you have php cli installed
 * - Configure database to use in config.php
 * - Schedule this file:
 *   cd <file location>;php collectsummary.php
 *************************************************************/

$WWWDIR  = dirname(__FILE__)."/../www";
$CONFDIR = dirname(__FILE__)."/../config";
require $CONFDIR."/config.php";
require $WWWDIR."/functions/config_functions.php";
require $WWWDIR."/functions/network_functions.php";
require $WWWDIR."/phpfile/file_class.php";
require $WWWDIR."/phpldap/ldap_class.php";
require $WWWDIR."/functions/ldap_functions.php";
require $WWWDIR."/functions/sql_functions.php";
require $WWWDIR."/functions/mail_functions.php";
require $WWWDIR."/objects/monitorobject.php";
require $WWWDIR."/objects/snmpobject.php";
require $WWWDIR."/objects/querytimeobject.php";
require $WWWDIR."/objects/serverobject.php";
require $WWWDIR."/functions/collect_functions.php";
require $WWWDIR."/objects/rootdnobject.php";
require $WWWDIR."/objects/replicainfoobject.php";
require $WWWDIR."/objects/replicationagreementobject.php";

$languageCode = getLanguage();
require $WWWDIR."/language/".$languageCode.".php";

// setup file logger
$oFile = new File();
$logFileName = getTempFileName("collectservermessage");
$logServerFileName = getTempFileName("collectservermessage_server");
$pidFileName = getTempFileName("collectservermessage_pid");
$currentTime = date("Y-m-d H:i:s");
// this date can be used for unavailable purposes
// as it is a value of current date only one message / day will be set
$datewithouttime = date("Y-m-d");

// Setup new collect server message log
removeTempFileName("collectservermessage");
$logStr  = "Start collecting Server Messages (cn=monitor)\n";
$logStr .= "-------------------------------------------------\n";
addToLogger($logStr, $oFile, $logFileName);
addToLogger("Started: ".$currentTime."\n", $oFile, $logFileName);

// Verify that we have a database Connection
verifyDatabaseConnection($oFile, $logFileName);

// Verify PID file exist
// This will include the _server log in the notification if pid file removed
verifyPIDFileExist($pidFileName, $oFile, $logFileName, "collectservermessage");

// Remove collect server message log
// we do not remove this file at the end of this script as it can be used to verify connection status
removeTempFileName("collectservermessage_server");

/******************************************************************************
* Checks completed. Script starts here!
******************************************************************************/
$oFile->append($pidFileName, "Started: ".$currentTime);

// array of message type.
// always included:
// port_unavailable = Can connect to port
// ldap_unavailable = verify ldap connection

$env =  getEnvironments();
// list all environments
for($i = 0; $i < count($env); $i++) {
  // log header for this environment
  $logStr = "";
  $logStr .= "Environment: ".$i." ) ".$env[$i][1]."\n";
  $logStr .= "-------------------------------------------------\n";

  // Log
  addToLogger($logStr, $oFile, $logFileName);

  $servers =  getServers($env[$i][0]);
  for($j = 0; $j < count($servers); $j++) {
    $currentServer = trim($servers[$j][1]);
    // remove any server files, if available
    removeTempFileName("server_".$currentServer);

    // main log
    addToLogger($j." ) ".$currentServer."\n", $oFile, $logFileName);

    // if ok to collect data
    $nocollect = getConfigValue("nocollect", $servers[$j][2], $servers[$j][0]);
    $collectretry = getConfigValue("collectretry", $servers[$j][2], $servers[$j][0]);
    if(empty($nocollect)) {

      // server log
      addToLogger("        : ".$currentServer." - Connecting...\n", $oFile, $logServerFileName);

      /*** Get Server Object with retries ***/
      $oServer = getServerWithRetries($currentServer, $oFile, $logFileName, $currentTime);

      /*** Verify Port ***/
      if(!$oServer->getRespondPort()) {
        if(ignoreNotificationByRetrySetting($currentServer, $collectretry, $oFile, $logFileName)) {
          continue;
        } // if retry settings enabled
        if(setServerMessage($currentServer, $currentTime, "port_unavailable", "")) {
          sendNotification("Registered ".$currentTime, 
                           $oFile, $logFileName, "port_unavailable", $currentServer);
        } // set message, send notification
        addToLogger("        : ".$currentServer." - Check Completed\n\n", $oFile, $logServerFileName);
        continue;
      }

      /*** Verify LDAP ***/
      if(!$oServer->getRespondLDAP()) {
        if(ignoreNotificationByRetrySetting($currentServer, $collectretry, $oFile, $logFileName)) {
          continue;
        } // if retry settings enabled
        if(setServerMessage($currentServer, $currentTime, "ldap_unavailable", "")) {
          sendNotification("Registered ".$currentTime, 
                           $oFile, $logFileName, "ldap_unavailable", $currentServer);
        } // if set message
        addToLogger("        : ".$currentServer." - Check Completed\n\n", $oFile, $logServerFileName);
        continue;
      }

      /*** Get Start Time if available ***/      
      $monitor = setMonitor($currentServer, array("start_time"));
      $starttime = $monitor->getStartTime();
      if(!empty($starttime)) {
        addToLogger("    Start Time: ".$starttime."\n", $oFile, $logFileName);
      }

      // Check if server is back online
      if(setServerMessage($currentServer, $currentTime, "ldap_available", $starttime)) {      
        sendNotification("Server Start Time ".$starttime, 
                         $oFile, $logFileName, "ldap_available", $currentServer);
      } // if server back online
      
      // Check if server has been restarted (if start time can be fetched)
      if(!empty($starttime)) {
        if(setServerRestarted($currentServer, $currentTime, "ldap_available", $starttime)) {
          // for notification we can change status to ldap_restart to split online/restart status
          sendNotification("New Server Start Time detected ".$starttime, 
                           $oFile, $logFileName, "ldap_restart", $currentServer);
        }        
      } // if server has been restarted

      // check configuration      
      $errorArray = checkConfiguration($currentServer, $oServer, $oFile);
      for($k = 0; $k < count($errorArray); $k++) {
        if(setServerMessage($currentServer, $currentTime, $errorArray[$k][0], $errorArray[$k][1])) {
          sendNotification($errorArray[$k][1], 
                           $oFile, $logFileName, $errorArray[$k][0], $currentServer);    
        
        }      
      }
      // remove temporary files      
      removeTempFileName("server_".$currentServer);
      addToLogger("        : ".$currentServer." - Check Completed\n\n", $oFile, $logServerFileName);
    }
    else {
      // collect of data disabled
       addToLogger("    Collect of Data disabled\n", $oFile, $logFileName);
    } // collect disabled
  } // for each server
  addToLogger("\n", $oFile, $logFileName);
} // for each environment

removeTempFileName("collectservermessage_pid");

addToLogger("Completed!\n\n", $oFile, $logFileName);

?>
