var should = require('should'),
    url = require('url'),
    qs = require('qs'),
    oauthTokenCheck = require('../lib/oauthTokenCheck').oauthTokenCheck,
    testUrl = 'http://api.stanford.edu/test/v1';

function makeResponse() {
    return {
        _status: 0,
        _headers: {},
        _body: '',
        writeHead: function (status, headers) {
            this._status = status;
            this._headers = headers;
        },
        write: function (body) {
            this._body += body;
        },
        end: function (body) {
            if (body) {
                this._body += body;
            }
        },
        sendError: function (status, body) {
            this._status = status;
            this._body = body;
            // block any further writes
            this.writeHead = this.write = this.end = function() {};
        }
        
    }
}

function makeRequest() {
    return {
        method: 'GET',
        url: testUrl,
        headers: {
            'Content-Type': 'application/json'
        },
        query: {},
        api: {
            proxy: {
                jsonp: false
            }
        }
    }
}
    
describe('OAuthTokenCheck', function() {
    
    it('should fail with a 401 status if a request is made without a token, but a token is required', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        req.api.proxy.requireOAuthToken = true;
        if (req.headers.Authorization) {
            delete req.headers.Authorization;
        }
        
        (function () {
            oauthTokenCheck(req, res, function () {});
            res.writeHead(200, { 'Content-type': 'application/json'});
            res.write('{a: 1,');
            res.write(' b: 2,');
            res.end(' c: 3}');
        }).should.not.throw();

        res._status.should.equal(401);
    });

    it('should succeed if a request is made with a token, and a token is required', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        req.headers.authorization = 'Bearer 12345';
        req.api.proxy.requireOAuthToken = true;
        
        (function () {
            oauthTokenCheck(req, res, function () {});
            res.writeHead(200, { 'Content-type': 'application/json'});
            res.write('{a: 1,');
            res.write(' b: 2,');
            res.end(' c: 3}');
        }).should.not.throw();

        res._status.should.equal(200);
    });

    it('should succeed if a request is made with a token, even if a token is not required', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        req.headers.authorization = 'Bearer 12345';
        req.api.proxy.requireOAuthToken = false;
        
        (function () {
            oauthTokenCheck(req, res, function () {});
            res.writeHead(200, { 'Content-type': 'application/json'});
            res.write('{a: 1,');
            res.write(' b: 2,');
            res.end(' c: 3}');
        }).should.not.throw();

        res._status.should.equal(200);
    });

});

