var should = require('should'),
    url = require('url'),
    qs = require('qs'),
    methodOverride = require('../lib/methodOverride').methodOverride;

function makeResponse() {
    return {
        _status: 0,
        _headers: {},
        _body: '',
        writeHead: function (status, headers) {
            this._status = status;
            this._headers = headers;
        },
        write: function (body) {
            this._body += body;
        },
        end: function (body) {
            if (body) {
                this._body += body;
            }
        },
        sendError: function (status, body) {
            this._status = status;
            this._body = body;
        }
        
    }
}

function makeRequest() {
    return {
        method: 'POST',
        url: 'http://api.stanford.edu/test/v1',
        headers: {
            'Content-Type': 'application/json'
        },
        query: {},
        api: {
            proxy: {
                methodOverride: false
            }
        }
    }
}
    
describe('MethodOverride', function() {
    
    /*
     * the methodOverride code should never change a request if
     * there's no override, nor should it throw an error
     */
    it('should not change a request with no override', function() {
        var req = makeRequest(),
            res = makeResponse();

        (function () { methodOverride(req, res, function () {}) } ).should.not.throw();
        req.url.should.equal('http://api.stanford.edu/test/v1');
        req.method.should.equal('POST');
        res._status.should.equal(0);
    });
    
    it('should remove override headers', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        req.headers['x-http-method'] = 'DELETE';
        req.headers['x-http-method-override'] = 'DELETE';
        req.headers['x-method-override'] = 'DELETE';

        (function () { methodOverride(req, res, function () {}) } ).should.not.throw();
        req.url.should.equal('http://api.stanford.edu/test/v1');
        res._status.should.equal(400);
        req.headers.hasOwnProperty('x-http-method').should.be.false;
        req.headers.hasOwnProperty('x-http-method-override').should.be.false;
        req.headers.hasOwnProperty('x-method-override').should.be.false;
    });
    
    it('should remove the _method query parameter', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        req.url = req.url + '?_method=DELETE';
        req.query = qs.parse(url.parse(req.url).query);

        (function () { methodOverride(req, res, function () {}) } ).should.not.throw();
        req.url.should.equal('http://api.stanford.edu/test/v1');
        Object.keys(req.query).length.should.equal(0);
    });


    it('should fail with a 400 status if an override is requested but not allowed', function () {
        var req = makeRequest(),
            res = makeResponse();
        
        // add an override to the request, but don't allow overrides
        req.headers['x-http-method'] = 'DELETE';

        (function () { methodOverride(req, res, function () {}) } ).should.not.throw();
        req.url.should.equal('http://api.stanford.edu/test/v1');
        res._status.should.equal(400);
    });
    
    it('should change request method and set origMethod', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        // set up an override, and allow overrides
        req.headers['x-http-method'] = 'DELETE';
        req.api.proxy.methodOverride = true;

        (function () { methodOverride(req, res, function () {}) } ).should.not.throw();
        req.url.should.equal('http://api.stanford.edu/test/v1');
        // status should not be set to an error
        res._status.should.equal(0);
        req.method.should.equal('DELETE');
        req.origMethod.should.equal('POST');
        req.hasOwnProperty('origMethod').should.be.true;
    });

    it('should fail with a 405 status if the request method is not POST', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        // setup an override, and allow overrides
        req.headers['x-http-method'] = 'DELETE';
        req.api.proxy.methodOverride = true;

        // set the method to something other than POST
        req.method = 'GET';

        (function () { methodOverride(req, res, function () {}) } ).should.not.throw();
        res._status.should.equal(405);
    });


});

