var should = require('should'),
    url = require('url'),
    qs = require('qs'),
    jsonpWrapper = require('../lib/jsonpWrapper').jsonpWrapper;

function makeResponse() {
    return {
        _status: 0,
        _reason: 'OK',
        _headers: {},
        _body: '',
        setHeader: function (name, value) {
            this._headers[name] = value;
        },
        writeHead: function (status, reason) {
            this._status = status;
            this._reason = reason;
        },
        write: function (body) {
            this._body += body;
        },
        end: function (body) {
            if (body) {
                this._body += body;
            }
        },
        sendError: function (status, body) {
            this._status = status;
            this._body = body;
            // block any further writes
            this.writeHead = this.write = this.end = function() {};
        }
        
    }
}

function makeRequest() {
    return {
        method: 'GET',
        url: 'http://api.stanford.edu/test/v1',
        headers: {
            'Content-Type': 'application/json'
        },
        query: {},
        api: {
            proxy: {
                jsonp: false
            }
        }
    }
}
    
describe('JSONPWrapper', function() {
    
    /*
     * the methodOverride code should never change a request if
     * there's no override, nor should it throw an error
     */
    it('should not change a request with no wrapper request', function() {
        var req = makeRequest(),
            res = makeResponse();

        (function () {
            jsonpWrapper(req, res, function () {});
            res.writeHead(200, { 'Content-type': 'application/json'});
            res.write('{a: 1,');
            res.write(' b: 2,');
            res.end('}');
        }).should.not.throw();
        req.url.should.equal('http://api.stanford.edu/test/v1');
        req.method.should.equal('GET');
        res._status.should.equal(200);
    });
    
    it('should fail with a 400 status if a wrapper request is made, but wrapper requests are not allowed', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        req.url += '?callback=foo';
        req.query = qs.parse(url.parse(req.url).query);
        
        (function () {
            jsonpWrapper(req, res, function () {});
            res.writeHead(200, { 'Content-type': 'application/json'});
            res.write('{a: 1,');
            res.write(' b: 2,');
            res.end(' c: 3}');
        }).should.not.throw();

        res._status.should.equal(400);
    });

    it('should succeed if a wrapper request is made, wrapper requests are allowed, and the API call was successful', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        req.url += '?callback=foo';
        req.query = qs.parse(url.parse(req.url).query);
        req.api.proxy.jsonp = true;
        
        (function () {
            jsonpWrapper(req, res, function () {});
            res.setHeader('Content-type', 'application/json');
            res.writeHead(200, 'OK');
            res.write('{a: 1,');
            res.write(' b: 2,');
            res.end(' c: 3}');
        }).should.not.throw();

        res._status.should.equal(200);
        res._body.should.equal('foo({a: 1, b: 2, c: 3});');
    });

    it('should not wrap failed API call responses', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        req.url += '?callback=foo';
        req.query = qs.parse(url.parse(req.url).query);
        req.api.proxy.jsonp = true;
        
        (function () {
            jsonpWrapper(req, res, function () {});
            res.writeHead(404, { 'Content-type': 'application/json'});
            res.end('{error: "thing not found"}');
        }).should.not.throw();

        res._status.should.equal(404);
        res._body.should.equal('{error: "thing not found"}');
    });

    it('should remove the callback query parameter', function() {
        var req = makeRequest(),
            res = makeResponse();
            
        req.url += '?callback=foo';
        req.query = qs.parse(url.parse(req.url).query);
        req.api.proxy.jsonp = true;

        (function () { jsonpWrapper(req, res, function () {}) } ).should.not.throw();
        req.url.should.equal('http://api.stanford.edu/test/v1');
        Object.keys(req.query).length.should.equal(0);
    });

});

