/*
 * add API handling to http-proxy, and fix logging
 *
 */

var httpProxy = require('http-proxy'),
    connect = require('connect'),
    logger = require('morgan'),
    query = require('connect-query'),
    url = require('url'),
    methodOverride = require('./methodOverride').methodOverride,
    jsonpWrapper = require('./jsonpWrapper').jsonpWrapper,
    oauthTokenCheck = require('./oauthTokenCheck').oauthTokenCheck;

/*
 * add logger tokens for the original request URL and method
 */
logger.token('orig-url', function (req, res) {
  // return origUrl if it was set, otherwise url
  return encodeURI(req.origUrl || req.url);
});
logger.token('orig-method', function (req, res) {
  return req.origMethod || req.method;
});

/*
 * add logger token for the backend host
 */
logger.token('api-host', function (req, res) {
  return (req.api && req.api.backend && req.api.backend.hostname) ? req.api.backend.hostname : '-';
});

/*
 * add logger token for ISO-format dates
 */
logger.token('isodate', function (req, res) {
  return (new Date()).toISOString();
});

/*
 * add logger token for response content length
 */
logger.token('len', function (req, res) {
  return parseInt(res.getHeader('content-length'),10) || 0;
});

/*
 * add some safer tokens
 */
logger.token('safe-url', function (req, res) {
  // if req.origUrl is set, return reg.url, otherwise -
  return req.origUrl ? encodeURI(req.url) : '-';
});
logger.token('safe-referrer', function (req, res) {
  return encodeURI(req.headers.referer || req.headers.referrer || '-');
});
logger.token('safe-ua', function (req, res) {
  return encodeURI(req.headers['user-agent'] || '-');
});

/*
 * define a JSON-based log format
 */
logger.format('json', [
  '{',
  ' ip:":remote-addr",',
  ' date:":isodate",',
  ' orig_method:":orig-method",',
  ' orig_url:":orig-url",',
  ' method:":method",',
  ' api_host:":apiHost",',
  ' url:":safeUrl",',
  ' status::status,',
  ' len::len,',
  ' response-time::response-time,',
  ' ref:":safeReferrer",',
  ' ua:":safeUa"',
  '}'
].join(''));

/*
 * define a field-based log format for syslog
 */
logger.format('syslog', [
  '[apirequest]',
  ' :remote-addr',
  ' :isodate',
  ' :orig-method',
  ' ":orig-url"',
  ' :method',
  ' :api-host',
  ' ":safe-url"',
  ' :status',
  ' :len',
  ' :response-time',
  ' ":safe-referrer"',
  ' ":safe-ua"'
].join(''));

function addSendError(req, res, next) {
  res.sendError = function sendError(status, body) {
    var headers = { 'Content-Type': 'application/json' };

    // convert plain strings to objects
    if (typeof body === 'string') {
      body = { 'error': body };
    }

    // body should be an object
    //  - but not an array (which are also objects)
    // and it should have an error property
    // otherwise, report as an internal error
    if (typeof body !== 'object' || body instanceof Array || !body.hasOwnProperty('error')) {
      status = 500; // bad error format, report as internal error
      body = { 'error': 'Internal Server Error' };
    }

    // convert body to string
    body = JSON.stringify(body);
    headers['Content-Length'] = body.length;
    this.writeHead(status, headers);
    this.end(body);
  };
  next();
}

/*
 * wrap the logger in a function that hijacks res.writeHead so that we
 * can log data from the response
 */
function proxyLogger(logger) {

  return function logAPIRequest(req, res, next) {
    var _writeHead = res.writeHead;

    /* http-proxy has response and res objects:
     *   response is the response from the backend
     *   res is our response object
     * http-proxy doesn't set the headers on our res object;
     * it just writes them out using
     *   res.writeHead(response.statusCode, response.headers);
     *
     * This is normally OK, except when we're trying to log parts
     * of the response using logger
     * So, once again, we hijack writeHead
     */

    res.writeHead = function (statusCode, reason, headers) {
      // restore the original res.writeHead function
      res.writeHead = _writeHead;
      // set our response object's statusCode and headers
      res.statusCode = statusCode;
      // log request
      logger(req, res, function () {});

      // write response back to client
      res.writeHead(statusCode, reason, headers);
    };

    next();
  };
}

function proxyAPIRequest(req, res, proxy) {
  var buffer = httpProxy.buffer(req),
      options = {
        host: req.api.backend.hostname,
        port: req.api.backend.port,
        buffer: buffer,
        target: {
          https: (req.api.backend.protocol && req.api.backend.protocol === 'https')
        }
      };

  proxy.proxyRequest(req, res, options);
}

exports.createHTTPSRedirector = function createHTTPSRedirector(options) {
  var loggerOptions = options.logger || {};

  return connect()
  .use(logger('combined', loggerOptions))
  .use(function (req, res) {
    var redirect = url.parse(req.url),
        headers = {
          'Content-Type': 'application/json'
        },
        status = 301,
        statusMessage = 'Moved Permanently',
        body = { message: "APIs are only available over HTTPS" },
        badHostBody = { message: "Invalid host in API request" },
        badAPIBody = { message: "Unknown API request" },
        reqHost = redirect.hostname || redirect.host || req.headers.host,
        matches;

    matches = reqHost.match(/^([^\s:]+)(:\d+)$/);
    if (matches) {
      reqHost = matches[1];
    }

    if (!reqHost || options.apiHosts.indexOf(reqHost) === -1) {
      status = 400;
      statusMessage = 'Bad Request';
      body = badHostBody;
    } else if (! options.apiList.getAPI(req.url)) {
      status = 404;
      statusMessage = 'Not Found';
      body = badAPIBody;
    } else {
      redirect.protocol = 'https';
      redirect.hostname = reqHost;
      redirect.port = options.apiPort || 443;
      redirect.host = redirect.hostname + ':' + redirect.port;
      headers.location = url.format(redirect);
    }
    body = JSON.stringify(body);
    headers['Content-Length'] = body.length;
    res.writeHead(status, statusMessage, headers);
    res.end(body);
  });
};

exports.createProxy = function createProxy(options) {
  var loggerOptions = options.logger || {},
      proxyOptions = options.proxy,
      loggerFormat = loggerOptions.format || 'syslog';

  return httpProxy.createServer(
    proxyOptions,
    query(),
    proxyLogger(logger(loggerFormat, loggerOptions)),
    addSendError,
    options.apiList.addAPIDetails,
    oauthTokenCheck,
    methodOverride,
    jsonpWrapper,
    proxyAPIRequest
  );
};
