/*
 * add a JSONP wrapper around an API response, if it's allowed by configuration
 *
 * This is very similar to the connect jsonp middleware, but this one fixes
 * the content-length header (connect-jsonp expects responses to be generated within connect,
 * not proxied from another server)
 */

var url = require('url');

exports.jsonpWrapper = function jsonpWrapper(req, res, next) {
  //
  // we need to save some functions from the response,
  // since we'll replace them with our versions that
  // store the response
  //
  var saved = {
        writeHead: res.writeHead,
        write: res.write,
        end: res.end
      },
      callback,
      tmpUrl;

  if (!req.query.callback) {
    // nothing to do!
    next();
  } else {

    // save the callback name, and remove the callback parameter
    // from the request URL

    callback = req.query.callback;
    tmpUrl = url.parse(req.url);
    tmpUrl.query = req.query;
    delete tmpUrl.query.callback;
    delete tmpUrl.search;
    req.url = url.format(tmpUrl);
    req.query = tmpUrl.query;

    if (!req.api.proxy.jsonp) {

      // not supported for this API, so return an invalid request
      res.sendError(400, 'JSONP not supported for this API');

    } else {

      // replace the writeHead method on the response, which in turn replaces
      // the write and end methods

      res.writeHead = function jsonpWrapperWriteHead(status, origReason, origHeaders) {
        var jsonpBuffers = [],
            jsonp = ''; // save the response body and JSONP-ify it

        if (status === 200) {

          jsonpBuffers.push(new Buffer('' + callback + '('));

          res.write = function jsonpWrapperWrite(chunk, encoding) {
            // save partial writes
            jsonpBuffers.push(new Buffer(chunk, encoding));
          };
          res.end = function jsonpWrapperEnd(chunk, encoding) {
            // save the last chunk of data
            if (chunk) {
              jsonpBuffers.push(new Buffer(chunk, encoding));
            }

            jsonpBuffers.push(new Buffer(');'));

            jsonp = Buffer.concat(jsonpBuffers).toString();
            // fix the response headers: content type and length
            res.setHeader('content-type', 'application/javascript');
            res.setHeader('content-length', Buffer.byteLength(jsonp));

            // restore the original functions
            res.end = saved.end;
            res.write = saved.write;
            res.writeHead = saved.writeHead;

            // call the original functions to output the headers and body
            res.writeHead(status, origReason, origHeaders);
            res.end(jsonp, 'utf8');
          };

        } else {
          // on an error, just use the original functions
          res.writeHead = saved.writeHead;
          res.write = saved.write;
          res.end = saved.end;
          res.writeHead(status, origReason, origHeaders);
        }
      };

      next();
    }
  }
};
