var should = require('should'),
    shouldHttp = require('should-http'),
    url = require('url'),
    qs = require('qs'),
    child_process = require('child_process'),
    proxy = require('../lib/proxy'),
    apis = require('../lib/apis'),
    http = require('http'),
    fs = require('fs'),
    testlog = fs.createWriteStream('test.log', { 'flags': 'a', 'mode': 0644}),

    // object to save API state so that it can be checked by tests
    apiState = (function () {
        var _state = false,
            _headers = null,
            _url = null,
            _etag = null,
            _verbose = false;

            // throw an error if the API was not called, but should have been
            this.shouldHaveBeenCalled = function wasCalled(verbosity) {
                if (!_state) {
                    throw new Error('API was not called (should have been called)');
                } else if (_verbose || verbosity) {
                    console.log('API was called, as expected');
                }
            };

            // throw an error if the API was called, but should not have been
            this.shouldNotHaveBeenCalled = function wasCalled(verbosity) {
                if (_state) {
                    throw new Error('API was called (should not have been called)');
                } else if (_verbose || verbosity) {
                    console.log('API was not called, as expected');
                }
            };

            // called by the APIs to record that the API was called
            this.wasCalled = function wasCalled(verbosity) {
                _state = true;
                if (_verbose || verbosity) {
                    console.log('API was called');
                }
            };

            //
            // getters and setters for various parts of the API request
            // including URL, headers, ETag
            //
            
            this.setUrl = function setUrl(u) {
                _url = url.parse(u);
            };

            this.getUrl = function getUrl() {
                return _url;
            };

            this.setRequestHeaders = function setRequestHeaders(h) {
                _headers = h;
            };
            
            this.getRequestHeaders = function getRequestHeaders() {
                return _headers;
            };
            
            this.getETag = function getETag() {
                if (!_etag) {
                    _etag = Math.floor(1000000 * Math.random()).toString();
                }
                return _etag;
            };
            
            // reset the state
            this.reset = function reset(verbosity) {
                _state = false;
                _url = null;
                _etag = null;
                _headers = null;
                _verbose = verbosity || false;
                if (_verbose) {
                    console.log('API state was reset');
                }
            };

        return this;
    })(),
    
    // define two API proxies: one with all options enabled, and one without
    // any options
    
    proxyOptions = {
        apiHosts: [ 'localhost', 'good.local' ],
        apiPort: 3000,
        apiDefaultHost: [ 'localhost'],
        apiList: apis({
            "/allOpts/v1": {
                description: "Test API with all options enabled",
                proxy: {
                    jsonp: true,
                    methodOverride: true,
                    requireOAuthToken: true
                },
                backend: {
                    hostname: "localhost",
                    port: 3100,
                    protocol: "http",
                    pathname: "/api/v1"
                }
            },
            "/noOpts/v1": {
                description: "Test API with no options enabled",
                proxy: {
                    jsonp: false,
                    methodOverride: false,
                    requireOAuthToken: false
                },
                backend: {
                    hostname: "localhost",
                    port: 3200,
                    protocol: "http",
                    pathname: "/api/v1"
                }
            }
        }),
        logger: {
            stream: { write: function write (message, encoding) { testlog.write(message, encoding); }},
            format: 'syslog',
        },
        proxy: {
            target: {
                https: false
            },
            changeOrigin: true
        }
    },
    
    // wrapper function to make API calls simpler
    
    apiRequest = function apiRequest(options, callback) {
        var testparam = 'testparam=' + encodeURIComponent(this._runnable.title.replace(/\s+/g,'_')),
            _opts = {
                hostname: 'localhost',
                port: 3000,
                path: '/' + (options.api ? options.api : 'bad') + '/v1/x/' + (options.qs ? '?' + options.qs + '&' : '?') + testparam,
                method: options.method || 'GET',
                headers: { 'user-agent': 'api-gateway-test-suite' }
            };



        options.headers = options.headers || {};
        
        if (options.token && !options.headers.Authorization) {
            options.headers['Authorization'] = 'Bearer ' + options.token;
        }

        Object.keys(options.headers).forEach (function (k) {
            _opts.headers[k] = options.headers[k];
        });

        var req = http.request(_opts, function (res) {
            res.body = '';
            res.on('data', function (chunk) {
                res.body += chunk;
            });
            callback(res)
        });
        req.end();
    };


describe('API Proxy', function() {

    var goodBody1 = '{test:1234',
        goodBody2 = '}',
        goodBody = goodBody1 + goodBody2,
        errorBody = '{error:"Invalid Method"}',
        apiProxy,
        apiRedirector,
        allOptsAPI,
        noOptsAPI;

    /*
     * Before running the tests, start up
     * - a proxy
     * - an API with all options enabled
     * - an API with all options disabled
     */

    before(function(done) {

      apiProxy = proxy.createProxy(proxyOptions);
        
      allOptsAPI = http.createServer(function (req, res) {
        apiState.wasCalled();   // mark API as called
        apiState.setUrl(req.url); // save the request URL
        apiState.setRequestHeaders(req.headers); // and the request headers
        if (req.method === 'DELETE') { // don't allow deletes
          res.writeHead(405, {
            'Content-Type': 'application/json',
            'Content-Length': errorBody.length
          });
          res.write(errorBody);
        } else {
          res.writeHead(200, {
            'ETag': apiState.getETag(),
            'X-Foo-Bar-Baz': 'fubar',
            'Content-Type': 'application/json',
            'Content-Length': goodBody.length
          });
          res.write(goodBody1);
          res.write(goodBody2);
        }
        res.end();
      });

      noOptsAPI = http.createServer(function (req, res) {
        apiState.wasCalled();
        apiState.setUrl(req.url);
        apiState.setRequestHeaders(req.headers);
        res.writeHead(200, {
          'ETag': apiState.getETag(),
          'X-Foo-Bar-Baz': 'fubar',
          'Content-Type': 'application/json',
          'Content-Length': goodBody.length
        });
        res.write(goodBody);
        res.end();
      });

      apiProxy.listen(3000, function (e) {
        console.log('        Proxy on ' + this.address().address + ':' + this.address().port);
      });

      apiRedirector = proxy.createHTTPSRedirector(proxyOptions).listen(3080, function (e) {
        console.log('        Redirector on ' + this.address().address + ':' + this.address().port);
      });

      allOptsAPI.listen(3100, function (e) {
        console.log('        allOpts API on ' + this.address().address + ':' + this.address().port);
      });

      noOptsAPI.listen(3200, function (e) {
        console.log('        noOpts API on ' + this.address().address + ':' + this.address().port);
      });

      setTimeout(done, 1000);
    });

    after(function (done) {
      console.log('Closing apiRedirector');
      apiRedirector.close();
      console.log('Closing apiProxy');
      apiProxy.close();
      console.log('Closing allOptsAPI');
      allOptsAPI.close();
      console.log('Closing noOptsAPI');
      noOptsAPI.close();
      done();
    });

    beforeEach(function () {
        apiState.reset();
    });

    it('should return a 404 status for an unknown API @fail', function(done) {
        apiRequest.call(this, {}, function (res) {
            apiState.shouldNotHaveBeenCalled();

            res.should.have.status(404);
            done();
        });
        /*
        apiRequest.call(this, {}, function (res) {
            apiState.shouldNotHaveBeenCalled();

            res.should.have.status(404);
            done();
        });
        */
    });

    it('should return the backend status for a known API @noOpts @succeed', function(done) {
        apiRequest.call(this, {api: 'noOpts'}, function (res) {
            apiState.shouldHaveBeenCalled();

            res.should.have.status(200);
            done();
        });
    });

    it('should succeed if OAuth token received but not required @noOpts @succeed', function(done) {

        apiRequest.call(this, {api: 'noOpts', token: '12345'}, function (res) {
            apiState.shouldHaveBeenCalled();

            // this should succeed
            res.should.have.status(200);
            done();
        });
    });

    it('should fail if OAuth token required but not received @allOpts @fail', function(done) {

        apiRequest.call(this, {api: 'allOpts'}, function (res) {
            apiState.shouldNotHaveBeenCalled();

            res.should.have.status(401);
            done();
        });
    });

    it('should succeed if OAuth token required and received @allOpts @succeed', function(done) {

        apiRequest.call(this, {api: 'allOpts', token: '12345'}, function (res) {
            apiState.shouldHaveBeenCalled();

            // this should succeed
            res.should.have.status(200);
            done();
        });
    });

    it('should return JSONP if JSONP is requested and allowed @allOpts @succeed', function(done) {
        apiRequest.call(this, {api: 'allOpts', token: '12345', qs: 'callback=foo'}, function (res) {
            var expectedBody = 'foo(' + goodBody + ');';

            apiState.shouldHaveBeenCalled();
            res.should.have.status(200);
            res.should.have.header('content-type', 'application/javascript');
            res.should.have.header('content-length', '' + expectedBody.length);

            res.on('end', function (chunk) {
                if (chunk) {
                    res.body += chunk;
                }
                res.body.should.equal(expectedBody);
                done();
            });
        });
    });

    it('should fail if JSONP is requested but not allowed @noOpts @fail', function(done) {
        apiRequest.call(this, {api:'noOpts', qs: 'callback=foo'}, function (res) {
            apiState.shouldNotHaveBeenCalled();
            res.should.have.status(400);
            done();
        });
    });

    it('should override method if requested and allowed @allOpts @succeed', function(done) {
        apiRequest.call(this, {api: 'allOpts', token: '12345', method: 'POST', qs: '_method=DELETE'}, function (res) {
            apiState.shouldHaveBeenCalled();

            // this should fail with a bad method response, since the test API disallows DELETE
            res.should.have.status(405);
            done();
        });
    });

    it('should fail if override requested but not using POST @allOpts @fail', function(done) {
        apiRequest.call(this, {api: 'allOpts', token: '12345', qs: '_method=DELETE'}, function (res) {
            apiState.shouldNotHaveBeenCalled();
            // this should fail with a 405, since the method must be POST
            res.should.have.status(405);
            done();
        });
    });

    it('should fail if override requested but not allowed @noOpts @fail', function(done) {
        apiRequest.call(this, {api: 'noOpts', method: 'POST', qs: '_method=DELETE'}, function (res) {
            apiState.shouldNotHaveBeenCalled();

            // this should fail, since it's not allowed for this API
            res.should.have.status(400);
            done();
        });
    });

    it('should preserve the query string @noOpts @succeed', function(done) {
        var qs = 'foo=bar',
            testparam = 'testparam=' + encodeURIComponent(this._runnable.title.replace(/\s+/g,'_'));
        
        apiRequest.call(this, {api: 'noOpts', method: 'POST', qs: qs}, function (res) {
            apiState.shouldHaveBeenCalled();
            var url = apiState.getUrl();

            // this should succeed
            res.should.have.status(200);
            url.query.should.equal(qs + '&' + testparam);
            done();
        });
    });

    it('should preserve the query string even with an override @allOpts @succeed', function(done) {
        var qs = 'foo=bar',
            testparam = 'testparam=' + encodeURIComponent(this._runnable.title.replace(/\s+/g,'_'));

        apiRequest.call(this, {api: 'allOpts', method: 'POST', token: '12345', qs: '_method=DELETE&' + qs}, function (res) {
            apiState.shouldHaveBeenCalled();
            var url = apiState.getUrl();

            // this should succeed (which means fail, because DELETEs are not allowed)
            res.should.have.status(405);
            url.query.should.equal(qs + '&' + testparam);
            done();
        });
    });

    /*
     * check that the proxy passes the If-None-Match and X-Test-Header headers to the backend,
     * and that they have the correct values
     */

    it('should pass headers from client to backend @allOpts @succeed', function(done) {
        var testVal = Math.floor(1000000 * Math.random()).toString(),
            testETag = Math.floor(1000000 * Math.random()).toString();

        apiRequest.call(this, {api: 'allOpts', method: 'GET', token: '12345', headers: { 'X-Test-Header': testVal, 'If-None-Match': testETag }}, function (res) {
            apiState.shouldHaveBeenCalled();
            var headers = apiState.getRequestHeaders();
            
            res.should.have.status(200);
            // node always downcases header names
            headers['if-none-match'].should.equal(testETag);
            headers['x-test-header'].should.equal(testVal);
            done();
        });
    });

    /*
     * check that the proxy passes back the ETag and X-Foo-Bar-Baz headers,
     * and that they have the correct values
     */
    it('should pass headers from backend to client @allOpts @succeed', function(done) {
        apiRequest.call(this, {api: 'allOpts', method: 'GET', token: '12345'}, function (res) {
            apiState.shouldHaveBeenCalled();
            
            res.should.have.status(200);
            // node always downcases header names
            res.headers['etag'].should.equal(apiState.getETag());
            res.headers['x-foo-bar-baz'].should.equal('fubar');
            done();
        });
    });
    
    /*
     * check that the redirector redirects requests for known APIs to HTTPS
     */
    it('should redirect HTTP requests for known APIs to HTTPS @redirect @succeed', function (done) {
        var req = http.request({ hostname: 'localhost', port: 3080, path: '/noOpts/v1' }, function (res) {
                apiState.shouldNotHaveBeenCalled();
                res.should.have.status(301);
                res.should.have.header('location', 'https://localhost:3000/noOpts/v1');
                done();
            });

        req.end();
    });

    /*
     * check that the redirector does not redirect bad requests
     */
    it('should not redirect bad HTTP requests to HTTPS @redirect @fail', function (done) {
        var req = http.request({ hostname: 'localhost', port: 3080, path: '/noOpts/v1', headers: { Host: 'bad.local'} }, function (res) {
                apiState.shouldNotHaveBeenCalled();
                res.should.have.status(400);
                res.resume();
                done();
            });

        req.end();
    });

    /*
     * check that the redirector does not redirect bad requests
     */
    it('should not redirect bad HTTP requests with full URI in path to HTTPS @redirect @fail', function (done) {
        var req = http.request({ hostname: 'localhost', port: 3080, path: 'http://bad.local/noOpts/v1' }, function (res) {
                apiState.shouldNotHaveBeenCalled();
                res.should.have.status(400);
                res.resume();
                done();
            });

        req.end();
    });

    /*
     * check that the redirector does not redirect requests for unknown APIs
     */
    it('should not redirect requests for unknown APIs @redirect @fail', function (done) {
        var req = http.request({ hostname: 'localhost', port: 3080, path: '/noOpts/v2' }, function (res) {
                apiState.shouldNotHaveBeenCalled();
                res.should.have.status(404);
                res.resume();
                done();
            });

        req.end();
    });

    /*
     * check that the redirector works if the client includes a port in the Host header
     */
    it('should redirect requests with port in Host header @redirect @succeed', function (done) {
        var req = http.request({ hostname: 'localhost', port: 3080, path: '/noOpts/v1', headers: { 'Host': 'localhost:3080' } },
            function (res) {
                apiState.shouldNotHaveBeenCalled();
                res.should.have.status(301);
                res.should.have.header('location', 'https://localhost:3000/noOpts/v1');
                res.resume();
                done();
            });

        req.end();
    });

    /*
     * check that the redirector works if the client does not include a port in the Host header
     */
    it('should redirect requests without port in Host header @redirect @succeed', function (done) {
        var req = http.request({ hostname: 'localhost', port: 3080, path: '/noOpts/v1', headers: { 'Host': 'localhost' } },
            function (res) {
                apiState.shouldNotHaveBeenCalled();
                res.should.have.status(301);
                res.should.have.header('location', 'https://localhost:3000/noOpts/v1');
                res.resume();
                done();
            });

        req.end();
    });

});
